# -*- coding: utf-8 -*-

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.input_params as rm_params
import sandbox.projects.websearch.middlesearch.CompareMiddlesearchBinaries as cmb
from sandbox.projects import resource_types
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common.search import bugbanner
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.release_machine.components.configs.middle import MiddleCfg
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.websearch import acceptance
from sandbox.projects.websearch.middlesearch.PriemkaMiddlesearchBinary import params as pmb_params


_CTX_NAME_TEMPLATE = "priemka_{}"


class TriggerFail(sp.SandboxBoolParameter):
    name = "fail_on_problems"
    description = "Fail on problems"
    default_value = True


class CfgFromHamster(sp.SandboxBoolParameter):
    name = "set_hamster_config_source"
    description = "Use hamster cfg instead of prod (SEARCH-1918)"
    default_value = True


class RevisionTested(sp.SandboxStringParameter):
    name = "revision_tested"
    description = "Revision tested"


def binaries_to_compare(names):
    params = []
    for n, val in names.items():
        class BinaryCheckbox(sp.SandboxBoolParameter):
            name = _CTX_NAME_TEMPLATE.format(n)
            description = "Priemka {}".format(n)
            default_value = val
            group = "Launch priemka for search"
        params.append(BinaryCheckbox)
    return params


class PriemkaAllMiddlesearchBinaries(bugbanner.BugBannerTask):
    """
        Задача-агрегатор для задач приемки средних метапоисков v3.0
    """
    type = "PRIEMKA_ALL_MIDDLESEARCH_BINARIES"


    input_parameters = [
        rm_params.ComponentName,
        rm_params.ReleaseNum,
        TriggerFail,
        pmb_params.AcceptanceType,
        pmb_params.BuildTaskNew,
        pmb_params.BuildTaskOld,
        CfgFromHamster,
        RevisionTested,
        cmb.CompareBinariesResultResourceTTL,
    ] + binaries_to_compare(pmb_params.BIN_TYPES)
    client_tags = task_env.TaskTags.startrek_client
    cores = 1
    execution_space = 5000  # for tools/virtualenv
    environment = [task_env.TaskRequirements.startrek_client]

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.ReleaseMachine, component="priemka")

        if "launched_subtasks" not in self.ctx:
            self.ctx["launched_subtasks"] = []
            for bin_name in pmb_params.BIN_TYPES:
                if self.ctx.get(_CTX_NAME_TEMPLATE.format(bin_name)):
                    input_ctx = {
                        "notify_via": "email",
                        pmb_params.AcceptanceType.name: self.ctx[pmb_params.AcceptanceType.name],
                        pmb_params.BuildTaskNew.name: self.ctx[pmb_params.BuildTaskNew.name],
                        pmb_params.BuildTaskOld.name: self.ctx[pmb_params.BuildTaskOld.name],
                        "binary_type": bin_name,
                        rm_const.COMPONENT_CTX_KEY: self.ctx[rm_const.COMPONENT_CTX_KEY],
                        cmb.CompareBinariesResultResourceTTL.name: utils.get_or_default(
                            self.ctx,
                            cmb.CompareBinariesResultResourceTTL
                        )
                    }
                    input_ctx.update(self._set_notifications(bin_name))
                    self._handle_hamster_cfg_option(bin_name, input_ctx)
                    if bin_name in {"img", "imgquick", "video"}:
                        # tmp solution for https://st.yandex-team.ru/BETATEST-3053#1468854543000
                        input_ctx.update({
                            "models_source_for_new_mmeta": "last_released",
                            "models_source_for_old_mmeta": "last_released",
                        })
                    if bin_name == "img":
                        input_ctx.update({
                            "models_source_for_new_int": "specified",
                            "models_resource_for_new_int": "1573837010",
                            "models_source_for_old_int": "specified",
                            "models_resource_for_old_int": "1573837010",
                        })
                    if bin_name in {"quick"}:  # SEARCH-6459
                        input_ctx.update({
                            "data_source_for_new_mmeta": "last_released",
                            "data_source_for_old_mmeta": "last_released",
                        })
                    self.ctx["launched_subtasks"].append(self.create_subtask(
                        task_type='PRIEMKA_MIDDLESEARCH_BINARY',
                        description="{}, {}".format(self.descr, bin_name),
                        input_parameters=input_ctx,
                    ).id)
        requested_components = ','.join(bin_name for bin_name in pmb_params.BIN_TYPES if self.ctx.get(_CTX_NAME_TEMPLATE.format(bin_name)))
        acceptance.notify_st_on_acceptance(
            self, rmc.get_component(MiddleCfg.name),
            new_build_task_id=self.ctx[pmb_params.BuildTaskNew.name],
            old_build_task_id=self.ctx[pmb_params.BuildTaskOld.name],
            rev=self.ctx.get(RevisionTested.name),
            component_extra=requested_components,
            use_grouped_comment=True,
        )
        utils.check_subtasks_fails(fail_on_failed_children=utils.get_or_default(self.ctx, TriggerFail))

    def _handle_hamster_cfg_option(self, bin_name, input_ctx):
        if utils.get_or_default(self.ctx, CfgFromHamster):
            if bin_name == "quick":
                quick_hamster_cfg_id = utils.get_and_check_last_resource_with_attribute(
                    resource_types.MIDDLESEARCH_CONFIG,
                    attr_name="TE_quick_hamster_mmeta"
                ).id
                input_ctx.update({
                    "config_source_for_new_mmeta": "specified",
                    "config_source_for_old_mmeta": "specified",
                    "config_resource_for_new_mmeta": quick_hamster_cfg_id,
                    "config_resource_for_old_mmeta": quick_hamster_cfg_id,
                })
            else:
                input_ctx.update({
                    "config_source_for_new_mmeta": "from_hamster",
                    "config_source_for_old_mmeta": "from_hamster",
                    "config_source_for_new_int": "from_hamster",
                    "config_source_for_old_int": "from_hamster",
                })

    def _set_notifications(self, bin_name):
        if "release_number" not in self.ctx:
            return {
                'notify_if_finished': '',
                'notify_if_failed': '',
            }
        if bin_name == "video":
            return {
                'notify_if_finished': '',
                'notify_if_failed': 'mincer, juver',
            }
        return {}

    @property
    def footer(self):
        subtask_list = self.list_subtasks(load=True)
        subtasks = [
            {
                "Task id": lb.task_link(task.id),
                "Description": task.descr,
                "Status": "<span class='status status_{0}'>{0}</span>".format(task.status.lower())
            }
            for task in subtask_list
        ]
        return {"<h3>Subtask statuses</h3>": subtasks}


__Task__ = PriemkaAllMiddlesearchBinaries
