# -*- coding: utf-8 -*-

import json
import logging

from sandbox.common import rest
from sandbox.common.types.task import Status
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects.websearch.middlesearch import CompareMiddlesearchBinaries as cmb
from sandbox.projects.websearch import CalcEventlogStats
from sandbox.projects import ParallelDumpEventlog as pde
from sandbox.projects.common.search import functional_helpers as fh
from sandbox.projects.common.search import compare_middle_utils as cmu
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.differ import coloring
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import footers
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from . import params
from . import res

shooting_params = cmu.create_shooting_params()

TYPE_TO_ADD_CTX_KEY = {
    "web": "web_hamster",
    "img": "img_hamster_img",
    "imgquick": "img_hamster_img",
    "cbir": "img_hamster_img",
    "video": "video_hamster_video",
    "itditp": "web_hamster",
}
METASEARCH_COMPARE_BIN_MODES = {
    "int": ["nocache"],
    "mmeta": ["nocache", "cache", "cachehit", "cache_compatibility"],
}
MODELS_COMPARE_BIN_MODELS = {
    "int": [],
    "mmeta": ["cachehit"],
}
CACHE_METASEARCH_ONLY_COMPARE_BIN_MODES = {
    "int": [],
    "mmeta": ["cache"]
}
MARKS = ["old", "new"]
_MEMUSAGE_THRESHOLD = {  # per cent
    "int": 15.0,
    "mmeta": 3.0,
}
_MEMUSAGE_KEY = "memusage_task_{}_{}_id"
_FUNC_KEY = "functional_task_{}_id"
_COMPARE_BIN_KEY = "compare_binary_task_{}_{}_id"
_COMPARE_RESPONSES_KEY = "compare_responses_task_id"
_EXTRA_CGI = {
    "int": "&pron=ignore_db_timestamp",
    "mmeta": ""
}


class PriemkaMiddlesearchBinary(bugbanner.BugBannerTask):
    """
        Таск приемки средних поисков v3.0
        https://st.yandex-team.ru/SEARCH-1492
    """
    type = "PRIEMKA_MIDDLESEARCH_BINARY"

    input_parameters = params.all_input + [
        cmb.CompareBinariesResultResourceTTL,
    ]
    execution_space = 50 * 1024  # 50 Gb
    cores = 1

    @property
    def footer(self):
        client = rest.Client()
        foot = []
        foot.extend(self._get_functional_foot(client))
        foot.extend(self._get_memusage_foot(client))
        foot.extend(self._get_binary_compare_foot(client))
        foot.extend(self._get_responses_compare_foot(client))
        foot.extend(self._get_subsource_diff_foot(client))
        return foot

    def _get_functional_foot(self, client):
        foot = []
        if self.ctx.get(params.TestsToRun.TestFunctional.name):
            header = [
                {"key": "Task", "title": "Task"},
                {"key": "Status", "title": "Status"},
            ]
            body = {i["key"]: [] for i in header}
            for bin_level in params.BIN_LEVELS:
                child_id = self.ctx.get(_FUNC_KEY.format(bin_level))
                if child_id:
                    info = client.task.read(
                        id=child_id,
                        fields="status",
                        children=True,
                        hidden=True,
                        limit=1,
                    ).get("items", [{}])[0]
                    body["Task"].append(lb.task_link(child_id, "{}".format(bin_level)))
                    body["Status"].append(utils.colored_status(info["status"]))
            foot.append({
                "helperName": "",
                "content": {"<h3>Functional tests result</h3>": {
                    "header": header,
                    "body": body
                }},
            })
        return foot

    def _get_memusage_foot(self, client):
        foot = []
        if self.ctx.get(params.TestsToRun.TestMemoryUsage.name):
            for bin_level in params.BIN_LEVELS:
                if self.ctx.get("compare_{}".format(bin_level)):
                    foot.append(footers.memusage_footer(
                        [self.ctx.get(_MEMUSAGE_KEY.format(mark, bin_level)) for mark in MARKS],
                        client,
                        "<h3>Memory usage tests for {}</h3>".format(bin_level),
                        diff_threshold=_MEMUSAGE_THRESHOLD[bin_level]
                    ))
        return foot

    def _get_binary_compare_foot(self, client):
        if self.ctx.get(params.TestsToRun.TestBinaries.name):
            header = [
                {"key": "Task", "title": "Task"},
                {"key": "Status", "title": "Status"},
                {"key": "Accept", "title": "Accept"},
                {"key": "Explanation", "title": "Explanation"},
                {"key": "Summary", "title": "Summary"},
            ]
            body = {i["key"]: [] for i in header}
            for bin_level in params.BIN_LEVELS:
                for auto_mode in self._get_compare_bin_modes(bin_level):
                    if self.ctx.get("compare_{}".format(bin_level)):
                        child_id = self.ctx.get(_COMPARE_BIN_KEY.format(auto_mode, bin_level))
                        if child_id:
                            info = client.task.read(
                                id=child_id,
                                fields="status,context.auto_accepted,context.explanation,context.summary",
                                children=True,
                                hidden=True,
                                limit=1,
                            ).get("items", [{}])[0]
                            body["Task"].append(lb.task_link(child_id, "{} {}".format(bin_level, auto_mode)))
                            body["Status"].append(utils.colored_status(info["status"]))
                            body["Accept"].append(info["context.auto_accepted"])
                            body["Explanation"].append(info["context.explanation"])
                            body["Summary"].append(info["context.summary"])
            return [{
                "helperName": "",
                "content": {"<h3>Compare binaries performance</h3>": {
                    "header": header,
                    "body": body
                }},
            }]
        return []

    def _get_responses_compare_foot(self, client):
        if self.ctx.get(params.TestsToRun.TestResponses.name) and self.ctx.get("compare_mmeta"):
            child_id = self.ctx.get(_COMPARE_RESPONSES_KEY)
            if child_id:
                info = client.task.read(
                    id=child_id,
                    fields="context.diff_count",  # ,context.relevant_diff
                    children=True,
                    hidden=True,
                    limit=1,
                ).get("items", [{}])[0]
                return [{
                    "helperName": "",
                    "content": "<h3>Total: {} response diffs</h3>".format(info["context.diff_count"]),
                }]
        return []

    def _get_subsource_diff_foot(self, client):
        if not utils.get_or_default(self.ctx, params.CompareSubSourceRequests):
            return []

        compare_task_id = self.ctx.get(_COMPARE_BIN_KEY.format('nocache', 'mmeta'))
        if not compare_task_id:
            return []
        compare_task = channel.sandbox.get_task(compare_task_id)

        header = [
            {"key": "task_id", "title": "Task"},
            {"key": "resource_id", "title": "Diff resource link"},
        ]
        body = {i["key"]: [] for i in header}

        resource_infos = compare_task.ctx.get(cmb.CompareMiddlesearchBinaries._KEY_SUBSOURCE_REQUESTS_DIFF)
        if resource_infos:
            for task_id, resource_id in resource_infos:
                body["task_id"].append(lb.task_link(task_id))
                body["resource_id"].append(lb.resource_link(resource_id))

        title = "Compare subsource requests result" + ("" if resource_infos else " is not ready yet")
        return [{
            "helperName": "",
            "content": {("<h3>{}</h3>".format(title)): {
                "header": header,
                "body": body,
            }}
        }]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        # if self.ctx[params.BinType.name] == "cbir":
        #     self.ctx["compare_int"] = False
        # int for quick doesn't exist
        if self.ctx[params.BinType.name] in ("quick", "imgquick", "itditp"):
            self.ctx["compare_int"] = False
            self.ctx[params.TestsToRun.TestMemoryUsage.name] = False

    def on_execute(self):
        if not self.ctx.get("launched_subtasks"):
            self.add_bugbanner(bugbanner.Banners.WebMiddleSearch)

            self.ctx["launched_subtasks"] = True
            common_cgi_params = utils.get_or_default(self.ctx, params.AddCgi)
            new_additional_cgi_params = utils.get_or_default(self.ctx, params.AddToNewCgi)
            if utils.get_or_default(self.ctx, params.RearrangeAdjustersStat):
                common_cgi_params += "&debug=logpreparesrcs"
            for bin_level in params.BIN_LEVELS:
                if self.ctx.get(params.CTX_LEVEL_TEMPLATE.format(bin_level)):
                    cfg_sources = [
                        self.ctx.get(params.SOURCE_TEMPLATE.format("config", mark, bin_level)) for mark in MARKS
                    ]
                    # it's useless to compare prod and hamster configs between each other
                    # so use prod only if all sources use prod, otherwise use hamster
                    hamster_cfg = any(cfg_source == params.SourceType.FROM_HAMSTER for cfg_source in cfg_sources)
                    common_cgi_params = common_cgi_params + _EXTRA_CGI.get(bin_level, "")
                    r = res.MiddleResourceKeeper(bin_level, self.ctx, common_cgi_params, hamster_cfg=hamster_cfg, new_additional_cgis_str=new_additional_cgi_params)
                    self.compare_binaries(r)
                    self.test_functional(r)
                    self.test_memory_usage(r)
                    self.compare_responses(r)
                else:
                    logging.info("No need to test %s level", bin_level)
        if not utils.check_all_subtasks_done():
            utils.restart_broken_subtasks(use_rest=True)

        subtasks_info = utils.subtasks_id_and_status(self.id)
        failed_subtasks = [
            lb.task_wiki_link(task.get("id")) for task in subtasks_info if task.get("status") == Status.FAILURE
        ]
        if failed_subtasks:
            self.ctx[cmb.KEY_EXPLANATION] = "Child tasks with **!!FAILURE!!** status : {}".format(
                ', '.join(failed_subtasks)
            )
        utils.check_subtasks_fails()
        self.check_memusage()
        self.check_binaries()

    def compare_responses(self, res_keeper):
        if not self.ctx[params.TestsToRun.TestResponses.name] or res_keeper.bin_level == "int":
            return
        logging.info("Task for responses comparison is under construction")

    def compare_binaries(self, res_keeper):
        if not self.ctx[params.TestsToRun.TestBinaries.name]:
            return
        if utils.get_or_default(self.ctx, params.AcceptanceType) == params.AcceptanceType.MODELS_ACCEPTANCE:
            auto_diff_mode = "formulas"
        else:
            auto_diff_mode = "binary"
        for auto_mode in self._get_compare_bin_modes(res_keeper.bin_level):
            input_ctx = {
                "notify_via": "",
                cmb.AutoModeParam.name: auto_mode,
                cmb.AutoDiffMode.name: auto_diff_mode,
                shooting_params.Nruns.name: 10,
                shooting_params.ReqCount.name: 5000,
                shooting_params.WarmupRequestCount.name: 100,
                shooting_params.Rps.name: 20 if auto_mode == "cache" else 10,
                pde.RequestsTimeoutMilliseconds.name: self.ctx.get(pde.RequestsTimeoutMilliseconds.name),
                cmb.CheckSubSourceErrorCount.name: self.ctx.get(params.CheckSubSourceErrorCount.name, True),
                CalcEventlogStats.CompareSubSourceRequests.name:
                    utils.get_or_default(self.ctx, params.CompareSubSourceRequests) and auto_mode == 'nocache',
                CalcEventlogStats.stat_params.RearrangeAdjustersStat.name:
                    utils.get_or_default(self.ctx, params.RearrangeAdjustersStat),
                cmb.CompareBinariesResultResourceTTL.name:
                    utils.get_or_default(self.ctx, cmb.CompareBinariesResultResourceTTL)
            }
            for i, mark in enumerate(MARKS):
                input_ctx.update({
                    cmb.res_params[i].Binary.name: res_keeper.binary(mark),
                    cmb.res_params[i].Config.name: res_keeper.config(mark),
                    cmb.res_params[i].Evlogdump.name: res_keeper.evlogdump(mark),
                    cmb.res_params[i].Requests.name: res_keeper.queries(mark),
                    cmb.res_params[i].Data.name: res_keeper.data(mark),
                    cmb.res_params[i].Index.name: res_keeper.index(mark),
                    cmb.res_params[i].ArchiveModel.name: res_keeper.models(mark),
                    cmb.res_params[i].UseInt.name: res_keeper.bin_level == "int",
                })
            if self.ctx[params.BinType.name] in ("img", "video") and res_keeper.bin_level == "mmeta":
                input_ctx["stress_limiter_semaphore"] = "CompareMiddlesearchBinaries_semaphore_" + self.ctx[params.BinType.name]
                input_ctx["stress_limiter_capacity"] = 2
            # We put this as a default param because this param is required but it does not influence anything here
            input_ctx["component_name"] = "release_machine_test_tagged"
            create_task_params = {
                "task_type": "COMPARE_MIDDLESEARCH_BINARIES",
                "input_parameters": input_ctx,
                "description": "Compare {} binaries (auto_mode='{}'), {}".format(
                    res_keeper.bin_level, auto_mode, self.descr
                ),
                "arch": sandboxapi.ARCH_LINUX,
            }
            logging.info("Params for compare binaries:\n%s", json.dumps(create_task_params, indent=2))
            self.ctx[_COMPARE_BIN_KEY.format(auto_mode, res_keeper.bin_level)] = self.create_subtask(
                **create_task_params
            ).id

    def test_functional(self, res_keeper):
        # SEARCH-1533
        if not self.ctx[params.TestsToRun.TestFunctional.name]:
            return
        if self.ctx[params.BinType.name] in ["img", "imgquick", "cbir", "video"] and res_keeper.bin_level == "int":
            # test will fail because of absence quick collection, so do not launch it
            return
        input_ctx = self._common_input_ctx(res_keeper, "new")
        input_ctx.update({
            "queries_resource_id": res_keeper.queries("new"),
            "search_type": fh.MIDDLE,
            "search_subtype": (
                self.ctx[params.BinType.name] if self.ctx[params.BinType.name] not in ["img", "imgquick", "cbir"]
                else "images"
            ),
            'kill_timeout': 2 * 60 * 60,  # 2 hours
        })
        create_task_params = {
            'task_type': 'FUNCTIONAL_TEST_SEARCH',
            'input_parameters': input_ctx,
            'description': 'Func test for new {}, {}'.format(res_keeper.bin_level, self.descr),
            "arch": sandboxapi.ARCH_LINUX,
        }
        logging.info("Params for functional test:\n%s", json.dumps(create_task_params, indent=2))
        self.ctx[_FUNC_KEY.format(res_keeper.bin_level)] = self.create_subtask(**create_task_params).id

    def test_memory_usage(self, res_keeper):
        """
            Uses specified config by default. In other cases uses hamster config
        """
        if not self.ctx[params.TestsToRun.TestMemoryUsage.name]:
            return
        for mark in MARKS:
            input_ctx = self._common_input_ctx(res_keeper, mark)
            cfg_id = input_ctx.get(sc.DefaultMiddlesearchParams.Config.name)
            if not cfg_id:
                logging.info("test_memory_usage: fail to get config from _common_input_ctx")
                hamster_cfg_key = "special_{}_hamster_cfg".format(res_keeper.bin_level)
                if not self.ctx.get(hamster_cfg_key):
                    self.ctx[hamster_cfg_key] = self._get_hamster_config_id(res_keeper.bin_level)
                eh.check_failed(self.ctx.get(hamster_cfg_key) is not None)
                input_ctx[sc.DefaultMiddlesearchParams.Config.name] = self.ctx[hamster_cfg_key]
            input_ctx.update({
                "process_count": 4,  # do not overshoot basesearch (SEARCH-1044)
                "replace_basesearches": False,
                "dolbilo_plan_resource_id": res_keeper.plan(mark),
                "dolbilka_executor_max_simultaneous_requests": 4,  # modern version of process_count
            })
            create_task_params = {
                'task_type': 'MEASURE_MIDDLESEARCH_MEMORY_USAGE',
                'input_parameters': input_ctx,
                'description': 'Measure memusage for {} {}, {}'.format(mark, res_keeper.bin_level, self.descr),
                "arch": sandboxapi.ARCH_LINUX,
            }
            logging.info("Params for %s memory_usage test:\n%s", mark, json.dumps(create_task_params, indent=2))
            self.ctx[_MEMUSAGE_KEY.format(mark, res_keeper.bin_level)] = self.create_subtask(
                **create_task_params
            ).id

    @staticmethod
    def _common_input_ctx(res_keeper, mark):
        return {
            'notify_via': '',
            sc.DefaultMiddlesearchParams.Binary.name: res_keeper.binary(mark),
            sc.DefaultMiddlesearchParams.Config.name: res_keeper.config(mark),
            sc.DefaultMiddlesearchParams.Data.name: res_keeper.data(mark),
            sc.DefaultMiddlesearchParams.ArchiveModel.name: res_keeper.models(mark),
            sc.DefaultMiddlesearchParams.Index.name: res_keeper.index(mark),
            sc.DefaultMiddlesearchParams.UseInt.name: res_keeper.bin_level == "int",
        }

    def _get_hamster_config_id(self, bin_level):
        return utils.get_and_check_last_resource_with_attribute(
            resource_type='MIDDLESEARCH_CONFIG',
            attr_name="TE_{}_{}_cfg".format(TYPE_TO_ADD_CTX_KEY[self.ctx[params.BinType.name]], bin_level),
        ).id

    def check_memusage(self):
        client = rest.Client()
        if not self.ctx.get(params.TestsToRun.TestMemoryUsage.name):
            return

        for bin_level in params.BIN_LEVELS:
            if not self.ctx.get("compare_{}".format(bin_level)):
                continue

            new_mem = footers.get_task_fields(
                client, self.ctx[_MEMUSAGE_KEY.format("new", bin_level)], "context.memory_bytes"
            ).get("context.memory_bytes")[0]
            old_mem = footers.get_task_fields(
                client, self.ctx[_MEMUSAGE_KEY.format("old", bin_level)], "context.memory_bytes"
            ).get("context.memory_bytes")[0]
            if new_mem is None or old_mem is None:
                logging.error("Unable to compare memory usage! Check 'memory_bytes' field in child context")
                continue
            for i in ["rss", "vms", "uss", "pss", "anon", "shared", "vmlck"]:
                if i not in old_mem or i not in new_mem:
                    logging.info("Unable to find memory usage info of type '%s', skip it", i)
                    return
                diff = 100 * (new_mem[i] - old_mem[i]) / float(old_mem[i]) if old_mem[i] else None
                if coloring.color_diff(diff, _MEMUSAGE_THRESHOLD[bin_level]) == coloring.DiffColors.bad:
                    error_msg = "Limit of '{}' memory type for {} exceeded!".format(i, bin_level)
                    self.ctx[cmb.KEY_EXPLANATION] = "Task is not accepted. Reason: {}".format(error_msg)
                    eh.check_failed(error_msg)
                else:
                    logging.info("%s memory usage of type '%s' is OK", i, bin_level)

    def check_binaries(self):
        if not self.ctx.get(params.TestsToRun.TestBinaries.name):
            return

        for bin_level in params.BIN_LEVELS:
            for auto_mode in self._get_compare_bin_modes(bin_level):
                if not self.ctx["compare_{}".format(bin_level)]:
                    continue

                child_id = self.ctx.get(_COMPARE_BIN_KEY.format(auto_mode, bin_level))
                eh.ensure(
                    child_id,
                    "Compare '{}' binary task with mode '{}' doesn't exist".format(bin_level, auto_mode)
                )
                child = channel.sandbox.get_task(child_id)
                if not child.ctx.get(cmb.KEY_AUTO_ACCEPTED):
                    message = "Task {} is not accepted. Reason:<br />{}. Mode: {}".format(
                        lb.task_link(child_id), child.ctx.get(cmb.KEY_EXPLANATION), auto_mode
                    )
                    if auto_mode == 'cache_compatibility':
                        message += '<br /> <br />Please see links to <b>diff of subsource requests in the footer</b>'
                    self.set_info(message, do_escape=False)

                    self.ctx[cmb.KEY_EXPLANATION] = (
                        "**{}:{}** Task {} is not accepted. Reason: <# {} #>".format(
                            bin_level, auto_mode,
                            lb.task_wiki_link(child_id), child.ctx.get(cmb.KEY_EXPLANATION, "Unknown")
                        )
                    )
                    eh.check_failed(
                        "Priemka failed as child sandboxtask:{} is not accepted".format(child_id)
                    )
                else:
                    logging.info("Task '%s' is auto accepted", child_id)

    def _get_compare_bin_modes(self, bin_level):
        type_to_modes = {
            params.AcceptanceType.MODELS_ACCEPTANCE: MODELS_COMPARE_BIN_MODELS[bin_level],
            params.AcceptanceType.METASEARCH_ACCEPTANCE: METASEARCH_COMPARE_BIN_MODES[bin_level],
            params.AcceptanceType.CACHE_METASEARCH_ONLY_ACCEPTANCE: CACHE_METASEARCH_ONLY_COMPARE_BIN_MODES[bin_level]
        }
        try:
            return type_to_modes[utils.get_or_default(self.ctx, params.AcceptanceType)]
        except KeyError as e:
            return METASEARCH_COMPARE_BIN_MODES[bin_level]


__Task__ = PriemkaMiddlesearchBinary
