# -*- coding: utf-8 -*-
import sandbox.projects.websearch.middlesearch.resources as ms_resources

import sandbox.projects.release_machine.input_params as rm_params

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.search import cluster_names
from sandbox.projects.images.models import resources as images_models_resources
from sandbox.projects.images.metasearch import resources as images_metasearch_resources
from sandbox.projects.VideoSearch import video_resource_types
from sandbox.projects import ParallelDumpEventlog as pde

from sandbox.sandboxsdk import parameters as sp

BIN_TYPES = {
    "web": True,
    "img": True,
    "imgquick": True,
    "cbir": True,
    "video": True,
    "quick": True,
    "itditp": False,
}
BIN_LEVELS = ["mmeta"]
CTX_LEVEL_TEMPLATE = "compare_{}"
SOURCE_TEMPLATE = "{}_source_for_{}_{}"
RESOURCE_TEMPLATE = "{}_resource_for_{}_{}"
REQUIRED_RES = ["config", "queries"]
IMG_MODELS_RES_TYPES = [
    images_models_resources.IMAGES_MIDDLESEARCH_DYNAMIC_MODELS_ARCHIVE,
    images_models_resources.IMAGES_MIDDLESEARCH_PROD_DYNAMIC_MODELS_ARCHIVE,
    images_models_resources.IMAGES_MIDDLESEARCH_EXP_DYNAMIC_MODELS_ARCHIVE,
]
CLUSTERSTATE_MMETA_CONF = {
    "web": cluster_names.C.vla_jupiter_mmeta,
    "img": cluster_names.C.production_imgmmeta_man,
    "imgquick": cluster_names.C.production_imgmmeta_man,
    "cbir": cluster_names.C.production_imgmmeta_man,
    "video": cluster_names.C.production_vidmmeta_man,
    "quick": cluster_names.C.vla_jupiter_mmeta,
    "itditp": cluster_names.C.itditp_mmeta,
}
FETCH_PROD_BINARY_FROM = {
    "web": {
        "mmeta": "vla_jupiter_mmeta_yp",
        "int": "vla-web-int-resources",
    },
    "img": {
        "mmeta": "production_imgmmeta_vla",
        "int": "production_imgsint_vla",
    },
    "imgquick": {
        "mmeta": "production_imgmmeta_vla",
    },
    "cbir": {
        "mmeta": "production_imgmmeta_vla",
        "int": "production_imgscbrint_vla",
    },
    "video": {
        "mmeta": "production_vidmmeta_vla",
        "int": "vla_video_int",
    },
    "quick": {
        "mmeta": "saas-refresh-yp-shmick-mmeta",
    },
    "itditp": {
        "mmeta": "vla_itditp_mmeta_yp",
    }
}
RES_TYPES = {
    "binary": {
        "web": {
            "mmeta": ms_resources.RankingMiddlesearchExecutable,
            "int": ms_resources.L1IntsearchExecutable,
        },
        "img": {
            "mmeta": resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE,
            "int": ms_resources.L1IntsearchExecutable,
        },
        "imgquick": {
            "mmeta": resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE,
        },
        "cbir": {
            "mmeta": resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE,
            "int": ms_resources.IntsearchExecutable,
        },
        "video": {
            "mmeta": resource_types.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE,
            "int": video_resource_types.VIDEO_L1_INTSEARCH_EXECUTABLE,
        },
        "quick": {
            "mmeta": resource_types.FRESH_RANKING_MIDDLESEARCH_EXECUTABLE,
        },
        "itditp": {
            "mmeta": ms_resources.ItditpMiddlesearchExecutable
        },
    },
    "evlogdump": resource_types.EVLOGDUMP_EXECUTABLE,
    "evlogcachehit": resource_types.EVLOGCACHEHIT_EXECUTABLE,
    "config": {
        "web": resource_types.MIDDLESEARCH_CONFIG,
        "img": images_metasearch_resources.IMAGES_MIDDLESEARCH_CONFIG,
        "imgquick": images_metasearch_resources.IMAGES_MIDDLESEARCH_CONFIG,
        "cbir": images_metasearch_resources.IMAGES_MIDDLESEARCH_CONFIG,
        "video": resource_types.VIDEO_MIDDLESEARCH_CONFIG,
        "quick": resource_types.MIDDLESEARCH_CONFIG,
        "itditp": resource_types.MIDDLESEARCH_CONFIG,
    },
    "queries": {
        "web": [
            ms_resources.WebMiddlesearchPlainTextQueries,
            resource_types.PLAIN_TEXT_QUERIES,
        ],
        "img": resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        "imgquick": resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        "cbir": resource_types.IMAGES_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        "video": resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
        "quick": resource_types.PLAIN_TEXT_QUERIES,
        "itditp": [
            ms_resources.WebMiddlesearchPlainTextQueries,
            resource_types.PLAIN_TEXT_QUERIES,
        ],
    },
    "plan": {
        "web": ms_resources.MiddlesearchPlan,
        "img": resource_types.IMAGES_MIDDLESEARCH_PLAN,
        "imgquick": resource_types.IMAGES_MIDDLESEARCH_PLAN,
        "cbir": resource_types.IMAGES_MIDDLESEARCH_PLAN,
        "video": resource_types.VIDEO_MIDDLESEARCH_PLAN,
        "quick": ms_resources.MiddlesearchPlan,
        "itditp": ms_resources.MiddlesearchPlan,
    },
    "models": {
        "web": {
            "mmeta": resource_types.DYNAMIC_MODELS_ARCHIVE,
            "int": "DYNAMIC_MODELS_ARCHIVE_L1",
        },
        "img": {
            "mmeta": IMG_MODELS_RES_TYPES,
            "int": images_models_resources.IMAGES_L1_DYNAMIC_MODELS_ARCHIVE,
        },
        "imgquick": {
            "mmeta": IMG_MODELS_RES_TYPES,
        },
        "cbir": {
            "mmeta": IMG_MODELS_RES_TYPES,
        },
        "video": {
            "mmeta": resource_types.VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE,
            "int": video_resource_types.VIDEO_L1_DYNAMIC_MODELS_ARCHIVE,
        },
        "quick": {
            "mmeta": resource_types.DYNAMIC_MODELS_ARCHIVE,
        },
        "itditp": {
            "mmeta": resource_types.DYNAMIC_MODELS_ARCHIVE,
        },
    },
    "data": {
        "web": resource_types.MIDDLESEARCH_DATA,
        "img": images_metasearch_resources.IMAGES_MIDDLESEARCH_DATA,
        "imgquick": images_metasearch_resources.IMAGES_MIDDLESEARCH_DATA,
        "cbir": images_metasearch_resources.IMAGES_MIDDLESEARCH_DATA,
        "video": resource_types.VIDEO_MIDDLESEARCH_DATA,
        "quick": resource_types.QUICK_REARRANGE_RULES_DATA,
        "itditp": resource_types.MIDDLESEARCH_DATA,
    },
    "index": {
        "img": resource_types.IMAGES_MIDDLESEARCH_INDEX,
        "imgquick": resource_types.IMAGES_MIDDLESEARCH_INDEX,
        "cbir": resource_types.IMAGES_MIDDLESEARCH_INDEX,
        "video": resource_types.VIDEO_MIDDLESEARCH_INDEX,
    },
}


class SourceType:
    FROM_TESTENV = 'from_testenv'
    FROM_HAMSTER = 'from_hamster'
    LAST_RELEASED = 'last_released'
    SPECIFIED = 'specified'


class BinType(sp.SandboxStringParameter):
    name = "binary_type"
    description = "Binary type"
    choices = zip(BIN_TYPES.keys(), BIN_TYPES.keys())
    default_value = BIN_TYPES.keys()[0]
    required = True


class AcceptanceType(sp.SandboxStringParameter):
    METASEARCH_ACCEPTANCE = "metasearch"
    MODELS_ACCEPTANCE = "models"
    CACHE_METASEARCH_ONLY_ACCEPTANCE = "cache_metasearch_only"

    name = 'acceptance_type'
    description = 'Acceptance type'
    choices = [("Metasearch", METASEARCH_ACCEPTANCE), ("Models", MODELS_ACCEPTANCE), ("Cache metasearch only", CACHE_METASEARCH_ONLY_ACCEPTANCE)]
    default_value = METASEARCH_ACCEPTANCE
    sub_fields = {METASEARCH_ACCEPTANCE: [CTX_LEVEL_TEMPLATE.format("int")]}


class AddCgi(sp.SandboxStringParameter):
    name = 'new_cgi_params'
    description = (
        'Additional cgi params. Simple version. Expects one line like: "&param1=val1&param2&param3=val3". Applied to both baseline and test'
    )


class AddToNewCgi(sp.SandboxStringParameter):
    name = 'add_to_2_cgi_params'
    description = (
        'Additional cgi params. Expects one line like: "&param1=val1&param2&param3=val3". Applied to test only'
    )


def compare_these(names, sub_params=None):
    params = []
    for n in names:
        class Checkbox(sp.SandboxBoolParameter):
            name = CTX_LEVEL_TEMPLATE.format(n)
            description = "Compare {}".format(n)
            sub_fields = {"true": [r.name for r in sub_params[n] if "_source_" in r.name]} if sub_params else None
            required = False
            default_value = True
            group = "Compare or not"
        params.append(Checkbox)
    return params


class TestsToRun(object):
    class TestResponses(sp.SandboxBoolParameter):
        name = 'run_test_responses'
        description = 'Test responses'
        default_value = False
        required = False
        group = "Tests to run"

    class TestFunctional(sp.SandboxBoolParameter):
        name = 'run_test_functional'
        description = 'Test functional'
        default_value = True
        required = False
        group = "Tests to run"

    class TestBinaries(sp.SandboxBoolParameter):
        name = 'run_test_binaries'
        description = 'Test binaries'
        default_value = True
        required = False
        group = "Tests to run"

    class TestMemoryUsage(sp.SandboxBoolParameter):
        name = 'run_test_memory_usage'
        description = 'Test memory usage (SEARCH-1081)'
        default_value = True
        required = False
        group = "Tests to run"

    params = [
        TestResponses,
        TestFunctional,
        TestBinaries,
        TestMemoryUsage,
    ]


def create_resource_input(bin_level):
    group_name = "{} resources".format(bin_level)
    default_choices = [
        ("From testenv", SourceType.FROM_TESTENV),
        ("Last released", SourceType.LAST_RELEASED),
        ("Specify", SourceType.SPECIFIED),
    ]
    extra_cfg_choice = [("From hamster", SourceType.FROM_HAMSTER)]

    class ResourcesInput(object):
        class ResPair(object):
            def __init__(self, resource_name, mark):
                class Source(sp.SandboxStringParameter):
                    name = SOURCE_TEMPLATE.format(resource_name, mark, bin_level)
                    description = "{} source {}".format(resource_name, mark)
                    choices = default_choices if resource_name != "config" else default_choices + extra_cfg_choice
                    sub_fields = {SourceType.SPECIFIED: [RESOURCE_TEMPLATE.format(resource_name, mark, bin_level)]}
                    default_value = SourceType.FROM_HAMSTER
                    group = group_name

                class Resource(sp.ResourceSelector):
                    name = RESOURCE_TEMPLATE.format(resource_name, mark, bin_level)
                    description = "Existing {} {}".format(resource_name, mark)
                    resource_type = list(utils.flatten(r.values() if type(r) is dict else r for r in utils.flatten(RES_TYPES.get(resource_name, {}).values())))
                    required = resource_name in REQUIRED_RES
                    group = group_name

                self.source = Source
                self.resource = Resource

        resource_names = {
            "mmeta": {
                "different": ["config", "models", "data", "index", "queries", "plan"],
                "same": [],  # resources that should be equal for both searches
            },
            "int": {
                "different": ["config", "models", "queries", "plan"],
                "same": [],
            },
        }
        params = []
        for res_name in resource_names[bin_level]["different"]:
            for mark in ["old", "new"]:
                res_pair = ResPair(res_name, mark)
                params.extend([res_pair.source, res_pair.resource])
        for res_name in resource_names[bin_level]["same"]:
            res_pair = ResPair(res_name, "same")
            params.extend([res_pair.source, res_pair.resource])

    return ResourcesInput


resource_input = []
resources_dict = {}
for level in BIN_LEVELS:
    resources_dict[level] = create_resource_input(level).params
    resource_input.extend(resources_dict[level])


class BuildTaskNew(sp.TaskSelector):
    name = "build_task_new"
    description = 'Build task new (if not specified - use production)'
    task_type = ["BUILD_MIDDLESEARCH", "BUILD_SEARCH", "BUILD_RANKING_MIDDLE_VIDEOSEARCH"]
    group = "Build tasks"

    @classmethod
    def get_custom_parameters(cls):
        result = super(BuildTaskNew, cls).get_custom_parameters()
        result.update({'show_childs': True})
        return result


class BuildTaskOld(sp.TaskSelector):
    """
        Not required param! In case of not specified - get current production binary
    """
    name = "build_task_old"
    description = 'Build task old (if not specified - use production)'
    task_type = ["BUILD_MIDDLESEARCH", "BUILD_SEARCH", "BUILD_RANKING_MIDDLE_VIDEOSEARCH"]
    group = "Build tasks"

    @classmethod
    def get_custom_parameters(cls):
        result = super(BuildTaskOld, cls).get_custom_parameters()
        result.update({'show_childs': True})
        return result


class CheckSubSourceErrorCount(sp.SandboxBoolParameter):
    """MIDDLE-43"""
    name = 'check_sub_source_error_count'
    description = 'Check count of SubSourceError events'
    default_value = True


class CompareSubSourceRequests(sp.SandboxBoolParameter):
    """MIDDLE-103"""
    name = 'compare_subsource_requests'
    description = 'Extract and compare SubSource requests'
    default_value = True


class RearrangeAdjustersStat(sp.SandboxBoolParameter):
    """
        Copy of RearrangeAdjustersStat from compare_middle_utils.py
    """
    name = 'calc_rearrange_adjusters_stat'
    description = 'whether to calc detailed stat within PrepareClients'
    default_value = True


all_input = [
    rm_params.ComponentName,
    BinType,
    AcceptanceType,
    AddCgi,
    AddToNewCgi,
    BuildTaskNew,
    BuildTaskOld,
] + (
    compare_these(BIN_LEVELS, resources_dict) +
    TestsToRun.params +
    resource_input
) + [
    pde.RequestsTimeoutMilliseconds,
    CheckSubSourceErrorCount,
    CompareSubSourceRequests,
    RearrangeAdjustersStat,
]
