# -*- coding: utf-8 -*-

import logging
import json

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import sandboxapi

from sandbox.projects.common.search.response import cgi
from sandbox.projects.common.search import settings as ms
from sandbox.projects.common.search import production
from sandbox.projects.common import apihelpers
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from . import params


class MiddleResourceKeeper(object):
    __evlogdump = {"old": None, "new": None}
    __evlogcachehit = {"old": None, "new": None}
    __prod_build_task_id = None

    def __init__(self, bin_level, ctx, additional_cgis_str='', hamster_cfg=True, new_additional_cgis_str=''):  # use hamster config here by default
        self.bin_level = bin_level  # mmeta or int
        self._task_ctx = ctx
        self._additional_cgis = dict(cgi.split_to_cgi(additional_cgis_str))
        self._new_additional_cgis = dict(cgi.split_to_cgi(new_additional_cgis_str))
        self._additional_cgis["reqinfo"] = "sandboxtaskid={}".format(channel.task.id)
        self._new_additional_cgis["reqinfo"] = "sandboxtaskid={}".format(channel.task.id)
        self.build_task_ids = {
            "old": self._get_build_task_id(ctx, bin_level, "old"),
            "new": self._get_build_task_id(ctx, bin_level, "new"),
        }
        self._testenv_res_ids = {}  # SEARCH-2766
        self._last_released_res_ids = {}  # SEARCH-2766
        self.info = {
            "binary": {"old": None, "new": None},
            "config": {"old": None, "new": None},
            "queries": {"old": None, "new": None},
            "plan": {"old": None, "new": None},
            "models": {"old": None, "new": None},
            # only for mmeta:
            "index": {"old": None, "new": None},
            "data": {"old": None, "new": None},
        }
        self.attrs = get_testenv_attrs(bin_level, hamster_cfg=hamster_cfg)

    def binary(self, mark):
        if not self.info["binary"][mark]:
            self.info["binary"][mark] = apihelpers.get_task_resource_id(
                self.build_task_ids[mark],
                params.RES_TYPES["binary"][self._task_ctx[params.BinType.name]][self.bin_level],
                sandboxapi.ARCH_LINUX
            )
        return self.info["binary"][mark]

    def evlogdump(self, mark):
        if not self.__evlogdump[mark]:
            self.__evlogdump[mark] = apihelpers.get_task_resource_id(
                self.build_task_ids[mark],
                params.RES_TYPES["evlogdump"],
                sandboxapi.ARCH_LINUX
            )
        return self.__evlogdump[mark]

    def evlogcachehit(self, mark):
        if not self.__evlogcachehit[mark]:
            self.__evlogcachehit[mark] = apihelpers.get_task_resource_id(
                self.build_task_ids[mark],
                params.RES_TYPES["evlogcachehit"],
                sandboxapi.ARCH_LINUX
            )
        return self.__evlogcachehit[mark]

    def config(self, mark):
        if not self.info["config"][mark]:
            self.info["config"][mark] = self._get_res("config", mark)
        return self.info["config"][mark]

    def queries(self, mark):
        if not self.info["queries"][mark]:
            res_id = self._get_res("queries", mark)
            cgis_dict = self._additional_cgis.copy()
            if mark == "new":
                cgis_dict.update(self._new_additional_cgis)
            self.info["queries"][mark] =_patch_queries(res_id, mark, cgis_dict)
        return self.info["queries"][mark]

    def plan(self, mark):
        if not self.info["plan"][mark]:
            self.info["plan"][mark] = self._get_res("plan", mark)
        return self.info["plan"][mark]

    def _get_res(self, res_name, mark):
        source = self._task_ctx.get(params.SOURCE_TEMPLATE.format(res_name, mark, self.bin_level))
        if source is None:
            logging.info("Source for %s %s is not specified, set default source = 'from_hamster'", mark, res_name)
            source = params.SourceType.FROM_HAMSTER

        resource_type = params.RES_TYPES[res_name][self._task_ctx[params.BinType.name]]
        if isinstance(resource_type, dict):
            resource_type = resource_type[self.bin_level]
        if isinstance(resource_type, (list, tuple)):
            # for example, img models have different resource types
            resource_type = resource_type[0]

        res_id = None
        if source in [params.SourceType.FROM_TESTENV, params.SourceType.FROM_HAMSTER]:
            if res_name not in self._testenv_res_ids:
                self._testenv_res_ids[res_name] = utils.get_and_check_last_resource_with_attribute(
                    resource_type=resource_type,
                    attr_name=self.attrs[res_name][self._task_ctx[params.BinType.name]].get("attr_name"),
                    attr_value=self.attrs[res_name][self._task_ctx[params.BinType.name]].get("attr_value"),
                ).id
            res_id = self._testenv_res_ids[res_name]
        elif source == params.SourceType.LAST_RELEASED:
            if res_name not in self._last_released_res_ids:
                self._last_released_res_ids[res_name] = utils.get_and_check_last_released_resource_id(
                    resource_type=resource_type,
                    arch=sandboxapi.ARCH_ANY if resource_type.any_arch else sandboxapi.ARCH_LINUX
                )
            res_id = self._last_released_res_ids[res_name]
        elif source == params.SourceType.SPECIFIED:
            res_id = self._task_ctx[params.RESOURCE_TEMPLATE.format(res_name, mark, self.bin_level)]
        else:
            eh.fail("Unknown source for {}: {}".format(res_name, source))
        return res_id

    def _get_build_task_id(self, ctx, bin_level, mark):
        """If not specified - get from production"""
        build_task_id = self._task_ctx.get("build_task_{}".format(mark))
        binary_res_type = params.RES_TYPES["binary"][ctx[params.BinType.name]][bin_level]
        nanny_service = params.FETCH_PROD_BINARY_FROM[ctx[params.BinType.name]][bin_level]
        if not build_task_id:
            build_task_id = self._get_prod_build_task_id(binary_res_type, nanny_service, mark)
        logging.info("Got build task for %s %s: %s", binary_res_type, mark, build_task_id)
        return build_task_id

    def _get_prod_build_task_id(self, binary_res_type, nanny_service, mark):
        if self.__prod_build_task_id is None:
            binary_res = production.get_prod_binary_nanny(binary_res_type, nanny_service, just_return_binary=True)
            if not binary_res:
                logging.info("Get {} build task for last_stable_released_resource", mark)
                binary_res = apihelpers.get_last_released_resource(
                    binary_res_type,
                    arch=sandboxapi.ARCH_LINUX,
                )
            self.__prod_build_task_id = binary_res.task_id
        return self.__prod_build_task_id

    def models(self, mark):
        attrs_models = self.attrs["models"].get(self._task_ctx[params.BinType.name])
        if (attrs_models is None or not attrs_models.get("enable", True)) and not self.__need_resource("models", mark):
            return  # models are not used everywhere
        if not self.info["models"][mark]:
            self.info["models"][mark] = self._get_res("models", mark)
        return self.info["models"][mark]

    # only for mmeta:

    def data(self, mark):
        if self.bin_level != "mmeta" or self._task_ctx[params.BinType.name] not in self.attrs["data"]:
            return
        if not self.info["data"][mark]:
            self.info["data"][mark] = self._get_res("data", mark)
        return self.info["data"][mark]

    def index(self, mark):
        if self.bin_level != "mmeta":
            return
        if self._task_ctx[params.BinType.name] not in self.attrs["index"]:
            return  # index is not used everywhere
        if not self.info["index"][mark]:
            if self.attrs["index"][self._task_ctx[params.BinType.name]] is None:
                attr_values = ""
                queries_attrs = channel.sandbox.get_resource(self.queries(mark)).attributes
                if ms.META_SHARD_INSTANCE_ATTRIBUTE_NAME in queries_attrs:
                    attr_values = queries_attrs[ms.META_SHARD_INSTANCE_ATTRIBUTE_NAME]
                elif ms.SNIP_SHARD_INSTANCE_ATTRIBUTE_NAME in queries_attrs:
                    attr_values = queries_attrs[ms.SNIP_SHARD_INSTANCE_ATTRIBUTE_NAME]
                else:
                    eh.check_failed("Queries resource hasn't correct info about index")
                self.attrs["index"][self._task_ctx[params.BinType.name]] = {
                    "attr_name": ms.SHARD_INSTANCE_ATTRIBUTE_NAME,
                    "attr_value": attr_values.split(",")[0],
                }
            self.info["index"][mark] = self._get_res("index", mark)
        return self.info["index"][mark]

    # FIXME: This is a temporary solution for tests
    #        Use self.releases dictionary in addition to self.attrs after test period
    def __need_resource(self, res_name, mark):
        source = self._task_ctx.get(params.SOURCE_TEMPLATE.format(res_name, mark, self.bin_level))
        return source == params.SourceType.LAST_RELEASED or source == params.SourceType.SPECIFIED


def get_testenv_attrs(bin_level, hamster_cfg=True):
    return {
        "config": {
            "web": {
                "attr_name": ms.WebSettings.testenv_middle_cfg_attr_name("jupiter_{}".format(bin_level), hamster_cfg),
                "attr_value": None,
            },
            "quick": {
                "attr_name": ms.QuickSettings.testenv_middle_resources_attr_name(bin_level),
                "attr_value": None,
            },
            "img": {
                "attr_name": ms.ImagesSettings.testenv_resource_attributes(bin_level, ms.INDEX_MAIN, hamster_cfg)[0],
                "attr_value": None,
            },
            "imgquick": {
                "attr_name": ms.ImagesSettings.testenv_resource_attributes(bin_level, ms.INDEX_QUICK)[0],
                "attr_value": None,
            },
            "cbir": {
                "attr_name": ms.ImagesSettings.testenv_resource_attributes(bin_level, ms.INDEX_CBIR_MAIN)[0],
                "attr_value": None,
            },
            "video": {
                "attr_name": ms.VideoSettings.testenv_resource_attributes(bin_level, ms.VLA_REGION)[0],
                "attr_value": None,
            },
            "itditp": {
                "attr_name": ms.WebSettings.testenv_middle_cfg_attr_name("jupiter_{}".format(bin_level), hamster_cfg),
                "attr_value": None,
            },
        },
        "queries": {
            "web": {
                "attr_name": ms.WebSettings.testenv_middle_resources_attr_name(bin_level, "reqs"),
                "attr_value": None,
            },
            "quick": {
                "attr_name": ms.QuickSettings.testenv_middle_resources_attr_name(bin_level),
                "attr_value": None,
            },
            "img": {
                "attr_name": ms.ImagesSettings.testenv_middlesearch_queries_attributes(bin_level, ms.INDEX_MAIN)[0],
                "attr_value": None,
            },
            "imgquick": {
                "attr_name": ms.ImagesSettings.testenv_middlesearch_queries_attributes(bin_level, ms.INDEX_QUICK)[
                    0],
                "attr_value": None,
            },
            "cbir": {
                "attr_name":
                    ms.ImagesSettings.testenv_middlesearch_queries_attributes(bin_level, ms.INDEX_CBIR_MAIN)[0],
                "attr_value": None,
            },
            "video": {
                "attr_name":
                    ms.VideoSettings.testenv_middlesearch_queries_attributes(bin_level, ms.VLA_REGION, None)[0],
                "attr_value": None,
            },
            "itditp": {
                "attr_name": "TE_web_production_itditp_reqs",
                "attr_value": None,
            },
        },
        "plan": {
            "web": {
                "attr_name": ms.WebSettings.testenv_middle_resources_attr_name(bin_level, "plan"),
                "attr_value": None,
            },
            "quick": {
                "attr_name": "test_refresh_freshness",  # TODO: check when task will be ready
                "attr_value": None,
            },
            "img": {
                "attr_name": ms.ImagesSettings.testenv_middlesearch_queries_attributes(bin_level, ms.INDEX_MAIN)[0],
                "attr_value": None,
            },
            "imgquick": {
                "attr_name": ms.ImagesSettings.testenv_middlesearch_queries_attributes(bin_level, ms.INDEX_QUICK)[
                    0],
                "attr_value": None,
            },
            "cbir": {
                "attr_name":
                    ms.ImagesSettings.testenv_middlesearch_queries_attributes(bin_level, ms.INDEX_CBIR_MAIN)[0],
                "attr_value": None,
            },
            "video": {
                "attr_name":
                    ms.VideoSettings.testenv_middlesearch_queries_attributes(bin_level, ms.VLA_REGION, None)[0],
                "attr_value": None,
            },
            "itditp": {
                "attr_name": ms.WebSettings.testenv_middle_resources_attr_name(bin_level, "plan"),
                "attr_value": None,
            },
        },
        "models": {
            "web": {
                "attr_name": "current_production",
                "attr_value": None,
            },
            "img": {
                "enable": bin_level == "int",
                "attr_name": None,
                "attr_value": None,
            },
            "video": {
                "enable": bin_level == "int",
                "attr_name": None,
                "attr_value": None,
            },
            "quick": {
                "attr_name": ms.QuickSettings.testenv_middle_resources_attr_name(bin_level),
                "attr_value": None,
            },
            "itditp": {
                "attr_name": "current_production",
                "attr_value": None,
            },
        },
        "data": {
            "web": {
                "attr_name": ms.WebSettings.testenv_middle_resources_attr_name(bin_level, "data"),
                "attr_value": None,
            },
            "quick": {
                "attr_name": ms.QuickSettings.testenv_middle_resources_attr_name(bin_level),
                "attr_value": None,
            },
            "img": {
                "attr_name": ms.ImagesSettings.testenv_resource_attributes(bin_level, ms.INDEX_MAIN)[0],
                "attr_value": None,
            },
            "imgquick": {
                "attr_name": ms.ImagesSettings.testenv_resource_attributes(bin_level, ms.INDEX_QUICK)[0],
                "attr_value": None,
            },
            "cbir": {
                "attr_name": ms.ImagesSettings.testenv_resource_attributes(bin_level, ms.INDEX_CBIR_MAIN)[0],
                "attr_value": None,
            },
            "video": {
                "attr_name": ms.VideoSettings.testenv_resource_attributes(bin_level, ms.VLA_REGION)[0],
                "attr_value": None,
            },
            "itditp": {
                "attr_name": ms.WebSettings.testenv_middle_resources_attr_name(bin_level, "data"),
                "attr_value": None,
            },
        },
        "index": {
            "img": None,
            "imgquick": None,
            "cbir": None,
            "video": {
                "attr_name": ms.VideoSettings.RESOURCE_PRIEMKA_ATTRIBUTES[0],
                "attr_value": None,
            },
        }
    }


def _patch_queries(res_id, mark, cgis):
    res = channel.sandbox.get_resource(res_id)
    res_path = utils.sync_resource(res_id)
    logging.info("Patching queries taken from resource # %s for %s via %s", res_id, mark, json.dumps(cgis))

    patched_res = channel.task.create_resource(
        "Patched {} queries based on {}".format(mark, res_id),
        "{}_patched_queries_based_on_resource_{}.txt".format(mark, res_id),
        res.type,
    )

    queries = fu.read_line_by_line(res_path)
    for cgi_name, appended_val in cgis.iteritems():
        queries = cgi.append_cgi(cgi_name, appended_val, queries)

    fu.write_lines(patched_res.path, queries)
    utils.set_resource_attributes(patched_res, {k: v for k, v in res.attributes.iteritems() if "shard_instance" in k})

    channel.task.mark_resource_ready(patched_res)
    return patched_res.id
