import logging
import shutil
import time
import os
import collections

import sandbox.projects.websearch.middlesearch.base_tasks.multi_host_task as ms_multi_host
import sandbox.projects.common.search.components.mkl as sc_mkl
import sandbox.projects.common.search.bugbanner2 as bb2

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import file_utils as fu
from sandbox.projects import resource_types
from .helpers import Request, diff_requests


class ParsedEventlogDump(sdk2.Resource):
    """ Parsed eventlog dump data"""


class RequestsDiff(sdk2.Resource):
    """ Requests reasks diff """


class FilteredRequests(sdk2.Resource):
    """ Search requests and reasks """


class WebTestMiddlesearchReasks(ms_multi_host.WebMiddlesearchMultiHost):
    """
       Task for testing middlesearch reasks
    """

    def get_reask_config(self):
        config = {
            'ReAskIncompleteSources': 'yes',
            'Timeout': '3s',
            'MainThreads': '3',
            'GarbageThreads': '3',
            'EnqueueOnSave': 'True',
            'MinReAskDelay': '6s',
            'MaxReAskDelay': '12s',
            'MaxReAskQueueSize': '20000',
            'MaxReAskRps': '1000',
        }

        return config

    def get_connstats_config(self):
        return {
            'FailThreshold': '20',
            'CheckTimeout': '30s',
            'CheckInterval': '0'
        }

    def _get_basesearchers(self):
        # fake searches, that shouldn't answer
        # see MIDDLE-19 for details
        return [
            {
                'basesearch_type': 'PLATINUM',
                'searchers': [('localhost', 65000), ]
            },
            {
                'basesearch_type': 'WEB',
                'searchers': [('localhost', 65001), ]
            },
            {
                'basesearch_type': 'QUICK_SAMOHOD_ON_MIDDLE_EXP',
                'searchers': [('localhost', 65002), ],
                'options': "NoReAsk=False",
            },
            {
                'basesearch_type': 'QUICK_SAMOHOD_ON_MIDDLE',
                'searchers': [('localhost', 65003), ],
                'options': "NoReAsk=False",
            },
            {
                'basesearch_type': 'WEBFRESH_ON_MIDDLE',
                'searchers': [('localhost', 65004), ],
                'options': "NoReAsk=False",

            },

        ]

    def _parsed_log_file(self):
        return sdk2.ResourceData(ParsedEventlogDump(self, "Parsed eventlog dump", "evlog_parsed.txt"))

    def _diff_file(self):
        return sdk2.ResourceData(RequestsDiff(self, " Requests reasks diff ", "requests_diff.txt"))

    def _filtered_requests_file(self):
        return sdk2.ResourceData(FilteredRequests(self, "Search requests and reasks", "requests.txt"))

    def _get_evlogdump_path(self):
        evlogdump_id = sdk2.Resource[self.Parameters.middlesearch.evlogdump_exec]
        evlogdump = sdk2.ResourceData(evlogdump_id)
        return str(evlogdump.path)

    def _config_string(self, config):
        items = ["{}={}".format(key, value) for key, value in config.iteritems()]
        return ', '.join(items)

    def get_reask_options(self):
        return self._config_string(self.get_reask_config())

    def get_connstats_options(self):
        return self._config_string(self.get_connstats_config())

    def patch_for_middlesearch_config(self):
        config = super(WebTestMiddlesearchReasks, self).patch_for_middlesearch_config()
        config['Collection.ReAskOptions'] = self.get_reask_options()
        config['Collection.ConnStatOptions'] = self.get_connstats_options()
        return config

    def wait_for_reasks(self, port):
        # itertools.repeat() doesn't work. Text to @johnnovikov for explanation
        logging.info('Started waiting for reasks')
        query = ['?info=get_active_reask_count', ]
        r_num, req, r_status, r_data = next(self.iter_responses(query, port, n_workers=1))
        while int(r_data) > 0:
            logging.debug("%s %s %s %s", r_num, r_data, req, r_status)
            time.sleep(5)
            r_num, req, r_status, r_data = next(self.iter_responses(query, port, n_workers=1))
        logging.info('All reasks were made')

    def filtered_eventlogs(self, logs_path):
        for line in open(logs_path, 'r'):
            tokens = line.split()
            if tokens[4] == '0':
                # source, request
                yield tokens[3], tokens[7]

    def compare_requests(self, logs_path):
        search = collections.defaultdict(dict)
        reasks = collections.defaultdict(dict)
        requests_file_path = str(self._filtered_requests_file().path)
        for source, request in self.filtered_eventlogs(logs_path):
            fu.append_lines(requests_file_path, (request,))
            request = Request(request)
            if request.is_reask:
                reasks[source][request.reqid] = request
            else:
                search[source][request.reqid] = request

        diffs = {}
        checked_requests = 0
        found_diffs = 0
        for source in search.keys():
            search_requests = search[source]
            reask_requests = reasks[source]
            reasks_num, search_num = len(reask_requests), len(search_requests)
            logging.info("Found %s search requests, %s reasks requests for source %s", search_num, reasks_num, source)

            for reqid, request in search_requests.items():
                checked_requests += 1
                if reqid in reask_requests:
                    has_diff, diff = diff_requests(request.cgi_params, reask_requests[reqid].cgi_params)
                    if has_diff:
                        found_diffs += 1
                        diffs[reqid] = diff
        return diffs, checked_requests, found_diffs

    def on_execute(self):
        self.add_bugbanner(bb2.Banners.WebMiddleSearch)
        wms = self.init_searchers(need_warmup=False)
        sc_mkl.configure_mkl_environment(wms)
        output_responses = self.output_responses()
        with wms:
            logging.info("Launch %s", wms.name)
            self.save_responses(
                wms.apphost_port,
                str(sdk2.ResourceData(output_responses).path),
            )
            self.wait_for_reasks(wms.apphost_port)

        parsed_dump = self._parsed_log_file()
        with open(str(parsed_dump.path), "w") as parsed_output:
            sdk2.helpers.subprocess.Popen(
                [self._get_evlogdump_path(), os.path.abspath(self._get_eventlog_path()), "-i", "SubSourceRequest"],
                stdout=parsed_output,
            ).wait()

        diffs, checked_requests, found_diffs = self.compare_requests(str(parsed_dump.path))

        logging.info('Checked requests: {}   Found diffs: {}'.format(checked_requests, found_diffs))

        if diffs:
            lines = []
            for reqid, (diff, absent, extra) in diffs.items():
                lines.append('{}: \n\tdiff:'.format(reqid))
                for key, (l, r) in diff.items():
                    lines.append('\t\t{} : {} -> {}'.format(key, l, r))
                lines.append('\tabsent:')
                for key, val in absent.items():
                    lines.append('\t\t{} : {}'.format(key, val))
                lines.append('\textra:')
                for key, val in extra.items():
                    lines.append('\t\t{} : {}'.format(key, val))
                lines.append('-------------------------------------')
            fu.write_lines(str(self._diff_file().path), lines)
            raise TaskFailure("Found difference in reasks and search requests. See log for details")

        if self.Parameters.middlesearch.save_eventlog:
            evlog = resource_types.EVENTLOG_DUMP(
                self, "Middlesearch eventlog, {}".format(self.Parameters.description.encode("utf-8")),
                "middlesearch_event.log"
            )
            shutil.move(self._get_eventlog_path(), str(evlog.path))

    def output_responses(self):
        if self.Parameters.make_binary:
            return resource_types.SEARCH_PROTO_RESPONSES(
                self, "Middlesearch responses, {}".format(self.Parameters.description.encode("utf-8")),
                "middlesearch_responses.bin"
            )
        else:
            return resource_types.BASESEARCH_HR_RESPONSES(
                self, "Middlesearch responses, {}".format(self.Parameters.description.encode("utf-8")),
                "middlesearch_responses.txt"
            )
