import logging
import os
import time
import urllib2

from sandbox.common.fs import make_folder
from sandbox import sdk2

from sandbox.projects.common import config_patcher_tool as cfg_patcher
from sandbox.projects.common import decorators
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search.config import _replace_basesearches
from sandbox.projects.common.search.components import component as components_common


class WebMiddlesearch(
    components_common.WaitPortComponentMixin,
    components_common.ProcessComponentMixinWithShutdownSDK2,
    components_common.Component,
):
    name = "web_middlesearch"

    LOGGER = logging.getLogger("components.WebMiddlesearch")
    _LOG_TMPL = name + "_{port}_{log_type}.log"
    _QUERY_CACHE_TMPL = name + "_{}_query_cache"
    _CONFIG_NAME_TMPL = name + "_{}_patched.cfg"

    def __init__(
        self, task, binary, config, models, rearr,
        basesearches=None,
        port=sc.DEFAULT_MIDDLESEARCH_PORT,
        start_timeout=sc.DEFAULT_START_TIMEOUT,
        cfg_patch_dict=None,
        disable_timeouts=True,
        patch_queues_size=True,
        query_cache_dir=None,
        query_cache_memory_limit=268435456,  # 256 Mb
        eventlog_path=None,
        patched_config_resource=None,
        need_warmup=True
    ):
        self.LOGGER.info("Starting init WebMiddlesearch")
        self.task = task
        self.binary_data = sdk2.ResourceData(self._to_resource_obj(binary))
        self.config_data = sdk2.ResourceData(self._to_resource_obj(config))
        self.models_data = sdk2.ResourceData(self._to_resource_obj(models))
        self.rearr_data = sdk2.ResourceData(self._to_resource_obj(rearr))
        self.port = port
        self.apphost_port = port + 1
        self.grpc_port = port + 2
        self.basesearches = basesearches
        self.start_timeout = start_timeout
        self.cfg_patch_dict = cfg_patch_dict or {}
        self.disable_timeouts = disable_timeouts
        self.patch_queues_size = patch_queues_size
        self.eventlog_path = eventlog_path or self.path_to_log("event")
        self._init_query_cache(query_cache_dir)
        self.query_cache_memory_limit = query_cache_memory_limit
        self.patched_config_resource = None
        self.need_warmup = need_warmup

        if patched_config_resource:
            self.patched_config_resource = sdk2.ResourceData(patched_config_resource)

        self.patch_config(task)

        components_common.WaitPortComponentMixin.__init__(
            self, endpoints=[("localhost", port)], wait_timeout=start_timeout
        )
        components_common.ProcessComponentMixinWithShutdownSDK2.__init__(
            self,
            args=[str(self.binary_data.path), "-d", "-p", str(self.port), self.cfg_dest_path],
            shutdown_url="http://localhost:{}/admin?action=shutdown".format(self.port),
            log_prefix=self.name,
        )
        self.LOGGER.info("Component '%s' initialized successfully", self.name)

    @staticmethod
    def _to_resource_obj(resource_obj_or_id):
        if isinstance(resource_obj_or_id, sdk2.Resource):
            return resource_obj_or_id
        return sdk2.Resource[resource_obj_or_id]

    @decorators.retries(10, delay=10, backoff=1)
    def warmup(self):
        if not self.need_warmup:
            self.LOGGER.info("Warming up disabled")
            return
        # todo: warmup through apphost port
        self.LOGGER.info("Warming up")
        timeout_sec = 80  # 80 seconds
        timeout_usec = timeout_sec * 1000000
        url = (
            "http://127.0.0.1:{}/yandsearch?"
            "&text=1%3A%3A798+%26/%28-3+3%29+2%3A%3A798+%26/%28-3+3%29+3%3A%3A798+%26/%28-3+3%29"
                "+4%3A%3A798+%26/%28-3+3%29+5%3A%3A798+%26/%28-3+3%29+6%3A%3A798+%26/%28-3+3%29"
                "+%287%3A%3A798+^+%D1%81%D0%B5%D0%BC%D0%B5%D1%80%D0%BA%D0%B0%3A%3A53621806%29+%26/%28-3+3%29"
                "+8%3A%3A798+%26/%28-3+3%29+9%3A%3A798+%26/%28-3+3%29+0%3A%3A798+softness%3A6"
            "&user_request=1+2+3+4+5+6+7+8+9+0"
            "&ms=proto"
            "&qtree=cHicvdS9i9RAFADw9ybZ3DgXlpDj4Ei1xGa0mk32I2slYnGIynGVXCWHwtpeJQvKHiKIgmh7p"
            "WLpIcct6LF-IFeI1WxvbeVfYOVkslmGEELSmCZv3mTeb3iBx24wl7Y8Zws6yIkAHwII4TJEcMWl4IHK"
            "AwcBV1vbrR24A3edMb5COEJ4g3CC8BlBPT8QJN4LHrGbLDtF1am0GnYD7Kzqoa6Huh5sQ1pvPHfzctg"
            "t1BOQ4LWLlHgQYDdcF6uny88tgfo6MMYJTBFUkeAxu1Xko5THTj0_qvAj04-4tOv5cRM_rvBj04_575"
            "p-r4nfq_B7pt_jp049v9_E71f4fdPv8z81_UETf1DhD0x_wM8v1POHTfxhhT80_SE_YvX8pImfVPiJ6"
            "Sf8W01_1MQfVfgj0x_xuVvil4wfoaZZXV5U8MLkBf9bMn7CbI6SjsVb4r-v0svufSf0cDrFTYDJ9eA9"
            "YQ8K7WgvDuVMzuVsMZVn8kOAl0Lky-asGbM-b84v-TrO21M4W9arGaYtDgpfhsBRdestydp18AJZ4Ys"
            "trSawaz37-nOPvDved1QZ4Bv7VL2JnHNrGX3UEQnI4mkeyZmOcLnLVGTLL_JTnl28XJ15oveJ3j_j1h"
            "ge4oRQXP4-m7qey221eo4bzFaNI_46pbtrFH3r9s79LGt7TkmWem5Jtu35hayr6zq-2juAbEm9trn01"
            "Bm9VFf6B6vLQQM%2C"
            "&relev=wizqbundle%3DehRMWi40AQAAAACAcwQB8jIKUypGehRMWi40AQAAAACANAABdRI0Eh0SCzAB"
            "APAZMRgAIAAqBQoBMRAAMgMQxAMgACoBMTIAOAJAAEoDEIIEUhUA7wAAADC_xM_DvOzf86EBVQAMFDJV"
            "ABEyVQAg0ARVABUyVQAykgVSFQAAVQCf4MTcvcuzpOq3VQANFDNVABEzVQAR5lUAFTNVABCsVQAAFQAA"
            "VQCfosP07cCjxfaeVQANFDRVABE0VQAgtAaqABU0VQAyqgdSFQAAVQDPnMiB5_OfzMehAQpSVAEKFDVV"
            "ABE1VQAR7FUAFTVVADKiCFIVAABVAJ_7kPiKx57puEZUAAwUNlQAETZUACDECakAFTZUACCEDKgBEglU"
            "AI-L8I261e_mz1QADRQ3VAARN1QAIJoKVAAVN1QAEPpUAAAVAACoAJ_A6f_TuJOm5AqoAAwUOFQAEThU"
            "ABHAVAAVOFQAMq4NUhUAAFQAn-Ke4v-mirC9TU8CDBQ5VAAROVQAILwMqAAVOVQAMoQQUhUAAFQAkIXc"
            "1P7ehqjM8aUBGEX5AkYzAAF2-QIUGFQAETBUABH6-AIVMFQAEMROAgAVAABUAPgBz86spOiK69LMAQqq"
            "ASqcAVYA8H2KAAH7DhKZAhLuARIO0YHQtdC80LXRgNC60LAYACABKhIKFgBqEAEqFAoQKgCM0LwQACoW"
            "ChIWAE7QuBAALgBN0YUQAFgAH7UUAAAPPgABP77QuVQAAD6-0Y5AAC3RgxQA-wC-0LoQADIEEIakXSCj"
            "AirbAPAIMgIIATgBQABKBRCGysQCUgQQhqRdIOADoDDj8KOV7eT567cBAQ_6AxQfAfoDDZ_Wzs6M6Mrf"
            "_lz-AQwH-QMAVAAP-QMKn5uR0MDmh7vYDv4BDAf4AwBUAADjAw_4AwavjaTK_t-dpuLjAVUADAf4AwBV"
            "AA_4Awqfg8DsuuDioeXLVQANB_gDAFUAD_gDCp_krum3ns7I7NZVAA0H-QMAVQAP-QMKn4PMwOThvujo"
            "jlUADQf6AwBVAADlAw_6AwafzsKxoZ_vu8-fUQINB_sDAFUAD_sDCp-vsO2B8ovGgFaoAQwH-wMAVAAP"
            "-wMKkMPGoouY0O_Q7qkAD_sDEwBUAADmAw_7AwbwB5qpyOertNb56gESpQIIChAAGg8KBSqJAFASAggA"
            "GgQABxEAIAEaBAAHEQAgAhoEAAcRACADGgQABxEAIAQaBAAHEQAgBRoEAAcRACAGGgQABxEAIAcaBAAH"
            "EQAgCBoEAAcRACAJGgQAsCEKFwgBEIGAgAEhGgTwAABAj0AqBRDc0JEzEgIIClYAoC0AAIA_MgMaATEF"
            "ABAyBQAQMwUAEDQFABA1BQAQNgUAEDcFABA4BQAQOQUA8RswQhsIAQgCCAQICAgQCCAIQAiAAQiAAgiA"
            "BBACGAoSbQgKEDQaCBICCAsgAQAKABEMGQEACgARDRIBAAoAEQ4LAQAKABEPBAEACgAREP0AAAoAEBGg"
            "AAFGABES7wAAFAARE-gAAAoAoBQaAggJLQAAgD8AAAA%2C"
            "&hr=da"
            "&nocache_completely=da"
            "&pron=tiermask3"
            "&rearr=qlang%3Dnon_words"
            "&rearr=all_off"  # SEARCH-1142
            "&timeout={}"  # SEARCH-1186
            "&waitall=da"
        ).format(self.port, timeout_usec)
        start_time = time.time()
        urllib2.urlopen(url, timeout=timeout_sec).read()
        answer_time = round(time.time() - start_time)
        eh.ensure(answer_time < timeout_sec / 10, "Answer time ({} sec) is not short enough".format(answer_time))

    def patch_config(self, task):
        cfg_patch_path = self.prepare_cfg_patch()
        init_cfg_path = str(self.config_data.path)
        if self.basesearches:
            # todo: use c++ config patcher for this
            tmp_cfg_path = "tmp_cfg_path"
            patched_cfg = _replace_basesearches(fu.read_file(init_cfg_path), self.basesearches)
            fu.write_file(tmp_cfg_path, patched_cfg)
        else:
            tmp_cfg_path = init_cfg_path
        patched_cfg = cfg_patcher.patch_cfg(task, tmp_cfg_path, cfg_patch_path, self.cfg_dest_path, self.name)
        # patched_cfg = cfgproc.patch_config(fu.read_file(str(self.config_data.path)), self.cfg_patch_dict.iteritems())

        if self.patched_config_resource:
            fu.write_file(str(self.patched_config_resource.path), patched_cfg)
        self.LOGGER.debug("Patched cfg:\n%s", patched_cfg)

    def _init_query_cache(self, query_cache_dir):
        if not query_cache_dir:
            query_cache_dir = self._QUERY_CACHE_TMPL.format(self.port)
        make_folder(query_cache_dir)
        self.__query_cache_dir = query_cache_dir

    def query_cache_dir(self, n):
        return os.path.abspath(os.path.join(self.__query_cache_dir, "tmpdir.{}".format(n)))

    @property
    def cfg_dest_path(self):
        return os.path.abspath(self._CONFIG_NAME_TMPL.format(self.port))

    def path_to_log(self, log_type):
        return os.path.abspath(self._LOG_TMPL.format(port=self.port, log_type=log_type))

    def prepare_cfg_patch(self):
        rearr_data_path = str(self.rearr_data.path)
        pure_dir = os.path.join(rearr_data_path, "pure")
        rearrange_dir = os.path.join(rearr_data_path, "rearrange")
        self.cfg_patch_dict.update({
            "Server.Port": str(self.port),
            "Server.AppHostOptions": (
                "Port=+1, "
                "Threads=60, "
                "GrpcPort=+2, "
                "GrpcThreads=60, "
                "AllowBidirectionalStreaming=true"
            ),
            'Server.LoadLog': self.path_to_log("load"),
            'Server.ServerLog': self.path_to_log("server"),
            'Server.EventLog': self.eventlog_path,
            'Server.LogExceptions': 'True',
            'Collection.UseRequiredBaseTimestamp': 'False',
            'Collection.IndexDir': pure_dir,
            'Collection.RearrangeDataDir': rearrange_dir,
            "Collection.MaxAllfactorsPerBasesearch": 50,
            "Collection.QueryCache[:].MemoryLimit": self.query_cache_memory_limit,
            'Collection.QueryCache[0].Dir': self.query_cache_dir(0),
            'Collection.MXNetFile': str(self.models_data.path),
            'Collection.CacheThreadLimit': 32,  # For SEARCH-1474
            'Collection.RequestTimeoutLimit': "40s",  # SEARCH-5816
            'Collection.AbortTout': '600s',  # prod/hamster 60s is sometimes too low for sanitizers
            'Server.ErrorBoosterUnifiedAgentLog.Uri': '',
        })
        if self.basesearches:
            self.cfg_patch_dict.update({
                'Collection.QueryCache[1:]': "__remove__",  # no need for additional caches in single host mode
                'DNSCache': "__remove__",
            })
        else:
            self.cfg_patch_dict.update({
                'Collection.QueryCache[1].Dir': self.query_cache_dir(1),
                'Collection.QueryCache[2].Dir': self.query_cache_dir(2),
            })

        if self.disable_timeouts:
            self.cfg_patch_dict.update({
                'Collection.ScatterOptions[:].TimeoutTable': '300s',
                'Collection.ScatterOptions[:].MessengerOptions': "__remove__",
                'Collection.ScatterOptions[:].RemoteTimeoutOptions': "__remove__",
                'Collection.ConnectTimeout': '300s',
                'Collection.MessengerOptions': "__remove__",
                'Collection.UseTimeoutTable': "__remove__",
            })
        if self.patch_queues_size:
            self.cfg_patch_dict.update({
                'Server.Threads': 100,
                'Server.QueueSize': 100,
                'Collection.RequestThreads': 100,
                'Collection.RequestQueueSize': 100
            })

        patch_path = os.path.abspath("cfg_patch.json")
        fu.json_dump(patch_path, self.cfg_patch_dict, indent=1)
        return patch_path
