import base64
import json
import logging
import sys

import sandbox.common.types.resource as ctr

import sandbox.common.types.client as ctc
import sandbox.sdk2.environments as sdkenv
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia

import sandbox.projects.common.app_host.converter as apphost_converter
import sandbox.projects.websearch.middlesearch.resources as ms_res

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common.search import requester_core
from sandbox.projects.common.search import requester_mixin


class WebMiddlesearchRequester(requester_mixin.SearchRequesterTask):
    """
       Base task for middlesearch task, working with requester
    """
    class Requirements(sdk2.Task.Requirements):
        ram = 60 * 1024  # 60 Gb
        disk_space = 75 * 1024  # 75 Gb
        client_tags = ctc.Tag.GENERIC
        environments = (sdkenv.PipEnvironment('protobuf', '2.6.1', use_wheel=True),)

    class Parameters(requester_mixin.SearchRequesterTask.Parameters):
        with requester_mixin.SearchRequesterTask.Parameters.requester_options() as requester_options:
            # SEARCH-5052
            always_use_apphost_port = sdk2.parameters.Bool("Use apphost port on any type of queries", default=False)

        with requester_mixin.SearchRequesterTask.Parameters.query_transform_options() as query_transform_options:
            verbose_transform = sdk2.parameters.Bool("Log queries before and after transformation", default=False)
            apphost_converter = sdk2.parameters.Resource(
                "Apphost converter tool",
                resource_type=resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE,
                required=False,
            )

    @property
    def is_apphost_requests(self):
        return isinstance(self.Parameters.requests, ms_res.WebMiddlesearchApphostRequests)

    def get_target_port(self, wms):
        """Sometimes port doesn't depend on queries type (SEARCH-5052)"""
        return wms.apphost_port if self.is_apphost_requests or self.Parameters.always_use_apphost_port else wms.port

    def query_transformer(self, base_url):
        if self.is_apphost_requests:
            return requester_core.ApphostQueryTransformer(
                base_url, self.apphost_converter_path,
                stabilize=self.Parameters.stabilize,
                need_dbgrlv=self.Parameters.need_dbgrlv,
                log_src_groupings=self.Parameters.log_src_groupings,
                get_all_factors=self.Parameters.get_all_factors,
                use_dcfm=self.Parameters.use_dcfm,
                verbose_transform=self.Parameters.verbose_transform,
            )
        return requester_mixin.SearchRequesterTask.query_transformer(self, base_url)

    def prepare_to_save(self):
        # self._get_correct_protobuf_schemes()
        if self.is_apphost_requests:
            if self.Parameters.apphost_converter:
                apphost_converter_resource = self.Parameters.apphost_converter
            else:
                apphost_converter_resource = sdk2.Resource.find(
                    resource_type=resource_types.APP_HOST_TOOL_CONVERTER_EXECUTABLE,
                    state=ctr.State.READY,
                ).first()
            logging.info("Use apphost converter resource: %s", apphost_converter_resource.id)
            self.apphost_converter_path = str(sdk2.ResourceData(apphost_converter_resource).path)
            # import app_host_pb2
            # self.apphost_proto_obj = app_host_pb2.TResponse()

    def _get_correct_protobuf_schemes(self):
        pack_path = str(self.path("meta_pb_pack"))
        Arcadia.export("arcadia:/arc/trunk/arcadia/search/idl/generated", pack_path)
        logging.debug("Pack path = %s", pack_path)
        # SEARCH-7148
        # copypasted this code from projects/RefreshTestingConfigGenerate/__init__.py
        # hope, that can help to use correct protobuf package
        # map(sys.modules.pop, [n for n in sys.modules if n.startswith("google.protobuf")])
        # google = sys.modules["google"] = types.ModuleType("google")
        # google.__dict__["__path__"] = [os.path.join(site.USER_SITE, "google")]
        sys.path.append(pack_path)

    def post_process_response(self, r_num, req, r_status, r_data):
        if self.is_apphost_requests:
            # logging.debug("r_data:%s", str(r_data))
            json_resp = apphost_converter.convert_input_response(self.apphost_converter_path, r_data)
            # logging.debug("json_resp:%s", json.dumps(json_resp, indent=2))
            try:
                for ans in reversed(json_resp["answers"]):
                    if not ans.get("type", "").startswith("pre_") and "binary" in ans:
                        return base64.decodestring(ans["binary"])
            except Exception:
                pass

            eh.check_failed("Unable to decode apphost response:\n{}\n{}\nRaw response:\n{}\nRequest ({}):\n{}".format(
                json.dumps(json_resp, indent=2), eh.shifted_traceback(), r_data, r_num, req
            ))
        return r_data

    def parse_request_for_diff(self, req):
        if self.is_apphost_requests:
            try:
                return apphost_converter.convert_input_request(self.apphost_converter_path, req[1])
            except Exception:
                pass
        return req

    def _get_eventlog_path(self):
        if self.Parameters.middlesearch.save_eventlog:
            evlog = resource_types.EVENTLOG_DUMP(
                self, "Middlesearch eventlog, {}".format(self.Parameters.description.encode("utf-8")),
                "middlesearch_event.log"
            )
            return str(evlog.path)

    def get_app_host_pb2(self):
        try:
            from search.idl import app_host_pb2
            logging.info("app_host_pb2 was loaded from binary directly. Success!")
            return app_host_pb2
        except ImportError:
            logging.info("Unable to use app_host_pb2 from ya.make. Will try to use downloaded one.")
        self._get_correct_protobuf_schemes()
        import app_host_pb2
        return app_host_pb2
