import os
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects import resource_types
from sandbox.projects.websearch.models_proxy import resources
from sandbox.projects.common import config_processor
from sandbox.projects.common.search import components, config


def create_models_proxy_params(n=None):
    n_in_name = "_" if n is None else "_{}_".format(n)
    n_in_text = " " if n is None else " {} ".format(n)
    group_name = 'models_proxy{}parameters'.format(n_in_text)

    class Parameters(object):
        class Binary(sp.ResourceSelector):
            name = 'models_proxy{}executable_id'.format(n_in_name)
            description = 'models_proxy{}executable'.format(n_in_text)
            group = group_name
            resource_type = resources.ModelsProxyExecutable

        class Config(sp.ResourceSelector):
            name = 'models_proxy{}mmetaconfig_resource_id'.format(n_in_name)
            description = 'models_proxy{}mmeta config'.format(n_in_text)
            group = group_name
            resource_type = resource_types.MIDDLESEARCH_CONFIG

        class Data(sp.ResourceSelector):
            name = 'models_proxy{}data_resource_id'.format(n_in_name)
            description = 'models_proxy{}other configs'.format(n_in_text)
            group = group_name
            resource_type = resources.ModelsProxyData

        class Evlogdump(sp.ResourceSelector):
            name = 'models_proxy{}evlogdump_resource_id'.format(n_in_name)
            description = 'Evlogdump'
            resource_type = resource_types.EVLOGDUMP_EXECUTABLE
            group = group_name
            required = False

        class Requests(sp.ResourceSelector):
            name = 'models_proxy{}requests_resource_id'.format(n_in_name)
            description = 'models_proxy{}requests'.format(n_in_text)
            group = group_name
            resource_type = resources.ModelsProxyRequests

        class SubSourcesMockerBinary(sp.ResourceSelector):
            name = 'models_proxy{}subsrc_mocker_resource_id'.format(n_in_name)
            description = 'models_proxy{}subsources mocker executable'.format(n_in_text)
            group = group_name
            resource_type = resources.ModelsProxySubsourceMockerExecutable

        class MockedFetchDocData(sp.ResourceSelector):
            name = 'models_proxy{}fetch_doc_data_resource_id'.format(n_in_name)
            description = 'models_proxy{}FetchDocData results'.format(n_in_text)
            group = group_name
            resource_type = resources.ModelsProxyFetchDocDataResults

        class MockedDynTableResults(sp.ResourceSelector):
            name = 'models_proxy{}dyntable_results_resource_id'.format(n_in_name)
            description = 'models_proxy{}DYNTABLE_HTTP_PROXY results'.format(n_in_text)
            group = group_name
            resource_type = resources.ModelsProxySubsourceResults

        class RequestPatches(sp.ListRepeater, sp.SandboxStringParameter):
            name = 'models_proxy{}request_patches'.format(n_in_name)
            description = 'Name=value strings to add to CustomParams for all models_proxy{}requests'.format(n_in_text)
            group = group_name
            default_value = []

        params = (
            Binary,
            Config,
            Data,
            Evlogdump,
            Requests,
            SubSourcesMockerBinary,
            MockedFetchDocData,
            MockedDynTableResults,
            RequestPatches
        )

    return Parameters


class NoConfig(config.SearchConfig):
    @classmethod
    def get_production_config(cls):
        return cls._get_config_from_text('No config')

    def save_to_file(self, file_path):
        pass


class ModelsProxyMocker(components.StandardSearch):
    name = 'models_proxy_subsources_mocker'

    def __init__(self, binary, port, fetchdocdata, dyntabledata):
        super(ModelsProxyMocker, self).__init__(None, binary, port, NoConfig, None, {})
        self.fetchdocdata = fetchdocdata
        self.dyntabledata = dyntabledata

    def _get_run_cmd(self, config_path):
        return ['-p', str(self.port), '--fetched-doc-data', self.fetchdocdata, '--aux-sources-data', self.dyntabledata]


class ModelsProxy(components.StandardSearch):
    name = 'models_proxy'
    http_collection = 'models_proxy'

    def __init__(self, work_dir, binary, port, config_file, data_dir, event_log, mocker):
        super(ModelsProxy, self).__init__(work_dir, binary, port, config.MidlesearchConfig, config_file, {})
        self.data_dir = data_dir
        self.mocker = mocker
        self.patched_data_dir = self.work_dir + '/patched_data'
        try:
            os.mkdir(self.patched_data_dir)
            os.mkdir(self.patched_data_dir + '/ModelsProxy')
            os.mkdir(self.patched_data_dir + '/FetchDocData')
        except OSError:
            pass
        mpcfg = '/ModelsProxy/models_proxy.cfg'
        fddcfg = '/FetchDocData/fetch_doc_data.cfg'
        with open(self.data_dir + mpcfg, "r") as f, open(self.patched_data_dir + mpcfg, "w") as g:
            for line in f:
                if not line.strip().startswith('Eventlog:'):
                    if not line.endswith('\n'):  # last line
                        line += '\n'
                    g.write(line)
            g.write("Eventlog: \"" + event_log + "\"")
        with open(self.data_dir + fddcfg, "r") as f, open(self.patched_data_dir + fddcfg, "w") as g:
            g.write(f.read())

        class ReplaceSearchSources(object):
            def __init__(self, prefix):
                self.prefix = prefix
                self.current_id = 0

            def __call__(self, oldvalue):
                result = '{}/{}'.format(self.prefix, self.current_id)
                result = result + ' ' + result + ' ' + result
                self.current_id += 1
                return result

        fddprefix = 'http://localhost:{}/FETCH_DOC_DATA'.format(self.mocker.port)
        self.config.text = config_processor.parameter_map(ReplaceSearchSources(fddprefix), self.config.text, 'Collection/SearchSource/CgiSearchPrefix')

        def replace_dyntable_source(value):
            if 'DyntableHttpProxy' in value:
                result = 'http://localhost:{}/DYNTABLE_HTTP_PROXY_SOURCE'.format(self.mocker.port)
                result = result + ' ' + result + ' ' + result
                return result
            return value

        self.config.text = config_processor.parameter_map(replace_dyntable_source, self.config.text, 'Collection/AuxSource/CgiSearchPrefix')

        self.config.save_to_file(self.patched_data_dir + '/FetchDocData/main.cfg')

    def start(self):
        super(ModelsProxy, self).start()
        self.mocker.start()

    def stop(self):
        super(ModelsProxy, self).stop()
        self.mocker.stop()

    def wait(self, timeout=None):
        super(ModelsProxy, self).wait(timeout)
        self.mocker.wait(timeout)

    # maybe we should implement models_proxy -v for consistency with other components,
    # to make this override go away
    def _get_info(self):
        return None

    def _get_run_cmd(self, config_path):
        return ["--data-dir", self.patched_data_dir, "-p", str(self.port), self.data_dir + "/server.cfg"]
