import datetime as dt
import logging
import os

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.websearch.models_services.common import get_service_type, ServiceType
from sandbox.projects.websearch.models_services.resources import resources as models_services_resources
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess


def get_logged_requests_table(logs_prefix, service):
    t = dt.datetime.now() - dt.timedelta(days=1)
    return "/".join([logs_prefix, service, "body-requests-log", t.strftime("%Y%m%d")])


class GetModelsServiceRequestsPlan(binary_task.LastBinaryTaskRelease, sdk2.Task):
    __logger = logging.getLogger("TASK_LOGGER")
    __logger.setLevel(logging.DEBUG)

    class Parameters(sdk2.Task.Parameters):
        logs_prefix = sdk2.parameters.String("Logs prefix in yt", default="//home/eventlogdata", required=True)
        service = sdk2.parameters.String("Service name in nanny", required=True)
        requests_count = sdk2.parameters.Integer("Count of requests for constructing plan", default=10000, required=True)
        dplanner = sdk2.parameters.Resource(
            "Dolbilka planner executable",
            required=False,
            resource_type=dolbilka_resources.DPLANNER_EXECUTABLE,
        )
        yt_token = sdk2.parameters.YavSecret("Yt token secret identifier", required=True)
        yt_proxy = sdk2.parameters.String("Yt proxy", default="hahn", required=True)
        yt_pool = sdk2.parameters.String("Yt pool", default="neural-net", required=False)
        do_not_validate = sdk2.parameters.Bool(
            "Do not validate output size",
            default=False,
        )
        tasks_archive_resource = binary_task.binary_release_parameters(stable=True)

    class Requirements(sdk2.Requirements):
        disk_space = 25 * 1024
        ram = 4 * 1024
        environments = [PipEnvironment("yandex-yt", version="0.10.8")]

    def on_enqueue(self):
        if self.Parameters.dplanner is None:
            self.Parameters.dplanner = sdk2.Resource.find(
                type=dolbilka_resources.DPLANNER_EXECUTABLE,
                attrs=dict(released="stable"),
                state="READY"
            ).first()

    def on_execute(self):
        import yt.wrapper as yt
        env = os.environ.copy()
        env["YT_PROXY"] = self.Parameters.yt_proxy
        env["YT_POOL"] = self.Parameters.yt_pool
        env["YT_TOKEN"] = self.Parameters.yt_token.data()[self.Parameters.yt_token.default_key]
        yt.config["token"] = env.get("YT_TOKEN")
        yt.config["proxy"]["url"] = env.get("YT_PROXY")

        table_path = get_logged_requests_table(self.Parameters.logs_prefix, self.Parameters.service)
        self.__logger.info("Calculated path to table with logged requests = %s" % table_path)

        requests_file = "{}_{}_requests.txt".format(self.Parameters.service, self.Parameters.requests_count)
        plan_file = "{}_{}_requests.plan".format(self.Parameters.service, self.Parameters.requests_count)
        with open(requests_file, "w") as out:
            if get_service_type(self.Parameters.service) == ServiceType.RTMODELS:
                ruchka = "/rtmodels?"
            else:
                ruchka = "/cfg_models?"
            current_requests_count = 0
            for row in yt.read_table(table_path):
                if row["event_type"] == "TRequestReceived":
                    out.write("\t".join(["10000", ruchka, "", "POST", row["event_data"].strip()]) + "\n")
                    current_requests_count += 1
                if current_requests_count >= self.Parameters.requests_count:
                    break

        self.__logger.info("Collected %s requests from table, constructing plan" % current_requests_count)

        dplanner_executable = str(sdk2.ResourceData(self.Parameters.dplanner).path)
        command = [
            dplanner_executable,
            "-l", requests_file,
            "-o", plan_file,
            "-t", "plain",
            "-h", "localhost"
        ]
        plan = models_services_resources.MODELS_SERVICE_REQUESTS_PLAN(
            self,
            "Requests plan for %s" % self.Parameters.service,
            plan_file
        )
        with sdk2.helpers.ProcessLog(self, logger="dplanner") as dplanner_pl:
            subprocess.check_call(command, stdout=dplanner_pl.stdout, stderr=dplanner_pl.stderr)

        set_meta_info(plan, self.Parameters.service)
        sdk2.ResourceData(plan).ready()


def set_meta_info(plan_resource, service):
    plan_resource.service = service
    plan_resource.service_type = get_service_type(service)
