import logging
import os
import random

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.client as ctc
from sandbox.common.utils import Enum
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common.dolbilka import resources as dolbilka_resources
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.common.network import get_free_port
from sandbox.projects.common.search.components import get_begemot
from sandbox.projects.websearch.begemot import resources as begemot_resources
from sandbox.projects.websearch.cfg_models import resources as cfgmodels_resources
from sandbox.projects.websearch.models_services.common import ServiceType
from sandbox.projects.websearch.models_services.resources import resources as models_services_resources
from sandbox.projects.websearch.rtmodels import resources as rtmodels_resources
from sandbox.sdk2.helpers import subprocess


class RunMode(Enum):
    LOCAL = "local"
    REMOTE = "remote"


class GetModelsServiceResponses(binary_task.LastBinaryTaskRelease, sdk2.Task):
    __logger = logging.getLogger("TASK_LOGGER")
    __logger.setLevel(logging.DEBUG)

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Service type") as service_type:
            service_type.values[ServiceType.RTMODELS] = service_type.Value(value=ServiceType.RTMODELS, default=True)
            service_type.values[ServiceType.CFG_MODELS] = service_type.Value(value=ServiceType.CFG_MODELS)
        dexecutor = sdk2.parameters.Resource(
            "Dolbilka executor executable",
            required=False,
            resource_type=dolbilka_resources.DEXECUTOR_EXECUTABLE,
        )
        rps = sdk2.parameters.Integer("RPS to shoot", required=True, default=100)
        requests_plan = sdk2.parameters.Resource(
            "Requests plan",
            required=False,
            resource_type=models_services_resources.MODELS_SERVICE_REQUESTS_PLAN,
        )
        with sdk2.parameters.String("Run mode") as run_mode:
            run_mode.values[RunMode.LOCAL] = run_mode.Value(value=RunMode.LOCAL, default=True)
            run_mode.values[RunMode.REMOTE] = run_mode.Value(value=RunMode.REMOTE)
        with run_mode.value[RunMode.LOCAL]:
            with service_type.value[ServiceType.RTMODELS]:
                rtmodels_binary = sdk2.parameters.Resource(
                    "RtModels executable",
                    required=True,
                    resource_type=rtmodels_resources.RtmodelsExecutable,
                )
                rtmodels_data = sdk2.parameters.Resource(
                    "RtModels data",
                    required=True,
                    resource_type=rtmodels_resources.RtmodelsData,
                )
                rtmodels_begemot_config = sdk2.parameters.Resource(
                    "RtModels config",
                    required=True,
                    resource_type=rtmodels_resources.RtmodelsConfig,
                )
            with service_type.value[ServiceType.CFG_MODELS]:
                cfg_models_binary = sdk2.parameters.Resource(
                    "CfgModels executable",
                    required=True,
                    resource_type=cfgmodels_resources.CfgModelsExecutable,
                )
                cfg_models_archive = sdk2.parameters.Resource(
                    "CfgModels archive",
                    required=True,
                    resource_type=cfgmodels_resources.CfgModelsArchive,
                )
                cfg_models_rule_config = sdk2.parameters.Resource(
                    "CfgModels rule config",
                    required=True,
                    resource_type=cfgmodels_resources.CfgModelsBegemotConfig,
                )
                cfg_models_begemot_config = sdk2.parameters.Resource(
                    "CfgModels begemot config",
                    required=True,
                    resource_type=cfgmodels_resources.CfgModelsBegemotConfig,
                )
        with run_mode.value[RunMode.REMOTE]:
            host = sdk2.parameters.String("Host to shoot", required=False)
            port = sdk2.parameters.Integer("Port to shoot", required=True)
            nanny_service = sdk2.parameters.String("Select host from Nanny")
            nanny_snapshot = sdk2.parameters.String("Verify Nanny snapshot")
            nanny_oauth_token = sdk2.parameters.YavSecret("OAuth token for Nanny")
        need_dump_responses = sdk2.parameters.Bool("Dump responses to separate file")
        with need_dump_responses.value[False]:
            shoot_time = sdk2.parameters.Integer("Time to shoot for measuring performance in seconds", required=True, default=600)
        queries_limit = sdk2.parameters.Integer("Limit number of requests")
        with sdk2.parameters.RadioGroup("Schedule to sandbox host in this DC") as restrict_dc:
            restrict_dc.values["any"] = restrict_dc.Value("any", default=True)
            for dc in ('SAS', 'MAN', 'VLA'):
                restrict_dc.values[dc] = restrict_dc.Value(dc)
        tasks_archive_resource = binary_task.binary_release_parameters(stable=True)

    class Requirements(sdk2.Requirements):
        disk_space = 40 * 1024
        ram = 40 * 1024

    def on_enqueue(self):
        if self.Parameters.dexecutor is None:
            self.Parameters.dexecutor = sdk2.Resource.find(
                type=dolbilka_resources.DEXECUTOR_EXECUTABLE,
                attrs=dict(released="stable"),
                state="READY"
            ).first()
        if self.Parameters.requests_plan is None:
            self.Parameters.requests_plan = sdk2.Resource.find(
                type=models_services_resources.MODELS_SERVICE_REQUESTS_PLAN,
                attrs=dict(released="stable", service_type=self.Parameters.service_type),
                state="READY"
            ).first()
        restrict_dc = {'SAS': ctc.Tag.SAS, 'MAN': ctc.Tag.MAN, 'VLA': ctc.Tag.VLA}.get(str(self.Parameters.restrict_dc))
        if restrict_dc is not None:
            self.Requirements.client_tags = restrict_dc

    def on_execute(self):
        dexecutor_executable = str(sdk2.ResourceData(self.Parameters.dexecutor).path)
        requests_plan_path = str(sdk2.ResourceData(self.Parameters.requests_plan).path)
        if self.Parameters.requests_plan.service_type != self.Parameters.service_type:
            raise TaskFailure(
                "Mismatch of service types: requests plan has %s, while task has %s" %
                (self.Parameters.requests_plan.service_type, self.Parameters.service_type)
            )
        shoot_dump_file = "{}_shoot.dump".format(self.Parameters.service_type)
        eventlog_path = "{}.evlog".format(self.Parameters.service_type)
        nanny_client = None
        nanny_service_paused = False
        if self.Parameters.run_mode == RunMode.LOCAL:
            host = "localhost"
            port = str(get_free_port())
            if self.Parameters.service_type == ServiceType.RTMODELS:
                service_executable_path = str(sdk2.ResourceData(self.Parameters.rtmodels_binary).path)
                data_path = str(sdk2.ResourceData(self.Parameters.rtmodels_data).path)
                config_path = str(sdk2.ResourceData(self.Parameters.rtmodels_begemot_config).path)
            else:
                service_executable_path = str(sdk2.ResourceData(self.Parameters.cfg_models_binary).path)
                models_archive_path = str(sdk2.ResourceData(self.Parameters.cfg_models_archive).path)
                rule_config_path = str(sdk2.ResourceData(self.Parameters.cfg_models_rule_config).path)
                data_path = os.path.join(self.abs_path(), "service_data", "ConfigurableModels")
                os.mkdir(data_path)
                os.symlink(models_archive_path, os.path.join(data_path, "cfg_models_bundle"))
                os.symlink(rule_config_path, os.path.join(data_path, "configurable_models.cfg"))
                config_path = str(sdk2.ResourceData(self.Parameters.cfg_models_begemot_config).path)
            service_interface = get_begemot(
                binary_path=service_executable_path,
                port=port,
                config_path=config_path,
                worker_dir=data_path,
                eventlog_path=eventlog_path,
                start_timeout=1800,
            )
            service_interface.start()
            service_interface.wait()
        else:
            port = str(self.Parameters.port)
            if self.Parameters.nanny_service:
                oauth_token = self.Parameters.nanny_oauth_token.data()[self.Parameters.nanny_oauth_token.default_key]
                nanny_client = NannyClient('http://nanny.yandex-team.ru/', oauth_token)
                instances = nanny_client.get_service_current_instances(self.Parameters.nanny_service)
                chosen_pod = random.choice(instances['result'])
                logging.info("chosen pod: {}".format(chosen_pod))
                host = chosen_pod["container_hostname"]
                if self.Parameters.nanny_snapshot:
                    service_state = nanny_client.get_service(self.Parameters.nanny_service)
                    logging.info("current service state: {}".format(service_state))
                    if service_state['current_state']['content']['is_paused']['value']:
                        raise Exception("service is already paused, probably somebody else is shooting")
                    active = [s for s in service_state['current_state']['content']['active_snapshots'] if s['state'] == 'ACTIVE']
                    if len(active) != 1:
                        raise Exception("no active snapshots found")
                    if not active[0]["snapshot_id"].startswith(self.Parameters.nanny_snapshot):
                        raise Exception("expected snapshot {}, active snapshot is {}".format(self.Parameters.nanny_snapshot, active[0]["snapshot_id"]))
                    pause_comment = "shooting task {} started".format(self.id)
                    nanny_client.create_event(self.Parameters.nanny_service, {"type": "PAUSE_ACTIONS", "content": {"comment": pause_comment}})
                    nanny_service_paused = True
            else:
                host = self.Parameters.host
                if not host:
                    raise Exception("Either host or nanny_service must be given for remote mode")
            # TODO: ping to ensure that service is alive; call `/admin?action=reopenlog`; download eventlog after shoot
        try:
            command = [
                dexecutor_executable,
                "--plan-file", requests_plan_path,
                "--output", shoot_dump_file,
                "--replace-host", host,
                "--replace-port", port,
                "--rps-fixed", str(self.Parameters.rps),
            ]
            if self.Parameters.queries_limit:
                command += ["--queries-limit", str(self.Parameters.queries_limit)]
            if self.Parameters.need_dump_responses:
                command += ["--dump-data", "--output-base64-body"]
            else:
                command += [
                    "--time-limit", str(self.Parameters.shoot_time),
                    "--circular",
                ]

            with sdk2.helpers.ProcessLog(self, logger="dexecutor") as dexecutor_pl:
                subprocess.check_call(command, stdout=dexecutor_pl.stdout, stderr=dexecutor_pl.stderr)

        finally:
            if nanny_service_paused:
                resume_comment = "shooting task {} done".format(self.id)
                nanny_client.create_event(self.Parameters.nanny_service, {"type": "RESUME_ACTIONS", "content": {"comment": resume_comment}})
                nanny_service_paused = False

        if self.Parameters.run_mode == RunMode.LOCAL:
            service_interface.stop()
            eventlog = begemot_resources.BEGEMOT_EVENTLOG(
                self,
                "%s eventlog" % self.Parameters.service_type,
                eventlog_path
            )
            sdk2.ResourceData(eventlog).ready()

        shoot_dump = resource_types.EXECUTOR_DUMP(
            self,
            "Dolbilka shoot dump for %s" % self.Parameters.service_type,
            shoot_dump_file
        )
        self.set_meta_info(shoot_dump)
        sdk2.ResourceData(shoot_dump).ready()

    def set_meta_info(self, shoot_dump):
        shoot_dump.service_type = self.Parameters.service_type
        shoot_dump.with_responses = self.Parameters.need_dump_responses
