# -*- coding: utf-8 -*-

import os

from sandbox import sdk2

from sandbox.common.types.task import Semaphores
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.websearch.performance_report import resources
from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault


class CalculateAppHostPerformanceReport(sdk2.Task):
    """
        Составляет отчет по временам источников аппхоста по часовым логам
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 50  # 50 Mb

        class Caches(sdk2.Requirements.Caches):
            pass  # do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        perf_report_tool = ResourceWithLastReleasedValueByDefault(
            'AppHost Performance Report tool (search/tools/app_host_perf_report)',
            resource_type=resources.PerformanceReportTool,
            required=True,
        )
        hour = sdk2.parameters.String('Log table name in yyyy-mm-dd[Thh:00:00] format', required=True)
        yt_pool = sdk2.parameters.String('YT pool')
        yt_token_owner = sdk2.parameters.String('YT_TOKEN owner', required=True)
        full_from_hours = sdk2.parameters.Bool('Concatenate hour tables of a day (YYYY-MM-DD) into a full day stat', default=False)
        vertical = sdk2.parameters.String('Vertical')

        with sdk2.parameters.Output:
            report = sdk2.parameters.Resource('Resulting report', resource_type=resources.PerformanceReport)

    def on_enqueue(self):
        pool = self.Parameters.yt_pool or self.Parameters.yt_token_owner
        semaphore_name = 'process_performance_report_{}'.format(pool)
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(
                    name=semaphore_name,
                    capacity=3
                ),
            ],
        )

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_owner, name='YT_TOKEN')
        perf_report_tool_path = str(sdk2.ResourceData(self.Parameters.perf_report_tool).path)

        stat_file = 'perf-report-{}'.format(self.Parameters.hour)
        self.Parameters.report = resources.PerformanceReport(
            self, self.Parameters.description, stat_file, ttl=90,
        )

        env = os.environ.copy()
        if self.Parameters.yt_pool:
            env['YT_POOL'] = self.Parameters.yt_pool

        perf_tool_args = [
            perf_report_tool_path,
            '--date', self.Parameters.hour,
            '--stat-file', stat_file,
            '--yt-token', yt_token,
            '--timeout', str(self.Parameters.kill_timeout),
        ]

        if self.Parameters.vertical:
            perf_tool_args.extend(['--vertical', self.Parameters.vertical])

        if self.Parameters.full_from_hours:
            perf_tool_args.append('--full-from-hours')

        process = subprocess.Popen(perf_tool_args, env=env)
        process.wait()
