# -*- coding: utf-8 -*-

import json
import pytz

from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.sandboxsdk import environments

from sandbox.projects.websearch.performance_report.CalculateAppHostPerformanceReport import CalculateAppHostPerformanceReport


class UpdateAppHostPerformanceReportDaily(sdk2.Task):
    """
        Задача обновляет отчет со статистикой по временам источников аппхоста за целый день
    """

    APPHOST_EVENT_LOG_PATH = '//logs/apphost-event-log'

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 50  # 50 Mb
        environments = [
            environments.PipEnvironment("requests", version="2.18.4", custom_parameters=["--upgrade-strategy only-if-needed"]),
            environments.PipEnvironment('python-statface-client', version="0.154.0", custom_parameters=["--upgrade-strategy only-if-needed"]),
            environments.PipEnvironment('yandex-yt'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass  # do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        day = sdk2.parameters.String('Select day in "YYYY-mm-dd" format (previous day if not specified)')
        only_full_day = sdk2.parameters.Bool('Stop if any of 24 hour tables do not exist', default=True)
        vertical = sdk2.parameters.String('Vertical', required=True, default='WEB')
        statface_report_path = sdk2.parameters.String('Statface report path', required=True)

        yt_pool = sdk2.parameters.String('YT pool')
        yt_token_owner = sdk2.parameters.String('YT_TOKEN owner', required=True)
        statface_token_owner = sdk2.parameters.String('STATFACE_TOKEN owner', required=True)

    def get_statface_report(self, report_path):
        import statface_client
        statface_token = sdk2.Vault.data(self.Parameters.statface_token_owner, name='STATFACE_TOKEN')
        sf_client = statface_client.StatfaceClient(host=statface_client.STATFACE_PRODUCTION, oauth_token=statface_token)
        return sf_client.get_report(report_path)

    def get_day(self):
        day = self.Parameters.day
        if not day:
            yesterday = datetime.now(pytz.timezone('Europe/Moscow')) - timedelta(days=1)
            day = yesterday.strftime('%Y-%m-%d')
        return day

    def on_execute(self):
        import statface_client
        from yt.wrapper import YtClient

        with self.memoize_stage.launch_calculation:
            yt_token = sdk2.Vault.data(self.Parameters.yt_token_owner, name='YT_TOKEN')
            yt_client = YtClient('hahn', yt_token)

            day = self.get_day()
            if self.Parameters.only_full_day:
                for h in range(24):
                    table = '{}/1h/{}T{:0>2}:00:00'.format(self.APPHOST_EVENT_LOG_PATH, day, h)
                    if not yt_client.exists(table):
                        raise errors.TaskFailure("table {} does not exist".format(table))

            self.Context.calc_subtask = CalculateAppHostPerformanceReport(
                self,
                description='Calculate AppHost Performance Report for {}'.format(day),
                kill_timeout=self.Parameters.kill_timeout - 300,
                hour=day,
                yt_pool=self.Parameters.yt_pool,
                yt_token_owner=self.Parameters.yt_token_owner,
                full_from_hours=True,
                vertical=self.Parameters.vertical,
            ).enqueue().id

            raise sdk2.WaitTask([self.Context.calc_subtask], Status.Group.FINISH | Status.Group.BREAK, wait_all=True)

        report_file_path = str(sdk2.ResourceData(sdk2.Task[self.Context.calc_subtask].Parameters.report).path)
        report = json.load(open(report_file_path, 'r'))
        sf_report = self.get_statface_report(self.Parameters.statface_report_path)
        sf_report.upload_data(scale=statface_client.constants.DAILY_SCALE, data=report)
