# -*- coding: utf-8 -*-

import copy
import itertools
import logging

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.input_params as rm_params
from sandbox.projects import resource_types
from sandbox.projects.websearch.upper.CombineNoapacheupperRearranges import Params as CombRearrParams

from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common.noapacheupper.instance import Params as NoapacheParams
from sandbox.projects.common.noapacheupper.resources import ImportNoapacheupperResources
from sandbox.projects.common.noapacheupper.search_component import Params as ComponentParams
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine.components.configs.upper import UpperCfg

from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search.components import VerifyStderrCommon
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.common.search.requester import Params as RequesterParams

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel

import sandbox.common.types.client as ctc


_CHANGE_THRESHOLD = 0.8


class NoapacheKind:
    """
        Описание подвида noapache + ресурсов для его тестирования
    """

    def __init__(self, name, beta_pname, sample_beta_pname, resources_marker):
        self.name = name
        self.beta_pname = beta_pname
        self.sample_beta_pname = sample_beta_pname
        self.resources_marker = resources_marker


class Params:

    class SubtestsSelector(parameters.SandboxStringParameter):
        CACHEHIT = 'Cachehit'
        FAULT_TOLERANCE = 'Fault tolerance'

        name = 'subtests_selector'
        description = 'Subtests selector'
        default_value = CACHEHIT
        choices = [
            (CACHEHIT, CACHEHIT),
            (FAULT_TOLERANCE, FAULT_TOLERANCE),
        ]

    class CachehitRequestsLimit(parameters.SandboxIntegerParameter):
        name = 'cachehit_requests_limit'
        description = 'Cachehit tests requests limit'
        default_value = 1000

    class CombineRearrangeRequestsLimit(parameters.SandboxIntegerParameter):
        name = 'combine_rearrange_requests_limit'
        description = 'Rearrange combine tests requests limit'
        default_value = 1000

    class CheckedWebKubrBeta(parameters.SandboxStringParameter):
        name = 'checked_beta_web_kubr'
        description = 'Checked WEB_KUBR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.ru/search/?'

    class CheckedImgsKubrBeta(parameters.SandboxStringParameter):
        name = 'checked_beta_imgs_kubr'
        description = 'Checked IMGS_KUBR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.ru/images/search/?'

    class SampleWebKubrBeta(parameters.SandboxStringParameter):
        name = 'sample_beta_web_kubr'
        description = 'Sample WEB_KUBR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.ru/search/?'

    class SampleImgsKubrBeta(parameters.SandboxStringParameter):
        name = 'sample_beta_imgs_kubr'
        description = 'Sample IMGS_KUBR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.ru/images/search/?'

    class CheckedWebTrBeta(parameters.SandboxStringParameter):
        name = 'checked_beta_web_tr'
        description = 'Checked WEB_TR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.com.tr/search/?'

    class CheckedImgsTrBeta(parameters.SandboxStringParameter):
        name = 'checked_beta_imgs_tr'
        description = 'Checked IMGS_TR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.com.tr/images/search/?'

    class SampleWebTrBeta(parameters.SandboxStringParameter):
        name = 'sample_beta_web_tr'
        description = 'Sample WEB_TR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.com.tr/search/?'

    class SampleImgsTrBeta(parameters.SandboxStringParameter):
        name = 'sample_beta_imgs_tr'
        description = 'Sample IMGS_TR beta (or \'-\' if not need)'
        default_value = 'https://upper-hamster.hamster.yandex.com.tr/images/search/?'

    class AdditionalCgiParamsForSampleBeta(parameters.SandboxStringParameter):
        name = 'sample_beta_additional_cgi_params'
        description = 'Additional cgi params for sample betas'
        default_value = ''

    class AdditionalCgiParamsForCheckedBeta(parameters.SandboxStringParameter):
        name = 'checked_beta_additional_cgi_params'
        description = 'Additional cgi params for checked betas'
        default_value = ''

    params = (
        SubtestsSelector,
        CachehitRequestsLimit,
        CombineRearrangeRequestsLimit,
        CheckedWebKubrBeta,
        SampleWebKubrBeta,
        CheckedImgsKubrBeta,
        SampleImgsKubrBeta,
        CheckedWebTrBeta,
        SampleWebTrBeta,
        CheckedImgsTrBeta,
        SampleImgsTrBeta,
        AdditionalCgiParamsForSampleBeta,
        AdditionalCgiParamsForCheckedBeta,
    )


"""_NOAPACHE_KINDS = [ #TODO uncomment SEARCH-5466
    NoapacheKind(
        'WEB_KUBR',
        Params.CheckedWebKubrBeta.name,
        Params.SampleWebKubrBeta.name,
        'web_kubr',
    ),
    NoapacheKind(
        'IMGS_KUBR',
        Params.CheckedImgsKubrBeta.name,
        Params.SampleImgsKubrBeta.name,
        'images_kubr',
    ),
    NoapacheKind(
        'WEB_TR',
        Params.CheckedWebTrBeta.name,
        Params.SampleWebTrBeta.name,
        'web_tr',
    ),
    NoapacheKind(
        'IMGS_TR',
        Params.CheckedImgsTrBeta.name,
        Params.SampleImgsTrBeta.name,
        'images_tr',
    ),
]"""

_NOAPACHE_KINDS = [
    NoapacheKind(
        'WEB_KUBR',
        Params.CheckedWebKubrBeta.name,
        Params.SampleWebKubrBeta.name,
        'web_kubr',
    ),
    NoapacheKind(
        'WEB_TR',
        Params.CheckedWebTrBeta.name,
        Params.SampleWebTrBeta.name,
        'web_tr',
    ),
]


@rm_notify.notify2()
class AcceptanceTestNoapacheupper(bugbanner.BugBannerTask, ImportNoapacheupperResources):
    """
        **Описание**

        Приёмочный тест (только на стабильность/производительность, но не качество выходных даных) noapacheupper.
        Является мета-задачей sandbox для запуска специализированных тестов:

            * WORKALOAD_NOAPACHEUPPER
               * обстрел на стабильность (падения/повисания) - используется самый свежий ресурс
                 типа GZIPPED_REQUESTS_FOR_NOAPACHEUPPER, помеченный атрибутом
                 acceptance_test=вид_noapache (web_kubr, images_kubr, web_tr, images_tr)
            * CHECK_CACHEHIT_FOR_BETA
               * оценка попаданий в кеш среднего, набранный с использованием предыдущей версии noapacheupper
                 + проверка уровня попаданий в кеш, набранный самим собой.
            * TEST_NOAPACHEUPPER_MEMCHECK
               * проверка проблем с памятью, при использовании отладочных cgi-параметров в запросе
    """

    type = 'ACCEPTANCE_TEST_NOAPACHEUPPER'
    environment = [task_env.TaskRequirements.startrek_client]
    client_tags = task_env.TaskTags.startrek_client & ctc.Tag.Group.LINUX
    input_parameters = Params.params + (
        NoapacheParams.RequestsLimit,
        NoapacheParams.WorkersCount,
    )
    execution_space = 1024  # 1 Gb Average usage is near 10 Mb
    noapache_kinds = None

    def _knd_list(self):
        return [copy.deepcopy(knd) for knd in _NOAPACHE_KINDS if len(self.ctx[knd.beta_pname]) > 4]

    def on_enqueue(self):
        # перечисляем виды noapache-ей, которые нужно тестировать
        self.noapache_kinds = self._knd_list()
        self.ctx['need_import'] = [self.ctx[knd.beta_pname] for knd in self.noapache_kinds]
        self.ctx['beta_resources'] = {}
        if self.ctx[Params.SubtestsSelector.name] == Params.SubtestsSelector.CACHEHIT:
            # ресурс со статистикой кешхита
            self.ctx["cache_miss_stats_res_id"] = self.create_resource(
                'Cache miss stats', "cache_miss_stats.txt", resource_types.UPPER_CACHE_MISS_STATS
            ).id

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.NoapacheUpper)
        self.noapache_kinds = sorted(self._knd_list(), key=lambda x: x.name)
        # импортируем ресурсы всех типов бет
        # (import_resources запускает подзадачи, так что при его вызове присутствует sandbox-овая магия)
        while self.ctx['need_import']:
            current_imported_beta = self.ctx['need_import'][0]
            if current_imported_beta not in self.ctx['beta_resources']:
                resources = self.import_resources(current_imported_beta)
                self.ctx['beta_resources'][current_imported_beta] = resources
            self.ctx['need_import'].pop(0)

        if 'test_tasks' not in self.ctx:
            self._run_tests()
        utils.check_if_tasks_are_ok(self.ctx['test_tasks'])
        if self.ctx[Params.SubtestsSelector.name] == Params.SubtestsSelector.CACHEHIT:
            self._get_stats()

    def _run_tests(self):
        # готовим ресурсы для тестов
        all_requests_id = channel.sandbox.list_resources(
            resource_types.GZIPPED_BINARY_REQUESTS_FOR_NOAPACHEUPPER,
            omit_failed=True,
            attribute_name='acceptance_test',
            attribute_value='web_all',
        )[0].id
        for knd in self.noapache_kinds:
            knd.all_requests_id = all_requests_id
            knd.resources = self.ctx['beta_resources'][self.ctx[knd.beta_pname]]

            attr_name = 'autoupdate_resources_noapacheupper_{}_trunk_task_id'.format(knd.resources_marker)
            logging.debug("Getting BINARY_REQUESTS_FOR_NOAPACHEUPPER with attribute %s", attr_name)
            knd.requests_id = channel.sandbox.list_resources(
                resource_types.BINARY_REQUESTS_FOR_NOAPACHEUPPER,
                omit_failed=True,
                attribute_name=attr_name,
            )[0].id

            # если есть пользовательские запросы, помеченные атрибутом использования в acceptance_test-е
            # сохраняем их id, тогда будут использованы соотв. тесты - cachehit/combine_rearrange
            lr = channel.sandbox.list_resources(
                resource_types.USERS_QUERIES,
                omit_failed=True,
                attribute_name='acceptance_test',
                attribute_value=knd.resources_marker,
            )
            knd.queries_id = lr[0].id if lr else None

        # запускаем тесты
        tasks = []
        if self.ctx[Params.SubtestsSelector.name] == Params.SubtestsSelector.FAULT_TOLERANCE:
            self._add_workaload_test(tasks)
            for knd in self.noapache_kinds:
                self._add_fault_tolerance_tests(knd, tasks)
        elif self.ctx[Params.SubtestsSelector.name] == Params.SubtestsSelector.CACHEHIT:
            for knd in self.noapache_kinds:
                self._add_cachehit_tests(knd, tasks)
        self.ctx['test_tasks'] = tasks
        self.wait_all_tasks_completed(self.ctx['test_tasks'])

    def _add_workaload_test(self, tasks):
        """Simple run and checking fails"""
        for k, v in itertools.groupby(self.noapache_kinds, key=lambda x: x.name.split("_")[0]):
            # https://st.yandex-team.ru/SEARCH-3126#1507635980000
            knd = list(v)[0]  # one task for group (WEB, IMAGES)
            tasks.append(self.create_subtask(
                task_type='WORKALOAD_NOAPACHEUPPER',
                arch='linux',
                input_parameters={
                    'notify_if_finished': '',
                    'kill_timeout': 60 * 60 * 4,
                    ComponentParams.Binary.name: knd.resources['NOAPACHE_UPPER'],
                    ComponentParams.AppHostMode.name: True,
                    'use_http_get': False,
                    ComponentParams.RearrangeData.name: knd.resources['REARRANGE_DATA'],
                    ComponentParams.RearrangeDynamicData.name: knd.resources['REARRANGE_DYNAMIC_DATA'],
                    ComponentParams.Config.name: knd.resources['NOAPACHEUPPER_CONFIG'],
                    NoapacheParams.Requests.name: knd.all_requests_id,
                    NoapacheParams.RequestsLimit.name: utils.get_or_default(self.ctx, NoapacheParams.RequestsLimit),
                    NoapacheParams.WorkersCount.name: utils.get_or_default(self.ctx, NoapacheParams.WorkersCount),
                },
                description='{}: stability test: {}'.format(self.descr, k)
            ).id)

    def _add_fault_tolerance_tests(self, knd, tasks):
        if knd.queries_id:
            req_limit = utils.get_or_default(self.ctx, Params.CombineRearrangeRequestsLimit)
            # Тесты проверки на падения при отключении отдельных переранжирований
            tasks.append(self.create_subtask(
                task_type='COMBINE_NOAPACHEUPPER_REARRANGES',
                arch='linux',
                input_parameters={
                    'notify_if_finished': '',
                    'kill_timeout': 60 * 60 * 3,
                    CombRearrParams.ReportUrl.name: self.ctx[knd.beta_pname],
                    CombRearrParams.UsersQueries.name: knd.queries_id,
                    ComponentParams.Binary.name: knd.resources['NOAPACHE_UPPER'],
                    ComponentParams.AppHostMode.name: True,
                    ComponentParams.RearrangeData.name: knd.resources['REARRANGE_DATA'],
                    ComponentParams.RearrangeDynamicData.name: knd.resources['REARRANGE_DYNAMIC_DATA'],
                    ComponentParams.Config.name: knd.resources['NOAPACHEUPPER_CONFIG'],
                    NoapacheParams.Requests.name: knd.requests_id,
                    NoapacheParams.RequestsLimit.name: req_limit,
                    NoapacheParams.WorkersCount.name: utils.get_or_default(self.ctx, RequesterParams.WorkersCount),
                    VerifyStderrCommon.name: False,
                },
                description='{}: stability test: {}'.format(self.descr, knd.name)
            ).id)

    def _add_cachehit_tests(self, knd, tasks):
        beta = self.ctx[knd.beta_pname]
        sample_beta = self.ctx[knd.sample_beta_pname]
        if len(beta) > 4 and len(sample_beta) > 4 and knd.queries_id:
            if knd.name.startswith('WEB'):
                tasks.append(self.create_subtask(
                    task_type='CHECK_CACHEHIT_FOR_BETA',
                    arch='linux',
                    input_parameters={
                        'notify_if_finished': '',
                        'checked_beta': beta,
                        'sample_beta': sample_beta,
                        'nmeta_type': 'WEB',
                        'mmeta_type': 'WEB',
                        'users_queries_resource_id': knd.queries_id,
                        'limit_users_queries': self.ctx[Params.CachehitRequestsLimit.name],
                        'checked_beta_additional_cgi_params': self.ctx[Params.AdditionalCgiParamsForCheckedBeta.name],
                        'sample_beta_additional_cgi_params': self.ctx[Params.AdditionalCgiParamsForSampleBeta.name],
                    },
                    description='{}: cache WEB hit test: {}'.format(self.descr, knd.name)
                ).id)
            if knd.name.startswith('WEB') or knd.name.startswith('IMGS'):
                nmeta_type = 'IMGS' if knd.name.startswith('IMGS') else 'WEB'
                tasks.append(self.create_subtask(
                    task_type='CHECK_CACHEHIT_FOR_BETA',
                    arch='linux',
                    input_parameters={
                        'notify_if_finished': '',
                        'checked_beta': beta,
                        'sample_beta': sample_beta,
                        'nmeta_type': nmeta_type,
                        'mmeta_type': 'IMGS',
                        'users_queries_resource_id': knd.queries_id,
                        'limit_users_queries': self.ctx[Params.CachehitRequestsLimit.name],
                        'checked_beta_additional_cgi_params': self.ctx[Params.AdditionalCgiParamsForCheckedBeta.name],
                        'sample_beta_additional_cgi_params': self.ctx[Params.AdditionalCgiParamsForSampleBeta.name],
                    },
                    description='{}: cache IMGS hit test: {}'.format(self.descr, knd.name)
                ).id)

    def _get_stats(self):
        """
            Анализируем оценку попаданий запросами в кеш, набранный предыдущим noapache.
            Оценку/прогноз понижения попаданий прописываем в контекст задачи (для использования извне)
        """
        idx_sample = 0
        idx_beta = 1
        cache_miss_stats = ""
        for task_id in self.ctx['test_tasks']:
            task = channel.sandbox.get_task(task_id)
            if task.type == "CHECK_CACHEHIT_FOR_BETA":
                logging.debug('analyze check cachehit task:{} result'.format(task_id))
                for k, v in task.ctx['stat'].iteritems():
                    if k == 'ALL':
                        continue
                    sample_cache_hit_ratio = v['cache_hit_ratio'][idx_sample]
                    beta_cache_hit_ratio = v['cache_hit_ratio'][idx_beta]
                    if sample_cache_hit_ratio > 0.1:
                        change_cache_hit_ratio = beta_cache_hit_ratio / sample_cache_hit_ratio
                        if change_cache_hit_ratio < _CHANGE_THRESHOLD:
                            cache_miss_stats += (
                                "nmeta({}) -> mmeta({})/{} "
                                "cachehit ratio "
                                "old({:.3f}) -> new({:.3f}) == {:.2f}%\n"
                            ).format(
                                task.ctx['nmeta_type'], task.ctx['mmeta_type'], k,
                                sample_cache_hit_ratio, beta_cache_hit_ratio, change_cache_hit_ratio * 100
                            )
        path_to_stats = channel.sandbox.get_resource(self.ctx["cache_miss_stats_res_id"]).path
        fu.write_file(path_to_stats, cache_miss_stats or "Cache is Ok")
        self.ctx["short_cache_result"] = ["Cache:Fail", "Cache:Ok"][not cache_miss_stats]
        self.ctx["long_cache_result"] = cache_miss_stats or "Cache is Ok"
        st_helper = STHelper(self.get_vault_data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        st_helper.comment(
            self.ctx.get(rm_params.ReleaseNum.name),
            text="Acceptance task ({cache_result}): {wikilink}".format(
                cache_result=self.ctx["short_cache_result"],
                wikilink=lb.task_wiki_link(self.id)
            ),
            c_info=rmc.get_component(UpperCfg.name)
        )

    def get_short_task_result(self):
        return self.ctx.get("short_cache_result")


__Task__ = AcceptanceTestNoapacheupper
