# -*- coding: utf-8 -*-
import logging

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common import apihelpers
from sandbox.projects.websearch.upper.BuildNoapacheupperRequests import Params as RequestsParams
from sandbox.projects.websearch.upper.BuildNoapacheupperRequests2 import Params as RequestsParams2
from sandbox.projects.websearch.upper.UpdateStandaloneNoapacheupperResources import UpdateStandaloneNoapacheupperResources
from sandbox.projects.websearch.upper.RequestSampler import RequestSampler
from sandbox.projects.websearch.upper import resources as upper_resources

from sandbox import sdk2
from sandbox.sandboxsdk.task import SandboxTask


def create_update_task(
    current_task, descr, db, subtype, config_url, report_url,
    rtmr_url=None,
    evlogdump_resource_id=None,
    additional_cgi_params=None,
    apphost_mode=True,
    requests_limit=None,
):
    params = {
        UpdateStandaloneNoapacheupperResources.Parameters.testenv_db.name: db,
        UpdateStandaloneNoapacheupperResources.Parameters.test_subtype.name: subtype,
        UpdateStandaloneNoapacheupperResources.Parameters.noapacheupper_config_url.name: config_url,
        UpdateStandaloneNoapacheupperResources.Parameters.noapacheupper_apphost_mode.name: apphost_mode,
        RequestsParams.ReportUrl.name: report_url,
        RequestsParams.RtmrUrl.name: rtmr_url,
        RequestsParams.AdditionalCgiParams.name: additional_cgi_params,
        RequestsParams2.EvlogdumpExecutable.name: evlogdump_resource_id,
    }
    if requests_limit:
        params[UpdateStandaloneNoapacheupperResources.Parameters.requests_limit.name] = requests_limit

    logging.info("create task {} with params: {}".format(UpdateStandaloneNoapacheupperResources.type, str(params)))
    task_class = sdk2.Task['UPDATE_STANDALONE_NOAPACHEUPPER_RESOURCES']
    task_class(
        task_class.current,
        description=descr,
        owner=current_task.owner,
        priority=current_task.priority,
        **params
    ).enqueue()


def create_request_sampler_task(
    current_task, descr, request_sampler_resource_id,
    sources=('NOAPACHE_ADJUSTER', 'BLENDER'),
    output_type="grpc-client-plan",
    graphs='WEB',
    number_of_requests=20000,
    attr_name_suffix=''
):
    suffix = '{}_'.format(graphs.lower())
    if attr_name_suffix:
        suffix += '{}_'.format(attr_name_suffix)
    attr_name_to_set = "autoupdate_resources_noapacheupper_request_sampler_{}trunk_task_id".format(suffix).replace('-', '_')
    params = {
        RequestSampler.Parameters.attribute_name.name: attr_name_to_set,
        'sources': list(sources),
        'request_sampler': request_sampler_resource_id,
        'number_of_requests': number_of_requests,
        'output_type': output_type,
        'graphs': graphs,
        'yt_token_vault_owner': 'PROD-RESOURCE-GETTER',
        'yt_token_vault_key': 'yt_token_for_eventlogs',
    }
    logging.info("create task {} with params: {}".format(RequestSampler.type, str(params)))
    task_class = sdk2.Task['REQUEST_SAMPLER']
    task_class(
        task_class.current,
        description=descr,
        owner=current_task.owner,
        priority=current_task.priority,
        **params
    ).enqueue()


class Vertical(object):
    def __init__(self, name, path):
        self.name = name
        self.path = path

    def __eq__(self, other):
        return self.name == other.name and self.path == other.path


class Locale(object):
    def __init__(self, report_host, name):
        self.report_host = report_host
        self.name = name

    def __eq__(self, other):
        return self.name == other.name and self.report_host == other.report_host

    def config_url(self):
        return 'https://{}/viewconfig'.format(self.report_host)

    def report(self, path):
        if isinstance(path, Vertical):
            return 'https://{}{}'.format(self.report_host, path.path)
        return 'https://{}{}'.format(self.report_host, path)


class BatchNoapacheTestResourcesUpdaters(SandboxTask, object):
    """
        Метатаск предназначен для регулярного оптового обновления ресурсов,
        используемых в покомитных testenv тестах noapacheupper
        (регулярного автоматического запуска из scheduler-а).
        Для db noapacheupper-trunk генерируются apphost тесты,
        для прочих (noapacheupper2-trunk - video+images+misc) НЕ apphost
    """

    execution_space = 1024  # 1Gb
    type = "BATCH_NOAPACHE_TEST_RESOURCES_UPDATER"

    def on_execute(self):
        self.ctx['testenv_db'] = 'noapacheupper-trunk'
        if not self.list_subtasks():
            db = self.ctx['testenv_db']
            evlogdump_resource_id = apihelpers.get_last_released_resource(resource_types.EVLOGDUMP_EXECUTABLE).id
            request_sampler_resource_id = apihelpers.get_last_released_resource(
                upper_resources.REQUEST_SAMPLER_EXECUTABLE
            ).id

            web_vrt = Vertical('web', '/yandsearch?')
            # images_vrt = Vertical('imgs', '/images/search?')
            # video_vrt = Vertical('video', '/video/search?')
            # all_vrt = [web_vrt, images_vrt, video_vrt]

            ru_report = 'hamster.yandex.ru'

            ru_locale = Locale(ru_report, 'kubr')
            all_locales = [ru_locale]

            no_tests_mode = '&no-tests=da&'
            # + '&flag=noapache_json_req%3Dapp_host_separate%3Aupper2%3Aupper'

            update_task_common_params = {
                # https://st.yandex-team.ru/UPREL-1308#1482925435000
                'rtmr_url': 'http://rtmr.search.yandex.net:8080/yandsearch?',
                'db': db,
                'evlogdump_resource_id': evlogdump_resource_id,
            }

            base_descr = 'update static resources for testenv db {} '.format(db)
            create_request_sampler_task(
                self,
                descr=base_descr + '(request_sampler for adjuster and blender grpc)',
                request_sampler_resource_id=request_sampler_resource_id,
            )
            create_request_sampler_task(
                self,
                descr=base_descr + '(request_sampler for images blender)',
                request_sampler_resource_id=request_sampler_resource_id,
                graphs='IMAGES',
                sources=('BLENDER',),
            )
            create_request_sampler_task(
                self,
                descr=base_descr + '(request_sampler for video adjuster and blender)',
                request_sampler_resource_id=request_sampler_resource_id,
                graphs='VIDEO',
                number_of_requests=10000,
            )

            create_request_sampler_task(
                self,
                descr='Update requests for testing noapache fast data (WEB)',
                request_sampler_resource_id=request_sampler_resource_id,
                sources=('BLENDER',),
                graphs='WEB',
                number_of_requests=1001,
                attr_name_suffix='fast-data',
            )
            create_request_sampler_task(
                self,
                descr='Update requests for testing noapache fast data (VIDEO)',
                request_sampler_resource_id=request_sampler_resource_id,
                sources=('BLENDER',),
                graphs='VIDEO',
                number_of_requests=1001,
                attr_name_suffix='fast-data',
            )

            for locale in all_locales:
                for vrt in [web_vrt]:  # apphost for images/video not enabled yet
                    create_update_task(
                        self,
                        descr=base_descr + '({}-{})'.format(vrt.name, locale.name),
                        subtype='{}_{}'.format(vrt.name, locale.name),
                        config_url=locale.config_url(),
                        report_url=locale.report(vrt),
                        additional_cgi_params=no_tests_mode,
                        **update_task_common_params
                    )

            create_update_task(
                self,
                descr=base_descr + '(web-kubr-farpages)',
                subtype='web_kubr_farpages',
                config_url=ru_locale.config_url(),
                report_url=ru_locale.report(web_vrt),
                additional_cgi_params=no_tests_mode + '&p=50',
                requests_limit=1000,
                **update_task_common_params
            )

            touch_vrt = Vertical('touch', '/search/touch/?')
            create_update_task(
                self,
                descr=base_descr + '({}-{})'.format(touch_vrt.name, ru_locale.name),
                subtype='{}_{}'.format(touch_vrt.name, ru_locale.name),
                config_url=ru_locale.config_url(),
                report_url=ru_locale.report(touch_vrt),
                additional_cgi_params=no_tests_mode + 'noredirect=1&',
                **update_task_common_params
            )

            iznanka_vrt = Vertical('iznanka', '')
            create_update_task(
                self,
                descr=base_descr + '({}-{})'.format(iznanka_vrt.name, ru_locale.name),
                subtype='misc_{}'.format(iznanka_vrt.name),
                config_url=ru_locale.config_url(),
                report_url=ru_locale.report(iznanka_vrt),
                additional_cgi_params=no_tests_mode + 'json_dump=app_host&json_dump=search_props.UPPER.0.properties&',
                **update_task_common_params
            )

            web_random_log_vrt = Vertical('web-kubr-randomlog', '/yandsearch?')
            create_update_task(
                self,
                descr=base_descr + '(web-kubr-randomlog)',
                subtype='web_kubr_randomlog',
                config_url=ru_locale.config_url(),
                report_url=ru_locale.report(web_random_log_vrt),
                additional_cgi_params=(
                    no_tests_mode +
                    'rearr=scheme_Local/RandomLog/Force=1'
                ),
                **update_task_common_params
            )
            utils.wait_all_subtasks_stop()
        else:
            utils.check_subtasks_fails(stop_on_broken_children=True)


__Task__ = BatchNoapacheTestResourcesUpdaters
