# -*- coding: utf-8 -*-

import json
import logging
import os
import shutil
import time

import sandbox.projects.release_machine.security as rm_sec
from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.noapacheupper import beta
from sandbox.projects.common.noapacheupper.search_component import Params as NoapacheParams
from sandbox.projects.common.noapacheupper.search_component import get_noapacheupper
from sandbox.projects.common.search import bisect
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search.bisect import BisectInfo
from sandbox.projects.common.search.eventlog.check_eventlog import Params as EvlogdumpParams
from sandbox.projects.common import decorators
from sandbox.projects.websearch import utils as websearch_utils
from sandbox.sandboxsdk import parameters as params
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.channel import channel

EvlogdumpExecutable = EvlogdumpParams.EvlogdumpExecutable

_RESPONSES = "responses"


class Params:
    class NoapacheBeta(params.SandboxStringParameter):
        name = "noapache_beta"
        description = (
            "Noapache beta with checked noapache version "
            "(MUST be noapache.priemka.yandex.ru or *.hamster.yandex.ru)"
        )
        default_value = "noapache.priemka.yandex.ru"
        required = True

    class MinRevision(params.SandboxIntegerParameter):
        name = "min_revision"
        description = "Minimal revision (optional)"
        default_value = None
        required = False

    class MaxRevision(params.SandboxIntegerParameter):
        name = "max_revision"
        description = "Maximum revision (optional)"
        default_value = None
        required = False

    class CustomConfig(NoapacheParams.Config):
        required = False

    class WaitAfterBisectEnd(params.SandboxIntegerParameter):
        name = "wait_after_bisect_end"
        description = "Interval before bisect end and component shutdown (seconds)"
        default_value = 1
        required = False

    lst = (
        NoapacheBeta,
        MinRevision,
        MaxRevision,
        NoapacheParams.RearrangeDynamicData,
        CustomConfig,
        EvlogdumpExecutable,
        WaitAfterBisectEnd,
    )


class BisectNoapacheupperDiffReason(bisect.Bisect, bugbanner.BugBannerTask):
    """
        Ищем номер комита в noapache, вызвавший diff в выдаче на указанном запросе
        (+ указывается позиция, в которой изменился документ)
        Для определения дипазона ревизий, в которых нужно искать diff, берётся бета noapache.priemka.yandex.ru.

        В процессе поиска поднимаются noapache-и разных ревизий, через которые прогоняется указанный запрос
        (запросы перенаправляются на локально поднятый noapache добалением параметров к запросу-входному
        параметру задачи)
        Данные/конфиг для noapache берутся с беты noapache.priemka.yandex.ru

        На выходе - номер ревизии, изменившей выдачу + описание, какой документ на какой поменялся (url-ы)
    """

    type = "BISECT_NOAPACHEUPPER_DIFF_REASON"

    input_parameters = Params.lst + bisect.Bisect.input_parameters

    config_file = 'noapache.cfg'

    evlogdump = None
    src_acc = {}

    def on_execute(self):
        evlogdump_id = self.ctx.get(EvlogdumpExecutable.name)
        self.evlogdump = self.sync_resource(evlogdump_id) if evlogdump_id else None
        logging.debug("use evlogdump: {}".format(self.evlogdump))
        self.src_acc = {}
        self.add_bugbanner(bugbanner.Banners.ReleaseMachine)

        beta_name = self.ctx[Params.NoapacheBeta.name]
        nanny_token = rm_sec.get_rm_token(self)
        instances = beta.noapache_list(beta_name, nanny_token=nanny_token)
        ver_major, ver_minor = beta.get_noapache_version(instances)
        self.set_bisect_info(BisectInfo.version, ver_major)
        custom_config_resource_id = utils.get_or_default(self.ctx, Params.CustomConfig)
        if custom_config_resource_id:
            # SEARCH-2741
            shutil.copy(self.sync_resource(custom_config_resource_id), self.config_file)
            logging.info("Imported custom config file from resource %s", custom_config_resource_id)
        else:
            websearch_utils.import_config(instances, filename=self.config_file)

        branch_path_template = 'arcadia:/arc/branches/upple/stable-{}'

        # если вручную указаны ревизии, то используем их
        rev_min = utils.get_or_default(self.ctx, Params.MinRevision)
        if rev_min:
            rev_min = int(rev_min)
        else:
            rev_min, _, _ = bisect.branch_last_rev_first_minor_last_minor(branch_path_template.format(ver_major - 1))

        rev_max = utils.get_or_default(self.ctx, Params.MaxRevision)
        if rev_max:
            rev_max = int(rev_max)
            minor_build_info = []
        else:
            # Получаем дополнительно рендж теговых(минорных) ревизий, запускаем get_build_info для бранчовой базы,
            # ниже они будут слиты вместе с транковыми в один build_info
            rev_max, minor_rev_min, minor_rev_max = bisect.branch_last_rev_first_minor_last_minor(
                branch_path_template.format(ver_major),
                ver_minor=ver_minor,
            )
            minor_build_info = bisect.get_build_info(
                'noapacheupper-stable-{}'.format(ver_major),
                'BUILD_NOAPACHEUPPER_RELEASE_LINUX',
                minor_rev_min,
                minor_rev_max,
            )

        # защита от странностей (SEARCH-2503)
        eh.ensure(rev_min < rev_max, "Min revision should be not greater than max revision.")

        self.set_bisect_info(BisectInfo.revisions, '{}:{}'.format(rev_min, rev_max))
        build_info = bisect.get_build_info(
            'noapacheupper-trunk',
            'BUILD_NOAPACHEUPPER_RELEASE_LINUX',
            rev_min,
            rev_max,
        ) + minor_build_info
        # put data resources id to task context
        self.import_rearrange_data(instances)
        self.bisect(build_info)
        if self.src_acc:
            logging.debug("Sources accessibility:\n{}\n\n".format(json.dumps(self.src_acc, indent=4)))
            inaccessible_sources = ''
            for src_name, acc in self.src_acc.iteritems():
                if acc[0] == 0:
                    inaccessible_sources += ' {}({})'.format(src_name, acc[1])
            self.set_bisect_info(BisectInfo.src_acc, inaccessible_sources)

        if os.path.exists(_RESPONSES):
            self.create_resource("Responses", _RESPONSES, resource_type=resource_types.OTHER_RESOURCE)
        else:
            self.set_info("Responses folder does not exist, cannot save. ")

    def import_rearrange_data(self, instances):
        resources = beta.get_resources_id(instances)
        self.ctx[NoapacheParams.RearrangeData.name] = resources['REARRANGE_DATA']
        if not self.ctx.get(NoapacheParams.RearrangeDynamicData.name):
            logging.info("No REARRANGE_DYNAMIC_DATA specified in context, using resources from beta")
            self.ctx[NoapacheParams.RearrangeDynamicData.name] = resources['REARRANGE_DYNAMIC_DATA']
        logging.info("Used REARRANGE_DYNAMIC_DATA resource id: %s", self.ctx[NoapacheParams.RearrangeDynamicData.name])

    def get_docs(self, row, request_url):
        """
            Method for call from bisect algo
        """
        logging.debug(bisect.format_row(row))
        task_id = row['task_id']
        resources = channel.sandbox.list_resources(task_id=task_id, resource_type='NOAPACHE_UPPER')
        if not resources:
            eh.check_failed('Fail get resource NOAPACHE_UPPER from ' + str(task_id))

        try:
            docs = self._get_docs(resources, request_url)
        except Exception as e:
            docs = []
            eh.log_exception("Tried multiple times, but still got empty search result, it's bad(", e)
            self.set_info("Tried multiple times, but still got empty search result, it's bad(")

        result = []
        for i, doc in enumerate(docs, 1):
            doc_data = doc['url']
            logging.debug("#%s %s", i, doc_data)
            for attr in self.doc_attrs:
                doc_data += ' {}={}'.format(attr, repr(doc.get(attr)))
            result.append(doc_data)
        logging.debug("Docs:\n%s\n\n", json.dumps(result, indent=4))
        if not result:
            result = ["document not exists"] * 30
        return result

    @decorators.retries(max_tries=4, delay=5, backoff=5)
    def _get_docs(self, resources, request_url):
        resp = self.get_response(resources[0].id, request_url)
        try:
            resp = json.loads(resp.decode('latin1'))
        except Exception as err:
            eh.log_exception("JSON response parse error", err)
            raise
        docs = resp['searchdata']['docs']
        if not docs:
            logging.debug(resp)
            raise Exception("WARNING: Empty search result, be ready for problems")
        return docs

    def get_response(self, exe_res_id, request_url):

        self.ctx[NoapacheParams.Binary.name] = exe_res_id
        search_component = get_noapacheupper(
            config_file=self.config_file,
            use_verify_stderr=False,
        )
        search_component.start()
        search_component.wait()
        try:
            get_version_url = "http://{}:{}/yandsearch?info=getversion".format(
                self.local_host,
                search_component.port,
            )
            logging.debug('UPPER:[VERSION]\n' + self.read(get_version_url))

            checkconfig_url = "http://{}:{}/yandsearch?info=checkconfig:read-only:1".format(
                self.local_host,
                search_component.port,
            )
            logging.debug('UPPER:[CHECKCONFIG]\n' + self.read(checkconfig_url))

            url = request_url + (
                '&flag=disable_mda'
                '&no-tests=1'
                '&rearr=Personalization_off'
                '&nocache=da'
                '&waitall=da'
                '&timeout=9999999'
                '&srcrwr=UPPER%3A{}%3A{}'
                '&noapache_json_req=0'  # SEARCH-3333 spike
                '&noapache_json_res=0'  # SEARCH-3333 spike
                '&json_dump=searchdata'.format(
                    self.local_host,
                    search_component.port,
                )
            )
            json_response = self.read(url)
            paths.make_folder(_RESPONSES)
            response_file_name = "{}/{}.json".format(_RESPONSES, exe_res_id)
            fu.write_file(response_file_name, json_response)
            return json_response
        finally:
            time_to_wait = int(utils.get_or_default(self.ctx, Params.WaitAfterBisectEnd))
            logging.info("Make a chance to remotely debug this crap (%s seconds)", time_to_wait)
            time.sleep(time_to_wait)

            search_component.stop()
            if self.evlogdump:
                search_component.update_src_accessibility(self.src_acc, self.evlogdump)


__Task__ = BisectNoapacheupperDiffReason
