import os.path

from sandbox import sdk2
from sandbox.common import errors
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.nanny import nanny
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault

TESTENV_TAG_PREFIX = "TESTENV-DATABASE"


class BuildConfigsWithPva(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Builds config files using pva
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024
        ram = 1024

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
        pva = ResourceWithLastReleasedValueByDefault(
            'PVA executable',
            resource_type=app_host_resources.APP_HOST_TOOL_PVA_EXECUTABLE,
            required=True,
        )
        config_types = sdk2.parameters.String('List of resources to build', multiline=True)
        with sdk2.parameters.Output:
            config_resources = sdk2.parameters.Resource(
                'Resulting configs',
                multiple=True,
                default=[]
            )

    class Context(sdk2.Task.Context):
        pass

    def __run_pva(self, pva_config, vertical, config_name, output):
        self.set_info(
            'Building config {} for vertical {} into {} with PVA'
            .format(config_name, vertical, output)
        )
        try:
            run_process([self.Context.pva_path, pva_config, vertical, config_name, output], log_prefix='pva')
        except Exception as e:
            raise errors.TaskError('Unexpected PVA error: {}', e)

    def on_execute(self):
        self.Context.svn_info = Arcadia.info(self.Parameters.arcadia_url)
        self.Context.revision = self.Context.svn_info['commit_revision']
        self.Context.pva_path = str(sdk2.ResourceData(self.Parameters.pva).path)
        sdk2.paths.make_folder('pva_conf_dir')
        self.Context.pva_conf_dir = os.path.abspath('pva_conf_dir')
        config_resources = []
        for config_type in self.Parameters.config_types.split():
            res_type = sdk2.Resource[config_type]
            sdk2.paths.make_folder(config_type)
            config_name = res_type.config_name
            config_dir = os.path.abspath(config_type)
            arc_build_url = Arcadia.trunk_url(
                res_type.arcadia_build_path,
                revision=self.Context.revision,
            )
            Arcadia.export(arc_build_url, config_dir)
            config_path = os.path.join(config_dir, config_name)
            pva_config = os.path.join(config_dir, 'config.json')
            self.__run_pva(pva_config, 'VERTICAL', config_name, config_path)
            config_resource = res_type(
                task=self,
                description='[{}] @revision={}'.format(config_name, self.Context.revision),
                path=config_path,
                arch='any'
            )
            config_resources.append(config_resource)
        self.Parameters.config_resources = config_resources
        self.set_info('All configs are built')

    def on_release(self, additional_params):
        nanny.ReleaseToNannyTask2.on_release(self, additional_params)
        sdk2.Task.on_release(self, additional_params)
