# -*- coding: utf-8 -*-

import copy
import logging
import os

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.websearch import release_setup
from sandbox.projects import resource_types
from sandbox.projects.common.search import config as sconf
import sandbox.projects.common.config_processor as cfgproc


class Params(object):
    class SampleConfig(parameters.ResourceSelector):
        name = 'sample_config_resource_id'
        description = 'Sample config'
        resource_type = resource_types.NOAPACHEUPPER_CONFIG
        required = True

    class UpdateConfigOptionsFrom(parameters.SandboxArcadiaUrlParameter):
        name = 'new_options_svn_url'
        description = 'Svn url for branch with current config options'
        default_value = 'arcadia:/arc/trunk'

    class RearrangeOptionsTypes(parameters.SandboxStringParameter):
        name = 'rearrange_options_types'
        description = 'Rearrange options types'
        default_value = 'KUBR WEB_KUBR'

    class CorruptGroupings(parameters.SandboxBoolParameter):
        name = 'corrupt_groupings'
        description = 'Add rearrange for corrupting groupings (ensure groupings contain empty groups/docs)'
        default_value = False

    class SetInvalidSubSources(parameters.SandboxStringParameter):
        name = 'invalid_subsources'
        description = (
            "Set for fraction subsources invalid hostnames (localhost:1) - "
            "input format: total_fractions:invalid_fraction_num"
        )
        default_value = ''

    class RemoveSources(parameters.SandboxStringParameter):
        name = 'remove_sources'
        description = 'Remove part subsources from config - input format: total_parts:removed_part_num'
        default_value = ''

    params = (
        SampleConfig,
        UpdateConfigOptionsFrom,
        RearrangeOptionsTypes,
        CorruptGroupings,
        SetInvalidSubSources,
        RemoveSources,
    )


class BuildNoapacheupperConfig(SandboxTask):
    """
        Создать более свежий конфигурационный файл на базе эталонного
        (заменить только опции переранжирований, взяв новые из указанного репозитория)
        таск используется для получения более свежей конфигурации переранжирований в конфигурационном файле,
        используемом при стрельбах по кешу.
    """

    type = "BUILD_NOAPACHEUPPER_CONFIG"

    execution_space = 500  # 500M

    client_tags = release_setup.WEBSEARCH_TAGS_P1

    input_parameters = Params.params

    cfg_res_id = "config_resource_id"

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        descr = channel.sandbox.get_resource(self.ctx[Params.SampleConfig.name]).description
        arcadia_url = self.ctx[Params.UpdateConfigOptionsFrom.name]
        if arcadia_url and arcadia_url != '-':
            descr += ' updated from {}'.format(arcadia_url)
        invalidate_fraction = self.ctx.get(Params.RemoveSources.name, '')
        if invalidate_fraction:
            descr += ' +remove part sources({})'.format(invalidate_fraction)
        invalidate_fraction = self.ctx.get(Params.SetInvalidSubSources.name, '')
        if invalidate_fraction:
            descr += ' +invalid sub sources({})'.format(invalidate_fraction)
        self.ctx[self.cfg_res_id] = self.create_resource(
            descr,
            'noapacheupper.cfg',
            resource_types.NOAPACHEUPPER_CONFIG,
            arch='any',
            attributes={
                "ttl": 90,
            },
        ).id

    def on_execute(self):
        configurator_dir = None
        arcadia_url = self.ctx[Params.UpdateConfigOptionsFrom.name]
        search_type = None
        types = ['WEB', 'IMGS', 'NEWS', 'MISC']
        for t in types:
            if t in self.ctx[Params.RearrangeOptionsTypes.name]:
                search_type = t
                break

        if arcadia_url and arcadia_url != '-':
            configurator_dir = self.abs_path('rearranges_conf')
            if not os.path.isdir(configurator_dir):
                os.mkdir(configurator_dir)
            svn_info = Arcadia.info(arcadia_url)
            cfg_dir = 'arcadia/search/web/rearrs_upper'
            svn_url = '{}/{}@{}'.format(svn_info['url'], cfg_dir, svn_info['entry_revision'])
            Arcadia.export(svn_url, configurator_dir)

        result_cfg_path = channel.sandbox.get_resource(self.ctx[self.cfg_res_id]).path

        if self.ctx[Params.SampleConfig.name]:
            sample_cfg = self.sync_resource(self.ctx[Params.SampleConfig.name])
            cfg = sconf.NoapacheupperConfig.get_config_from_file(sample_cfg)
            cfg.save_to_file(result_cfg_path)
            cfg.apply_local_patch({
                'Collection/ReArrangeOptions': '',
                'Collection/CompiledInOptions': (
                    'ReArrangeOptions(ResourceName=/ReArrangeOptions/{})'.format(search_type)
                ),
            })

            cfg.save_to_file(result_cfg_path)
        else:
            raise SandboxTaskFailureError('not support perl config generator now')

        # TODO(velavokr,mvel): reimplement CorruptGroupings

        invalidate_fraction = self.ctx.get(Params.RemoveSources.name, '')
        if invalidate_fraction:
            cfg = sconf.NoapacheupperConfig.get_config_from_file(result_cfg_path)
            total, frac = invalidate_fraction.split(':')
            cfg.text = cfgproc.remove_part_sources(cfg.text, int(total), int(frac))
            cfg.save_to_file(result_cfg_path)

        invalidate_fraction = self.ctx.get(Params.SetInvalidSubSources.name, '')
        if invalidate_fraction:
            cfg = sconf.NoapacheupperConfig.get_config_from_file(result_cfg_path)
            total, frac = invalidate_fraction.split(':')
            cfg.text = cfgproc.replace_subsources_for_part_sources(
                cfg.text, int(total), int(frac), 'http://localhost:2/ya?@100'
            )
            cfg.save_to_file(result_cfg_path)


def svn_subfolder_url(svn_url, subfolder):
    svn_info = Arcadia.info(svn_url)
    return '{}/{}@{}'.format(svn_info['url'], subfolder, svn_info['entry_revision'])


def _hr_hash(h):
    return ', '.join(['{}={}'.format(k, v) for k, v in h.iteritems()])


__Task__ = BuildNoapacheupperConfig


def create_noapache_config_builder_task(sample_config, descr, **kwargs):
    task_params = copy.deepcopy(kwargs)
    task_params[Params.SampleConfig.name] = sample_config
    logging.info('create %s (%s)', BuildNoapacheupperConfig.type, _hr_hash(task_params))
    return channel.task.create_subtask(
        task_type=BuildNoapacheupperConfig.type,
        description=descr,
        input_parameters=task_params,
        arch='linux',
    )
