# -*- coding: utf-8 -*-

import logging
import os
import re
import json
import shutil
import tarfile
from multiprocessing.pool import ThreadPool

from sandbox import common
import sandbox.common.types.client as ctc

from sandbox.common import errors
from sandbox.sandboxsdk import environments as pyenv
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common import context_managers
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.utils import sync_resource
from sandbox.projects.websearch.upper import resources as upper_resources

from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import rm_notify as rm_notify
from sandbox.projects.release_machine.components.configs.rtcc import RtccCfg
from sandbox.projects.common.sdk_compat import task_helper


class PatchParameter(sp.LastReleasedResource):
    name = 'patch_resource_id'
    description = 'Patch from Maestro'
    resource_type = resource_types.MAESTRO_CONTOURS_INFO
    required = False


class RawPatchParameter(sp.SandboxStringParameter):
    name = 'raw_patch_parameter'
    description = 'Raw patch (dumped json)'
    default_value = ''
    required = False


class UseRawPatch(sp.SandboxBoolParameter):
    name = 'use_raw_patch'
    description = 'Use raw patch instead of Resource'
    default_value = False
    required = False


class GenerationConfigTypeParameter(sp.SandboxStringParameter):
    name = 'config_type'
    description = '--config-type argument, for example "noapache,balancer,sfront" (skip by default)'
    default_value = ''
    required = False


class GenerationTimestampParameter(sp.SandboxStringParameter):
    name = 'generation_timestamp'
    description = 'Generation Timestamp'
    default_value = ''
    required = False


class BundleTypeParameter(sp.SandboxSelectParameter):
    name = 'bundle_type'
    description = 'Bundle type'
    required = True
    default_value = 'production'
    choices = [
        ('priemka', 'priemka'),
        ('production', 'production'),
        ('production noapache', 'production_noapache'),
    ]


class GenerationModeParameter(sp.SandboxBoolGroupParameter):
    name = 'generation_mode'
    description = 'Optional generation modes'
    choices = [
        ('Emergency cache', 'emergency_cache'),
        ('Skip tests', 'skip_tests')
    ]
    default_value = None


class EmergencyCacheParameter(sp.ResourceSelector):
    name = 'emergency_cache'
    description = 'RTCC_CACHE for usage as emergency fallback'
    resource_type = [
        resource_types.RTCC_CACHE,
        upper_resources.RtccCacheNoapache,
    ]


@rm_notify.notify2()
class BuildUpperConfig(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_UPPER_CONFIG'

    input_parameters = [
        GenerationConfigTypeParameter,
        PatchParameter,
        BundleTypeParameter,
        RawPatchParameter,
        UseRawPatch,
        GenerationModeParameter,
        EmergencyCacheParameter,
    ] + build_params.get_arcadia_params()

    # set default timeout for whole bundle build
    @common.utils.singleton_classproperty
    def DEFAULT_CONTEXT(self):
        return dict(SandboxTask.DEFAULT_CONTEXT, kill_timeout=60 * 180)

    client_tags = ctc.Tag.GENERIC
    execution_space = 70 * 1024

    CHECKOUT_PATH = 'gencfg.upper'
    TEST_RESULTS_TGZ_PATH = 'test.results.tar.gz'
    RTCC_BUNDLE_PATH = 'RTCC_BUNDLE'
    RTCC_LOGS_PATH = 'RTCC_LOGS'
    CACHE_PATH = 'CACHE'
    REPORTS_PATH = 'REPORTS'

    def on_enqueue(self):
        task_helper.ctx_field_set(self, rm_const.COMPONENT_CTX_KEY, RtccCfg.name)
        self.ctx.setdefault(GenerationTimestampParameter.name, self.timestamp)
        version = self.ctx[consts.ARCADIA_URL_KEY]
        version = version.replace('arcadia:/arc/', '')
        version = "{}/{}".format(version, self.ctx[GenerationTimestampParameter.name])
        self.ctx["full_version"] = version

        bundle_type = self.ctx[BundleTypeParameter.name]

        cache_resource_type = resource_types.RTCC_CACHE
        if bundle_type == 'production_noapache':
            cache_resource_type = upper_resources.RtccCacheNoapache

        self.create_resource(
            description='RTCC cache {0}'.format(version),
            resource_path=self.path(self.CACHE_PATH),
            resource_type=cache_resource_type,
            arch='any'
        )
        self.create_resource(
            description='RTCC reports {0}'.format(version),
            resource_path=self.path(self.REPORTS_PATH),
            resource_type=resource_types.RTCC_REPORTS,
            arch='any'
        )
        self.create_resource(
            description='RTCC build logs',
            resource_type=resource_types.RTCC_LOGS,
            arch='any',
            resource_path=self.path(self.RTCC_LOGS_PATH)
        )

        bundle_description = 'Runtime Cloud Config {}'.format(version)
        if bundle_type == 'priemka':
            bundle_resource_type = resource_types.RTCC_BUNDLE_PRIEMKA
        elif bundle_type == 'production':
            bundle_resource_type = resource_types.RTCC_BUNDLE
        elif bundle_type == 'production_noapache':
            config_type = self.ctx[GenerationConfigTypeParameter.name]
            if 'noapache' not in config_type:
                self.ctx[GenerationConfigTypeParameter.name] += '{}noapache'.format(',' if len(config_type) > 0 else '')
            bundle_resource_type = upper_resources.RtccBundleNoapache
            bundle_description = 'Noapache Config {}'.format(version)
        self.create_resource(
            description=bundle_description,
            resource_path=self.path(self.RTCC_BUNDLE_PATH),
            resource_type=bundle_resource_type,
            arch='any'
        )

    def get_arcadia_src_dir(self, ignore_externals=False, copy_trunk=False):
        """
            Copy-pasted from sandbox.projects.common.build.ArcadiaTask
            Получить репозиторий с аркадией.

            Сначала ищем директорию в кеше, если не нашли то делаем полный чекаут.
            Директории с кешом удаляются специльным таском раз в неделю.

            :param ignore_externals: использовать ли параметр svn-а --ignore_externals
        """
        try:
            if not sdk.fuse_available():
                raise errors.TaskFailure('Fuse unavailable')
            arcadia_src_dir = sdk.mount_arc_path(self.ctx[consts.ARCADIA_URL_KEY])
            return arcadia_src_dir
        except errors.TaskFailure as e:
            logging.exception(e)
            arcadia_src_dir = Arcadia.get_arcadia_src_dir(
                self.ctx[consts.ARCADIA_URL_KEY], copy_trunk=copy_trunk)
            eh.ensure(arcadia_src_dir, 'Cannot get repo for url {0}'.format(self.ctx[consts.ARCADIA_URL_KEY]))
            return context_managers.nullcontext(arcadia_src_dir)

    def path(self, *args):
        return SandboxTask.path(self, os.path.join(*args))

    def _prepare_generation_kwargs(self):
        generation_kwargs = dict()
        if self.ctx[GenerationModeParameter.name]:
            params = self.ctx[GenerationModeParameter.name].split(' ')
            if 'skip_tests' in params:
                generation_kwargs['skip_tests'] = True
            if 'emergency_cache' in params:
                eh.ensure(
                    self.ctx[EmergencyCacheParameter.name],
                    'Expected cache resource for emergency generation, got nothing'
                )
                generation_kwargs['emergency_cache_path'] = self.sync_resource(self.ctx[EmergencyCacheParameter.name])
        return generation_kwargs

    def on_execute(self):
        arcadia_ctx = self.get_arcadia_src_dir()
        with arcadia_ctx as arcadia_src_dir:
            Arcadia.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())

            self.ctx.setdefault(GenerationTimestampParameter.name, self.timestamp)
            self.descr = self.ctx["full_version"]
            self.venv = pyenv.VirtualEnvironment(use_system=True)
            if self.ctx[BundleTypeParameter.name] == 'priemka':
                # use priemka patch if it was specified explicitly
                priemka_patch_resource = self.ctx.get(PatchParameter.name)
                patch_path = sync_resource(priemka_patch_resource)
            elif self.ctx.get(UseRawPatch.name):
                patch_data = self.ctx.get(RawPatchParameter.name)
                patch_path = self.path('raw_rtcc_patch')
                with open(patch_path, 'w+') as pf:
                    json.dump(json.loads(patch_data), pf)
            else:
                patch_path = None

            generation_kwargs = self._prepare_generation_kwargs()

            with self.venv:
                pyenv.PipEnvironment('pip', version="9.0.1", venv=self.venv).prepare()
                pyenv.PipEnvironment('setuptools', version="39.2.0", venv=self.venv).prepare()
                pyenv.PipEnvironment('setuptools-scm', version="5.0.2", venv=self.venv).prepare()

                rtcc_wrapper = RtccWrapper(
                    self.venv,
                    self.path(self.CHECKOUT_PATH),
                    arcadia_src_dir
                )

                # get list of config generators names from parameter GenerationConfigTypeParameter if it was specified
                # if GenerationConfigTypeParameter was not specified, default generators will be used
                cfg_types_parameter_value = self.ctx.get(GenerationConfigTypeParameter.name, '')
                cfg_types = None if not cfg_types_parameter_value else cfg_types_parameter_value.split(',')
                environ = self.__build_environ(
                    GENCFG_VERSION=self.ctx.get("full_version", ""),
                    SANDBOX_TASK=str(self.id),
                )
                generation_res = rtcc_wrapper.generate(cfg_types, environ, patch_path, **generation_kwargs)
                self.ctx["reports"] = self.get_reports(rtcc_wrapper.reports)
                test_wrapper = TestsWrapper(self.venv, self.log_path())
                tests_res = test_wrapper.test(rtcc_wrapper.bundle)
                if not generation_res:
                    test_wrapper.archive_result()

                self.__copytree(rtcc_wrapper.cache, self.path(self.CACHE_PATH, "cache"))
                self.__copytree(rtcc_wrapper.logs, self.path(self.RTCC_LOGS_PATH))
                self.__copytree(rtcc_wrapper.reports, self.path(self.REPORTS_PATH))
                self.__copytree(rtcc_wrapper.bundle, self.path(self.RTCC_BUNDLE_PATH))
                self.__copyfile(test_wrapper.archive, self.path(self.TEST_RESULTS_TGZ_PATH))
                self.__copyfile(test_wrapper.archive, self.path(self.REPORTS_PATH, self.TEST_RESULTS_TGZ_PATH))
                eh.ensure(not generation_res, "Generation was not succesfull. ")
                if tests_res:
                    # disable for dead host vla1-1555 DNS resolve
                    logging.error("Tests were not succesfull, see logs above for details. ")
                # eh.ensure(not tests_res, "Tests were not succesfull, see logs for details. ")

    def get_reports(self, path):
        logging.info(path)
        logging.info(filter(lambda x: x.endswith("errors.warns.txt"), os.listdir(path)))
        return {file: open(os.path.join(path, file)).read()
                for file in filter(lambda x: x.endswith("errors.warns.txt"), os.listdir(path))}

    @property
    def release_template(self):
        tag_url = self.ctx.get(consts.ARCADIA_URL_KEY)
        return self.ReleaseTemplate(
            subject='{0}: {1}'.format('rtcc_bundle', utils.branch_tag_from_svn_url(tag_url)[1])
        )

    @property
    def footer(self):
        return [
            {
                'helperName': '',
                'content': [
                    {
                        file: self.ctx.get("reports", [])[file],
                    }
                ],
            } for file in self.ctx.get("reports", [])
        ]

    def __build_environ(self, **kwargs):
        environ = os.environ.copy()
        environ.update(kwargs)
        return environ

    def __copytree(self, dir_src, dir_dst):
        if os.path.exists(dir_src):
            shutil.copytree(dir_src, dir_dst)
        else:
            os.makedirs(dir_dst)
        if not os.listdir(dir_dst):
            open(os.path.join(dir_dst, ".info"), "w+").close()

    def __copyfile(self, file_src, file_dst):
        if os.path.exists(file_src):
            shutil.copyfile(file_src, file_dst)
        else:
            open(file_dst, "w+").close()


__Task__ = BuildUpperConfig


class TestsWrapper(object):
    THREADS_COUNT = 10
    TESTS_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/search/garden/runtime_tests'
    TESTS_CHECKOUT_PATH = 'runtime_tests'
    TEST_RESULTS_TAR_GZ = "test.results.tar.gz"

    def __init__(self, venv, cwd):
        self.venv = venv
        self.cwd = cwd
        self.__checkout()

    def __checkout(self):
        Arcadia.export(self.TESTS_URL, self.TESTS_CHECKOUT_PATH)

    def test(self, path):
        pool = ThreadPool(self.THREADS_COUNT)
        result = pool.map(self.__test_single, self.__list_configs(path))
        logging.info("test results: {}".format(str(result)))
        return any(result)

    def __list_configs(self, path):
        result = []
        for path, dirs, files in os.walk(path):
            result.extend([os.path.join(path, _file) for _file in files])
        for _file in result:
            if _file.endswith("apache.ywsearch.cfg"):
                yield "noapache", _file, ".".join(_file.split(os.path.sep)[-4:-1])
            if _file.endswith("request.json"):
                yield "request", _file, ".".join(_file.split(os.path.sep)[-4:-1])

    def path(self, *args):
        return os.path.join(self.cwd, *args)

    def __test_single(self, args):
        config_type, config_path, config_id = args
        return_code = process.run_process(
            "py.test configuration/tests/test_{config_type}_config.py"
            " --config='{generated}' --junit-xml {report} -m 'not dynamic and not depricated'".format(
                config_type=config_type,
                generated=config_path,
                report=self.path(
                    'result.{config_type}.{cfg}.xml'.format(cfg=config_id, config_type=config_type))),
            work_dir=self.TESTS_CHECKOUT_PATH,
            log_prefix="tests.results.{}".format(config_id),
            check=False,
            shell=True).returncode
        if return_code:
            logging.error("Test FAILED (type %s, path %s, id %s)", config_type, config_path, config_id)
        return return_code

    @property
    def archive(self):
        return self.path(self.TEST_RESULTS_TAR_GZ)

    def archive_result(self):
        mask = "result.*.xml"
        with tarfile.open(self.archive, 'w:gz') as tar:
            for entry in os.listdir(self.path()):
                if re.match(mask, entry):
                    tar.add(self.path(entry), entry)


class RtccWrapper(object):
    CACHE_NAME = "Generation"
    ARCADIA_PATH = 'search/priemka/gencfg.upper'

    def __init__(self, venv, cwd, arcadia_src_dir):
        self.venv = venv
        self.cwd = cwd
        self.__arcadia_checkout(arcadia_src_dir)
        self.__install()

    def __install(self):
        req_specs = " ".join([line.strip() for line in open(self.path("requirements.txt")).readlines()])
        try:
            self.venv.pip(req_specs)
        except pyenv.PipError:
            self.venv.pip(req_specs)

    def __arcadia_checkout(self, arcadia_src_dir):
        shutil.copytree(os.path.join(arcadia_src_dir, self.ARCADIA_PATH), self.cwd)

    def path(self, *args):
        return os.path.join(self.cwd, *args)

    @property
    def cache(self):
        return self.path(self.CACHE_NAME, "cache")

    @property
    def logs(self):
        return self.path("rtcc", "logs")

    @property
    def bundle(self):
        return self.path(self.CACHE_NAME, "bundle")

    @property
    def reports(self):
        return self.path(self.CACHE_NAME, "reports")

    def generate(self, cfg_types=None, environment=None, patch=None, **kwargs):
        """
        To skip filtering bool(cfg_types) should be false
        :param cfg_types:
        :param environment:
        :param patch:
        :return:
        """
        environment = environment or os.environ.copy()
        cfg_types = cfg_types or []
        emergency_cache_path = kwargs.get('emergency_cache_path', None)
        skip_tests = kwargs.get('skip_tests', None)
        logging.info('Start generation: cfg_types={}, environment={}'.format(cfg_types, environment))
        cfg_types_args = ' '.join(['--config-type={}'.format(cfg_type) for cfg_type in cfg_types])
        patch_args = '--patch-file={}'.format(patch) if patch else ''
        emergency_cache_args = '--emergency-cache={}'.format(emergency_cache_path) if emergency_cache_path else ''
        skip_tests_args = '--skip-test-results=1' if skip_tests else ''
        result = process.run_process(
            '{executable} ./rtcc.py generate --generation-type=complete --cache-name={cache_name} {args}'.format(
                executable=self.venv.executable,
                cache_name=self.CACHE_NAME,
                args=' '.join([cfg_types_args, patch_args, emergency_cache_args, skip_tests_args]).strip()
            ),
            check=False,
            environment=environment,
            work_dir=self.cwd,
            log_prefix="request.generation",
            shell=True).returncode
        logging.info("generation result: {}".format(str(result)))
        return result
