# coding: utf-8

import itertools
import logging
import os
import time

from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.common import errors as se
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects import resource_types
from sandbox.projects.PatchQueries import AddCgiParameter
from sandbox.projects.release_machine.tasks.ScrapeRequests2 import static
from sandbox.projects.common import environments
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import gsid_parser
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import network
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common import utils
from sandbox.projects.common.noapacheupper import search_component as nsc
from sandbox.projects.common.search import bugbanner
from sandbox.projects.common.search.eventlog import eventlog as el
from sandbox.projects.common.search.response import cgi
from sandbox.projects.release_machine import rm_notify
from sandbox.projects.release_machine import events as rm_events
from sandbox.projects.release_machine import input_params as rm_input
from sandbox.projects.release_machine.components.configs.ab_experiments import AbExperimentsCfg
from sandbox.projects.release_machine.components.configs.rearrange_dynamic import RearrangeDynamicCfg
from sandbox.projects.release_machine.components.configs.upper import UpperCfg
from sandbox.projects.release_machine.core import const as RMconst
from sandbox.projects.release_machine.tasks.ScrapeRequests2.scraper_over_yt import ScraperOverYtWrapper, BatchStatus
from sandbox.projects.release_machine.helpers import soy_helper
from sandbox.projects.websearch.upper import resources as upper_resources


HAMSTER = "upper-hamster.hamster.yandex.ru"
N_REQS = 300
_FAIL_COUNT_KEY = "config_get_fail_count"


class Binary(sp.ResourceSelector):
    """For debug purposes! Leave empty if not sure what is it"""
    name = 'custom_noapacheupper_executable_resource_id'
    description = 'Custom noapacheupper executable'
    resource_type = upper_resources.NoapacheUpper
    required = False


class RearrangeDynamic(sp.ResourceSelector):
    """For debug purposes! Leave empty if not sure what is it"""
    name = 'custom_rearrange_dynamic_resource_id'
    description = 'Custom rearrange dynamic'
    resource_type = resource_types.REARRANGE_DYNAMIC_DATA
    required = False


class RearrangeData(sp.ResourceSelector):
    """For debug purposes! Leave empty if not sure what is it"""
    name = 'custom_rearrange_data_resource_id'
    description = 'Custom rearrange data'
    resource_type = resource_types.REARRANGE_DATA
    required = False


class RearrangeDataFast(sp.ResourceSelector):
    """For debug purposes! Leave empty if not sure what is it"""
    name = 'custom_rearrange_data_fast_resource_id'
    description = 'Custom rearrange data fast'
    resource_type = upper_resources.RearrangeDataFast
    required = False


class ScraperMode(sp.SandboxBoolParameter):
    name = "scraper_mode"
    description = "Scraper mode"
    default_value = True


class ScraperYtServer(sp.SandboxStringParameter):
    name = "scraper_yt_server"
    description = "YT server for SoY"
    default_value = "hahn"


class ScraperPool(sp.SandboxStringParameter):
    name = "scraper_pool"
    description = "Scraper pool name"
    default_value = "default"


ScraperMode.sub_fields = {"true": [ScraperPool.name]}


class Engine(sp.SandboxStringParameter):
    name = "engine"
    description = "Engine"
    choices = (
        ("WEB", "WEB"),
        ("TOUCH", "TOUCH"),
        ("TAB", "TAB"),
        ("VIDEO", "VIDEO"),
        ("IMAGES", "IMAGES"),
        ("PEOPLE", "PEOPLE"),
    )
    default_value = "WEB"


@rm_notify.notify2()
class CheckNoapacheExperiment(bugbanner.BugBannerTask):
    """
        Протестировать экспериментальные параметры на noapache.
        SEARCH-3403
    """
    type = "CHECK_NOAPACHEUPPER_EXPERIMENT"
    client_tags = ctc.Tag.GENERIC
    input_parameters = [
        rm_input.ComponentName,
        AddCgiParameter,
        Binary,
        RearrangeDynamic,
        RearrangeData,
        RearrangeDataFast,
        ScraperMode,
        ScraperYtServer,
        ScraperPool,
        Engine,
    ]
    execution_space = 70 * 1024  # 70 Gb
    required_ram = 80 * 1024  # 80 Gb

    environment = [
        environments.PipEnvironment('yandex-yt', use_wheel=True),
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True)
    ]

    def initCtx(self):
        self.ctx['kill_timeout'] = 6 * 60 * 60  # 6h (SEARCH-5854)

    def on_execute(self):
        if not self.ctx.get(RMconst.COMPONENT_CTX_KEY):
            self.add_bugbanner(bugbanner.Banners.NoapacheUpper)
            self.ctx[RMconst.COMPONENT_CTX_KEY] = UpperCfg.name
        elif self.ctx[RMconst.COMPONENT_CTX_KEY] == RearrangeDynamicCfg.name:
            self.add_bugbanner(bugbanner.Banners.NoapacheUpper)
        elif self.ctx[RMconst.COMPONENT_CTX_KEY] == AbExperimentsCfg.name:
            gsid = self.ctx.get("__GSID")
            revision = gsid_parser.get_svn_revision_from_gsid(gsid)
            job_name = gsid_parser.get_job_name_from_gsid(gsid)
            if revision and job_name:
                self.ctx["rm_event_type"] = rm_events.EventType.AcceptanceTest
                self.ctx["rm_event_data"] = {
                    "revision": revision,
                    "job_name": job_name,
                    "scope_number": revision,
                    "acceptance_type": "UNKNOWN",
                    "acceptance_result": "",
                }
            if not self.ctx.get(AddCgiParameter.name):
                self.set_info("No cgi param specified. Nothing to check")
                return
        upper_cfg_path = self.prepare_resources()
        random_users_queries = utils.get_and_check_last_resource_with_attribute(
            resource_type=resource_types.USERS_QUERIES,
            attr_name="random_requests",
            attr_value="2000"
        )
        bad_queries_path = Arcadia.export(
            Arcadia.trunk_url('sandbox/projects/websearch/upper/CheckNoapacheExperiment/bad_queries.txt'),
            'bad_queries.txt'
        )
        with open(bad_queries_path) as f:
            bad_queries = f.read().strip('\n')
        all_queries = itertools.chain(  # iterator throw (text, region) pairs
            (req[0: 2] for req in itertools.islice(utils.get_user_queries(random_users_queries), N_REQS)),
            (line.split('/t') for line in bad_queries.split('\n')),
        )
        upper = nsc.get_noapacheupper(config_file=upper_cfg_path)
        with upper:
            if utils.get_or_default(self.ctx, ScraperMode):
                self.check_apphost_mode(all_queries, upper)
            else:
                self.check_cgi_mode(all_queries, upper)
        evlog_path = upper.get_event_log_path()
        logging.info("Size of upper eventlog = %s bytes", os.path.getsize(evlog_path))
        with open(evlog_path, 'rb') as inp:
            eventlog_path = el.get_evlogdump(self.ctx[nsc.Params.Binary.name])
            process.run_process(
                [eventlog_path, '-i', 'SubSourceRequest'],
                stdin=inp,
                log_prefix="parse_evlog",
                wait=True,
                check=True,
                outputs_to_one_file=False,
            )

    def check_apphost_mode(self, queries, upper):
        os.environ["YT_TOKEN"] = self.get_vault_data("SEARCH-RELEASERS", "yt_token")
        os.environ["SOY_TOKEN"] = sdk2.Vault.data("EXPERIMENTS", "soy_token")

        requests = self.prepare_requests(queries, upper)
        job_id = self.ctx['soy_id'] = 'check-noapache-exp-{}'.format(self.id)
        channel.channel.sandbox.set_task_context_value(self.id, 'soy_id', job_id)
        input_table = '//tmp/check_noapache_experiment/{}'.format(job_id)
        ScraperOverYtWrapper.launch_scraper(
            requests,
            server=self.ctx[ScraperYtServer.name],
            input_table=input_table,
            guid=job_id,
            pool=self.ctx[ScraperPool.name],
            use_api=True,
        )

        self.set_info('Launched SoY batch: {}'.format(job_id))

        start_time = time.time()
        sleep_time = 60  # seconds
        checks_count = 0
        while True:
            try:
                batch_status, _ = ScraperOverYtWrapper.get_batch_status_via_api(
                    self.ctx[ScraperYtServer.name], job_id
                )
                logging.info('%s seconds of active waiting, got status %s', time.time() - start_time, batch_status)
                if (checks_count * sleep_time) % 600 == 0:  # print status every 10 minutes
                    self.set_info('Current status: {status}... ({soy_api}status?id={job_id})'.format(
                        status=BatchStatus.STATUS_NAME[batch_status],
                        soy_api=soy_helper.SOY_API.format(self.ctx[ScraperYtServer.name]),
                        job_id=job_id,
                    ))
                checks_count += 1
                eh.verify(batch_status != BatchStatus.FAILED, "Soy batch failed to download")
                if batch_status == BatchStatus.COMPLETED:
                    return
            except se.TaskError as e:
                raise e
            except Exception as e:
                logging.exception(e)
            time.sleep(sleep_time)

    def prepare_requests(self, queries, upper):
        return ScraperOverYtWrapper.get_queries(
            queries_list=queries,
            host=HAMSTER,
            platform='search',
            cgi=self.get_additional_cgi_params(upper),
            id_prefix='id',
            region_prefix='reg',
            uids=None,
        )

    def check_cgi_mode(self, queries, upper):
        """Old and deprecated. Replace to apphost mode after transition period"""
        common_url = cgi.UrlCgiCustomizer(
            base_url=HAMSTER + "yandsearch",
            params=self.ctx.get(AddCgiParameter.name) or None
        )
        common_url.add_custom_param("srcrwr", self._build_srcrwr(upper))
        for text, reg in queries:
            iter_url = cgi.UrlCgiCustomizer(common_url.base_url, common_url.params)
            iter_url.add_text(text).add_region(reg)
            logging.debug("Url: %s", str(iter_url))
            requests_wrapper.get_r(iter_url.base_url, params=iter_url.params, verify=False)

    def prepare_resources(self):
        msg = None
        upper_cfg = None
        try:
            too_many_reqs = 429
            upper_cfg = requests_wrapper.get_r(
                "https://{}/viewconfig".format(HAMSTER),
                no_retry_on_http_codes=[too_many_reqs],
                timeout=5
            )
        except Exception as e:
            # UPS-141
            msg = str(e)

        # SEARCH-7054
        if upper_cfg is not None:
            if int(upper_cfg.status_code) == too_many_reqs:
                msg = "Hamster is busy"
            elif "Server" not in upper_cfg.text:
                msg = "'Server' section not found in config"
            elif "Collection" not in upper_cfg.text:
                msg = "'Collection' section not found in config"
            # else OK

        if msg is not None:
            fail_count = int(self.ctx.get(_FAIL_COUNT_KEY, 0))
            fail_count += 1
            self.ctx[_FAIL_COUNT_KEY] = fail_count
            eh.ensure(fail_count <= 5, "Cannot get config 5 times, giving up")
            self.set_info(
                "Cannot download correct upper config from {}, let us wait for a while (300s). Reason: {}".format(
                    HAMSTER, msg
                )
            )
            # we've downloaded something strange
            self.wait_time(300)

        logging.debug("Config found:\n%s", upper_cfg.text)
        upper_cfg_path = self.abs_path("hamster_upper_cfg.txt")
        fu.write_file(upper_cfg_path, upper_cfg.text)
        self._get_binary_id()
        self._get_rearr_data_id()
        self._get_rd_id()
        self._get_rearr_data_fast_id()
        self.ctx[nsc.Params.AppHostMode.name] = True
        return upper_cfg_path

    def _get_resource_id(self, resource_class, custom_resource_class, resource_title):
        resource_id = utils.get_or_default(self.ctx, resource_class)
        if not resource_id:
            resource_id = utils.get_and_check_last_released_resource_id(
                custom_resource_class.resource_type,
                order_by="-release__creation_time"
            )
            self.set_info("{} found: {}".format(resource_title, lb.resource_link(resource_id)), do_escape=False)
        self.ctx[custom_resource_class.name] = resource_id

    def _get_rearr_data_id(self):
        self._get_resource_id(RearrangeData, nsc.Params.RearrangeData, "RearrangeData")

    def _get_rd_id(self):
        self._get_resource_id(RearrangeDynamic, nsc.Params.RearrangeDynamicData, "RearrangeDynamicData")

    def _get_rearr_data_fast_id(self):
        self._get_resource_id(RearrangeDataFast, nsc.Params.RearrangeDataFast, "RearrangeDataFast")

    def _get_binary_id(self):
        self._get_resource_id(Binary, nsc.Params.Binary, "Binary")

    @staticmethod
    def _build_srcrwr(upper):
        return 'UPPER:[{}]:{}:5000'.format(network.get_my_ipv6(), upper.get_port())

    def get_additional_cgi_params(self, upper):
        return "&{}&srcrwr={}&reqinfo=check_noapache_exp_{}".format(
            self.ctx.get(AddCgiParameter.name), self._build_srcrwr(upper), self.id
        )

    def _get_req_modifier(self, upper):
        add_cgi = static.convert_cgi_to_json(self.get_additional_cgi_params(upper))

        def req_modifier(req):
            req["per-set-parameters"]["additional-cgi"] = add_cgi

        return req_modifier

    def on_success(self):
        bugbanner.BugBannerTask.on_success(self)

    def on_break(self):
        self._abort_batch()
        bugbanner.BugBannerTask.on_break(self)

    def on_terminate(self):
        self._abort_batch()
        bugbanner.BugBannerTask.on_terminate(self)

    def on_failure(self):
        self._abort_batch()
        bugbanner.BugBannerTask.on_failure(self)

    def on_timeout(self):
        self._abort_batch()
        bugbanner.BugBannerTask.on_timeout(self)

    def _abort_batch(self):
        job_id = self.ctx.get("soy_id")
        if not job_id:
            return

        logging.info("Task broken, aborting soy batch with id %s", job_id)
        soy_api = soy_helper.SoyApi(
            token=sdk2.Vault.data("EXPERIMENTS", "soy_token"),
            server=self.ctx[ScraperYtServer.name]
        )
        res = soy_api.abort(job_id)
        logging.info("Abortion status: {}".format(res.get("status")))
        if "error_msg" in res:
            logging.error(res["error_msg"])


__Task__ = CheckNoapacheExperiment
