# -*- coding: utf-8 -*-

import logging
import socket

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common.noapacheupper.rearranges import get_rearranges
from sandbox.projects.common.noapacheupper.search_component import create_noapacheupper_params
from sandbox.projects.common.noapacheupper.search_component import get_noapacheupper
from sandbox.projects.common.search.requester import format_users_queries_line
from sandbox.projects.common.search.requester import Params as RequesterParams
from sandbox.projects.common.search.requester import Requester


class Params:
    class ReportUrl(sp.SandboxStringParameter):
        name = 'report_url'
        description = 'Report url (default: https://hamster.yandex.ru/search?)'
        default_value = 'https://hamster.yandex.ru/search?'

    class UsersQueries(sp.LastReleasedResource):
        name = 'users_queries_resource_id'
        description = 'Users queries'
        resource_type = 'USERS_QUERIES'
        default_value = 0

    lst = (
        ReportUrl,
        UsersQueries,
    )


class ReportRequester(Requester):
    # наследуемся от requester, чтобы направлять запросы не прямо в используемый компонент, а через report
    def __init__(self, report_url):
        super(ReportRequester, self).__init__()
        self.report_url = report_url
        # получаем наш хост (для использования в параметре srcrwr)
        self.hostname = socket.gethostname()
        logging.info('redirect requests to UPPER:' + self.hostname)
        self.re_try_limit = 6

    def build_request(self, req):
        return self.report_url + format_users_queries_line(req) + \
            "&srcrwr=UPPER:{}:{}".format(self.hostname, self.search_component.port)


def queries_iterator(fname, rearrs):
    with open(fname) as f:
        for line in f:
            for rearr in rearrs:
                tk = line.rstrip('\n').split('\t')
                while len(tk) < 3:
                    tk.append('')
                tk[2] += '&rearr={}_off'.format(rearr)
                yield '\t'.join(tk)


class CombineNoapacheupperRearranges(SandboxTask, object):
    """
        **Описание**
         Проверяем, что noapache, взятый с указанной беты не падает при различных комбинациях
         отключения переранжирований
        (перебираем переранжирования, выключая только одно переранжирование)
    """

    type = 'COMBINE_NOAPACHEUPPER_REARRANGES'
    input_parameters = create_noapacheupper_params().params + RequesterParams.lst + Params.lst
    execution_space = 6 * 1024  # 1 Gb, max usage is near 4 Gb

    def __init__(self, task_id=0):
        super(CombineNoapacheupperRearranges, self).__init__(task_id)
        self.ctx['kill_timeout'] = 4 * 60 * 60

    def on_execute(self):
        queries_path = channel.task.sync_resource(self.ctx[Params.UsersQueries.name])
        search_component = get_noapacheupper()
        search_component.start()
        search_component.wait()
        # получаем список доступных переранжиований
        rearrs = get_rearranges('localhost:{}'.format(search_component.port))
        search_component.stop()

        rr = ReportRequester(self.ctx[Params.ReportUrl.name])
        rr.use(
            queries_iterator(queries_path, rearrs),
            search_component,
            ctx=self.ctx,
        )

        if rr.error:
            raise SandboxTaskFailureError(rr.error)


__Task__ = CombineNoapacheupperRearranges
