# -*- coding: utf-8 -*-

import json

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types


class Params:
    class Timings1(sp.LastReleasedResource):
        name = 'timings1_resource_id'
        description = 'Timings 1'
        resource_type = 'NOAPACHEUPPER_REQUEST_TIMINGS'

    class Timings2(sp.LastReleasedResource):
        name = 'timings2_resource_id'
        description = 'Timings 2'
        resource_type = 'NOAPACHEUPPER_REQUEST_TIMINGS'

    params = (
        Timings1,
        Timings2,
    )


class CompareNoapacheupperRequestTimings(SandboxTask):
    type = 'COMPARE_NOAPACHEUPPER_REQUEST_TIMINGS'
    input_parameters = Params.params

    out_resource_id = 'out_resource_id'

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'compare_result.html', resource_types.NOAPACHEUPPER_REQUEST_TIMINGS_COMPARE_RESULT)
        self.ctx[self.out_resource_id] = resource.id

    def on_execute(self):
        tm1_path = self.sync_resource(self.ctx[Params.Timings1.name])
        tm2_path = self.sync_resource(self.ctx[Params.Timings2.name])

        with open(tm1_path) as f:
            tm1 = json.load(f)
        with open(tm2_path) as f:
            tm2 = json.load(f)

        rearr_tm1 = tm1['rearrange_timings']
        rearr_tm2 = tm2['rearrange_timings']
        result = {}

        class result_value:
            def __init__(self, val1, val2, diff_limit):
                self.val1 = float(val1)
                self.val2 = float(val2)
                self.diff = float(val2 - val1)
                self.diff_perc = (100. * self.diff / self.val1) if val1 else None

            def serialize_tr(self, name):
                color = ''
                if self.diff_perc is None:
                    if self.diff > 0:
                        color = 'red'
                    elif self.diff < 0:
                        color = 'green'
                else:
                    if self.diff_perc > sum_tm_diff_limit:
                        color = 'red'
                    elif self.diff_perc < -sum_tm_diff_limit:
                        color = 'green'
                if color:
                    color = ' style="color:{}"'.format(color)
                return (
                    '<tr>'
                    '<td>{}</td>'
                    '<td align="right">{:.0f}</td>'
                    '<td align="right">{:.0f}</td>'
                    '<td align="right">{:.0f}</td>'
                    '<td align="right"{}>{:.2f}</td>'
                    '</tr>'
                ).format(
                    name,
                    self.val1,
                    self.val2,
                    self.diff,
                    color,
                    self.diff_perc if self.diff_perc is not None else 0.,
                )
        has_diff = False
        diff = ''
        sum_tm_diff_limit = 3.  # limit for diff summary rearrange timings
        sum_tm1 = 0
        for k, v in rearr_tm1.iteritems():
            sum_tm1 += v
            if k in rearr_tm2:
                result[k] = result_value(v, rearr_tm2[k], sum_tm_diff_limit)
            else:
                result[k] = result_value(v, 0, sum_tm_diff_limit)
        sum_tm2 = 0
        for k, v in rearr_tm2.iteritems():
            sum_tm2 += v
            if k not in rearr_tm1:
                result[k] = result_value(0, v, sum_tm_diff_limit)
                if float(v) > 1000:  # detect new rearrange consumed more than 1millisec, count it as diff
                    diff += 'New slow rearrange detected: {} consume {} microsec.\n'.format(k, v)
                    has_diff = True
        if sum_tm1 != 0:
            sum_tm_diff_percent = (100. * (sum_tm2 - sum_tm1)) / sum_tm1
        elif sum_tm2 == 0:
            sum_tm_diff_percent = 0.
        else:
            sum_tm_diff_percent = 100. if sum_tm2 > sum_tm1 else -100.
        if abs(sum_tm_diff_percent) > sum_tm_diff_limit and abs(sum_tm2 - sum_tm1) > 1500:
            diff += 'Total rearrange timings diff={:.2f}% exceeds limit={}%\n'.format(sum_tm_diff_percent, sum_tm_diff_limit)
            has_diff = True
        avg_response_diff_percent = (100. * (tm1['avg_resp_time'] - tm2['avg_resp_time'])) / tm1['avg_resp_time']
        avg_response_diff_limit = 1.5
        if abs(avg_response_diff_percent) > avg_response_diff_limit:
            diff += 'Average response time diff={:.2f}% exceed limit={}%\n'.format(avg_response_diff_percent, avg_response_diff_limit)
            has_diff = True
        if not has_diff:
            if not diff and abs(avg_response_diff_percent) > 0.01 or abs(sum_tm_diff_percent) > 0.01:
                diff = 'not significant diff'
            else:
                diff = 'no diff'
        result['_SUMMARY'] = result_value(sum_tm1, sum_tm2, avg_response_diff_limit)
        with open(channel.sandbox.get_resource(self.ctx[self.out_resource_id]).path, 'w') as f:
            f.write("<html>\n")
            f.write("""<head><style>
table {
    border-collapse: collapse;
}
table, td, th {
    border: 1px solid black;
    padding: 3px;
}
</style></head>
""")
            f.write("<body>\n")
            f.write("<h3>Common response timing (milliseconds)</h3>\n")
            resp_time = result_value(tm1['avg_resp_time'], tm2['avg_resp_time'], avg_response_diff_limit)
            _serialize_results({'avg_resp_time': resp_time}, f)
            f.write("<h3>Rearrange timings (microseconds)</h3>\n")
            _serialize_results(result, f)
            f.write("</body></html>\n")
        self.ctx['diff'] = diff
        self.ctx['has_diff'] = has_diff

    def get_results(self):
        if not self.is_completed():
            return 'Results are not ready yet.'

        return self.ctx['diff']

    def get_short_task_result(self):
        if not self.is_completed():
            return None

        return self.ctx['diff']


def _serialize_results(results, f):
    f.write('<table>\n')
    f.write('<tr><th></td><th>value 1</th><th>value 2</th><th>diff</th><th>% diff</th></tr>')
    trs = []
    for k, r in results.iteritems():
        trs.append(r.serialize_tr(k))
    trs.sort()
    for tr in trs:
        f.write(tr)
        f.write('\n')
    f.write("</table>\n")


__Task__ = CompareNoapacheupperRequestTimings
