#!/usr/bin/python
# -*- coding: utf-8 -*-

import copy
import json
import sys
import urllib2


cur_id = 0


def next_id():
    global cur_id
    cur_id += 1
    return '44eb086f-13ec-135c-2957-469a8e952507-' + str(cur_id)


total_width = 6


def percentile_for(signal, scale=1):
    common_perc = [
    ["hperc({},0,{})", "#003366", "[0;150ms]", 150],
    ["hperc({},0,{})", "#3333cc", "[0;300ms]", 300],
    ["hperc({},0,{})", "#cc3399", "[0;500ms]", 500],
    ["hperc({},0,{})", "#ff3399", "[0;600ms]", 600],
    ["hperc({},0,{})", "#ff3300", "[0;1sec]", 1000],
]

    def set_signal(line):
        line2 = copy.deepcopy(line)
        line2[0] = line2[0].format(signal, line2[-1]*scale)
        return line2[:-1]

    return [set_signal(line) for line in common_perc]


QUANTILE_95 = (
    ("quant(unistat-all_dhhh, 95)", "#00ff00", "resp.time quantile(.95)"),
)

APPHOST_QUANTILE_95 = (
    ("div(quant(apphost_unistat-SELF-SS-ResponseTimesMcsHistogram_dhhh, 95), 1000)", "#00ff00", "A/H resp.time quantile(.95)"),
)

RESP_PERC = dict(
    signals=percentile_for('unistat-time_all-humans_dhhh'),
    title_suffix='resp. time perc.',
    patch={
        "yAxis": [{
            "maxValue": 100,
            "minValue": 0
        }]
    },
    height=2,
)

AH_RESP_PERC = dict(
    signals=percentile_for('apphost_unistat-SELF-SS-ResponseTimesMcsHistogram_dhhh', scale=1000),
    title_suffix='A/H resp. time perc.',
    patch={
        "yAxis": [{
            "maxValue": 100,
            "minValue": 0
        }]
    },
    height=2,
)

CHART_CRASHES = (
    ("messages-crashes_sig6", "#ff3300", "SIGABRT"),
    ("messages-crashes_sig9", "#2020ff", "SIGKILL"),
    ("messages-crashes_sig11", "#FF00FF", "SIGSEGV"),
    ("messages-crashes_sig15", "#00ff00", "SIGTERM"),
)

PAGE_FAULTS = (
    ("portoinst-major_page_faults_summ", "#ff0000", "major page faults"),
)

RESP_COMPLETE = (
    ("div(unistat-unanswer_dmmm,unistat-unanswer-denom_dmmm)", "#000099", "resp. incomplete"),
)

TOTAL = (
    ("unistat-all-reports_dmmm", "#4d4d4d", "total reqs"),
)

APPHOST_PERC = (
    ("perc(apphost_unistat-SELF-SS-Requests_dmmm, unistat-all-reports_dmmm)", "#00ff00", "A/H %"),
)

OVERLOAD = (
    ("unistat-error503_dmmm", "#ffcc00", "503 (overload)"),
)


class Graphic:
    def __init__(self, title, stacked=False):
        self.signals = []
        self.consts = []
        self.graph = {
            "id": next_id(),
            "type": "graphic",
            "signals": self.signals,
            "consts": self.consts,
            "title": title,
            "stacked": stacked
        }

    def set_grid_cell(self, col, row, width=2, height=2):
        self.graph.update(dict(col=col, row=row, width=width, height=height))

    def set_stacked(self, val):
        self.graph['stacked'] = val

    def add_signals(self, geo, lst, tag, tier):
        for cht in lst:
            self.add_signal(
                tag=tag,
                signal=cht[0],
                color=cht[1],
                title=cht[2],
                geo=geo,
                tier=tier,
            )

    def add_signal(self, tag, signal, color, title, geo=None, tier=None):
        if geo:
            tag += ';geo=' + geo
        if tier:
            tag += ';tier=' + tier
        self.signals.append({
            "tag": tag,
            "host": "ASEARCH",
            "name": signal,
            "title": title,
            "color": color
        })

    def add_constlines(self, constlines):
        self.consts += constlines


class Panel:
    def __init__(self, title, graphs_title, tag):
        self.charts = []
        self.panel = {
            "title": title,
            "type": "panel",
            "editors": [
                "mvel",
                "kulikov",
                "ironpeter"
            ],
            "charts": self.charts
        }
        self.graphs_title = graphs_title
        self.tag = tag

    def add_graphs(self, graphs_params):
        row = 1
        for params in graphs_params:
            height = params.get('height', 1)
            title_suffix = params.get('title_suffix', params['signals'][0][2])
            col = 1
            for geo in ('man', 'sas', 'msk'):
                gr = Graphic('{} {} {}'.format(self.graphs_title, geo.upper(), title_suffix))
                patch = params.get('patch', None)
                if patch:
                    gr.graph.update(patch)
                gr.set_grid_cell(col, row, height=height)
                gr.add_signals(geo, lst=params['signals'], tag=self.tag, tier=params.get('tier', None))
                gr.add_constlines(params.get('constlines', []))
                gr.set_stacked(params.get('stacked', False))
                self.charts.append(gr.graph)
                col += 2
            row += height

    def save(self):
        with open(self.panel['title'] + '.json', 'w') as f:
            f.write(json.dumps(self.panel, separators=(',', ': '), indent=4, sort_keys=True))

    def save_to_ambry(self):
        # https://wiki.yandex-team.ru/users/kaathewise/ambry/
        data = json.dumps({
            'keys': {
                'user': 'and42',
                'key': self.panel['title']
            },
            'values': self.panel
        })
        req = urllib2.urlopen('http://yasm.yandex-team.ru/srvambry/upsert', data=data)
        req.read()
        print 'save to ambry panel {}'.format(self.panel['title'])


def print_usage():
    print "usage:\ngolovan_gen.py [itype]\n\tdefault itype == tdi"
    sys.exit(1)


def noapache_geo_panel(prj):
    title = "noapache_geo"
    tag = "itype=noapache;ctype=prestable,prod"
    graphs_title = 'NOAPACHE'
    if prj:
        title = prj + '_' + title
        tag += ';prj={}-main'.format(prj)
        graphs_title = prj.upper() + ' ' + graphs_title
    panel = Panel(
        title=title,
        graphs_title=graphs_title,
        tag=tag,
    )
    panel.add_graphs([
        RESP_PERC,
        AH_RESP_PERC,
        dict(
            signals=QUANTILE_95,
            constlines=[dict(title='KPI', value=500, color='#ff2020', active=True)],
        ),
        dict(
            signals=APPHOST_QUANTILE_95,
            constlines=[dict(title='KPI', value=500, color='#ff2020', active=True)],
        ),
        dict(
            signals=RESP_COMPLETE,
        ),
        dict(
            signals=TOTAL,
        ),
        dict(
            signals=APPHOST_PERC,
        ),
        dict(
            title_suffix='sum grp. factor',
            signals=(
                ("div(unistat-sum_grp_dmmm, unistat-all-reports_dmmm)", "#3333ff", "SGF all"),
                ("div(unistat-sum_grp-humans_dmmm, unistat-total-humans_dmmm)", "#33ff33", "SGF humans"),
            ),
        ),
        dict(
            signals=OVERLOAD,
        ),
        dict(
            signals=CHART_CRASHES,
            title_suffix='kills',
        ),
        dict(
            signals=PAGE_FAULTS,
        ),
    ])
    return panel


def cfg_from(geo):
    u = urllib2.urlopen('http://{}.yandex.ru/viewconfig'.format(geo))
    cfg = u.read()
    u.close()
    return cfg


def get_noapache_subsources(geo=['msk', 'man', 'sas']):
    cfg = '\n'.join([cfg_from(g) for g in geo])
    cfg_srcs = set()
    for line in cfg.split('\n'):
        line = line.strip(' ')
        if line.startswith('ServerDescr'):
            cfg_srcs.add(line.split(' ')[-1])
    return cfg_srcs


def signals_last(tail):
    cfg_srcs = get_noapache_subsources()
    src_groups = {}

    def ext_group(grp_id, src):
        if grp_id in src_groups:
            src_groups[grp_id].append(src)
        else:
            src_groups[grp_id] = [src]

    for src in cfg_srcs:
        if src == 'WEB' or src == 'WEB_MISSPELL':
            ext_group('WEB/WEB_MISSPELL', src)
        elif src.startswith('YABS'):
            ext_group('YABS*', src)
        elif src.startswith('QUICK'):
            ext_group('QUICK*', src)
        elif src.startswith('VIDEO'):
            ext_group('VIDEO*', src)
        elif src.startswith('IMAGES'):
            ext_group('IMAGES*', src)
        else:
            ext_group('sum(*)', src)

    color_map = {
        'WEB/WEB_MISSPELL': '#37bff2',
        'YABS*': '#888833',
        'QUICK*': '#e85b4e',
        'VIDEO*': '#c9bedd',
        'IMAGES*': '#e85b6e',
        'sum(*)': '#8e8e8e',
    }

    signals = []
    for title, srcs in src_groups.iteritems():
        total = 'hcount(unistat-h_last_src_total_dhhh, {}, inf)'.format(tail)
        if len(srcs) == 1:
            sum_signals = 'hcount(unistat-h_last_src_{}_dhhh, {}, inf)'.format(srcs[0], tail)
        else:
            signals_lst = ', '.join(['hcount(unistat-h_last_src_{}_dhhh, {}, inf)'.format(src, tail) for src in srcs])
            sum_signals = 'sum(' + signals_lst + ')'
        name = 'perc({}, {})'.format(sum_signals, total)
        color = color_map[title]
        signals.append((name, color, title))
    return signals


def noapache_last_panel(prj, tail=300, tier='web-www'):
    title = "noapache_last300"
    tag = "itype=noapache;ctype=prestable,prod"
    graphs_title = 'NOAPACHE'
    if prj:
        title = prj + '_' + title
        tag += ';prj={}-main'.format(prj)
        graphs_title = prj.upper() + ' ' + graphs_title
    panel = Panel(
        title=title,
        graphs_title=graphs_title,
        tag=tag,
    )
    graphs = [RESP_PERC]

    graphs.append(
        dict(
            signals=signals_last(tail),
            title_suffix='last sources for [{};inf]'.format(tail),
            stacked=True,
            tier=tier,
            height=2,
        )
    )
    colors = (
        "#ffff88",
        "#cdeb8b",
        "#c3d9ff",
        "#ff1a00",
        "#008c00",
        "#4096ee",
        "#cc0000",
        "#c79810",
        "#6bba70",
        "#ff7400",
        "#006e2e",
        "#3f4c6b",
        "#ff0084",
        "#eeeeee",
        "#d01f3c",
    )
    cfg_srcs = get_noapache_subsources()

    def chunks(l, n):
        n = max(1, n)
        return [l[i:i + n] for i in range(0, len(l), n)]

    for src15 in chunks(sorted(list(cfg_srcs)), 15):
        icolor = 0
        signals = []
        for src in src15:
            signals.append((
                "hcount(unistat-h_last_src_{}_dhhh, {}, inf)".format(src, tail),
                colors[icolor],
                src,
            ))
            icolor += 1
        graphs.append(
            dict(
                signals=signals,
                title_suffix='last sources for [{};inf]'.format(tail),
                stacked=True,
                tier=tier,
            )
        )
    panel.add_graphs(graphs)
    return panel


def update_yasm():
    for prj in ('web', 'video', 'imgs', None):
        noapache_geo_panel(prj).save_to_ambry()
    noapache_last_panel('web').save_to_ambry()


if __name__ == '__main__':
    arg = sys.argv[1:]
    debug = False
    if len(arg) >= 1:
        if len(arg) == 1 and arg[0] == '-d':
            debug = True
        else:
            print_usage()
            sys.exit(1)
    if debug:
        noapache_last_panel('web').save()
    else:
        update_yasm()
