# -*- coding: utf-8 -*-

import datetime
import urlparse
import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects import resource_types
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common.noapacheupper import resources as nr
from sandbox.projects.common.noapacheupper import search_component as nsc
from sandbox.projects.common.search import config as sconf
from sandbox.projects.websearch.upper import BuildNoapacheupperNehCache as BNNehCache
from sandbox.projects.websearch.upper import BuildNoapacheupperRequests as BNRequests
from sandbox.projects.websearch.upper import BuildNoapacheupperRequests2 as BNRequests2
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.testenv_client import TEClient


SUPTYPES_WHICH_NEED_BLENDER = {
    "web_kubr_randomlog",
    "web_kubr",
}


class UpdateStandaloneNoapacheupperResources(nr.ImportNoapacheupperResources2):
    """
        Предназначен для создания свежих ресурсов, используемых в testenv-тестах standalone noapacheupper
        (регулярного автоматического запуска из scheduler-а).
        За основу берётся последний успешный запуск задачи GET_STANDALONE_NOAPACHEUPPER_RESPONSES от БД, для которой
        нужны новые ресурсы, негенерируемая часть ресурсов (exe/rearrange data) берётся та же, что использовалась
        в данной задаче, и используются для прогона задачи генерации neh_cache со свежим конфигом и запросами.

        На входе
            testenv-база, ресурсы для которой обновлены;
            url report-а, через который будет производиться переколдовка запросов;
            url с которого будет взята эталонная конфигурация noapacheupper;

        из sandbox берётся самый последний ресурс пользовательких запросов с атрибутом
        for_testenv_database_{testenv-база},
        например for_testenv_database_noapacheupper-trunk

        На выходе:
            neh cache;
            запросы, для которых набран кеш;
            конфиг, при котором набирался кеш;

        - помеченные атрибутом autoupdate-resources-{testenv-база}-task-id={id-этой-таски}
    """

    class Parameters(sdk2.Task.Parameters):
        testenv_db = sdk2.parameters.String("TestEnv database", default="noapacheupper-trunk")
        noapacheupper_config_url = sdk2.parameters.String(
            "Get noapacheupper config from url",
            # TODO(mvel): Fix it with somewhat better value
            default="http://zelo.yandex.ru:10157/yandsearch?info=getconfig"
        )
        noapacheupper_apphost_mode = sdk2.parameters.Bool("Build apphost requests", default=False)
        requests_limit = sdk2.parameters.Integer("Requests limit (0 = no limit)", default=0)
        noapacheupper_orig_config_resource_id = sdk2.parameters.Resource(
            "Get noapacheupper config from resource (if url empty)",
            resource_type=resource_types.NOAPACHEUPPER_CONFIG,
        )

        with sdk2.parameters.String("Subtype test") as test_subtype:
            test_subtype.values.web_kubr_farpages = "WEB_KUBR_FARPAGES"
            test_subtype.values.web_kubr = "WEB_KUBR"
            test_subtype.values.web_tr = "WEB_TR"
            test_subtype.values.imgs_kubr = "IMGS_KUBR"
            test_subtype.values.imgs_tr = "IMGS_TR"
            test_subtype.values.video_kubr = "VIDEO_KUBR"
            test_subtype.values.video_tr = "VIDEO_TR"
            test_subtype.values.misc_web_kubr = "MISC_WEB_KUBR"

        report_url = sdk2.parameters.String(
            "Use this upper/report for 'perekoldovka' requests",
            default="http://hamster.yandex.ru/yandsearch?"
        )
        rtmr_url = sdk2.parameters.String(
            "Use rtmr for get cgi param yandexuid= possible values (not use param if url empty)",
            default="http://rtmr.search.yandex.net:8080/yandsearch?"
        )
        additional_cgi_params = sdk2.parameters.String(
            "Additional cgi params for report (string for trivial append to query for report)",
            default=""
        )
        evlogdump_executable_resource_id = sdk2.parameters.LastReleasedResource(
            "Evlogdump executable",
            resource_type=resource_types.EVLOGDUMP_EXECUTABLE,
            required=False,
        )

    def on_execute(self):
        testenv_database = self.Parameters.testenv_db
        subtype_test = self.Parameters.test_subtype
        self.Context.current_te_config_id = None

        if (
            (subtype_test.startswith('misc_') or self.Parameters.noapacheupper_apphost_mode)
            and not self.Context.misc_resources
        ):
            # for run BuildNoapacheRequests2 need get run-time noapache resources
            self._nanny_token = sdk2.Vault.data("SEARCH-RELEASERS", "nanny_oauth_token")
            logging.info("Import resources from report_url")
            self.Context.misc_resources = self.import_resources(self.Parameters.report_url)

        testenv_database_with_subtype_test_mark = (
            testenv_database if not subtype_test
            else testenv_database.replace('-trunk', '-' + subtype_test + '-trunk')
        )

        if not self.Context.build_requests_task_id:
            self.import_config()

            # находим самый последний ресурс пользовательских запросов с атрибутом-меткой использования
            # в нашей testenv_database + метка подтипа теста
            uq_attr_name = "for_testenv_database_" + testenv_database_with_subtype_test_mark

            # генерируем запросы (cgi) для noapacheupper (переколдуем)
            report_url = url_with_cgi(self.Parameters.report_url, '')
            if subtype_test.startswith('misc_'):
                # у misc-ов свой, - хитрый формат запросов,
                # обычный способ их переколдовки не подходит
                misc_queries = resource_types.MISC_TEXT_QUERIES.find(
                    attrs={uq_attr_name: 1}
                ).first()

                if not misc_queries:
                    eh.check_failed("Not found MISC_TEXT_QUERIES with attribute {}".format(uq_attr_name))

                # здесь нам нужно взять из hamster-а текущие exe/cfg/data
                # для поднятия на этом локального noapache
                reqs_task = BNRequests2.create_build_noapacheupper_requests2(
                    report_url,
                    misc_queries.id,
                    resources=self.Context.misc_resources,
                    descr="update testenv {}/{} resources".format(testenv_database, subtype_test),
                    additional_cgi_params=self.Parameters.additional_cgi_params,
                    noapacheupper_apphost_mode=True,
                    current_sdk2_task=self,
                )
            else:
                users_queries = resource_types.USERS_QUERIES.find(
                    attrs={uq_attr_name: 1}
                ).first()

                if not users_queries:
                    eh.check_failed("Not found USERS_QUERIES with attribute {}".format(uq_attr_name))

                if self.Parameters.noapacheupper_apphost_mode:
                    # здесь нам нужно взять из hamster-а текущие exe/cfg/data для поднятия на этом локального noapache
                    reqs_task = BNRequests2.create_build_noapacheupper_requests2(
                        report_url,
                        users_queries.id,
                        resources=self.Context.misc_resources,
                        descr="update testenv {}/{} resources".format(testenv_database, subtype_test),
                        additional_cgi_params=self.Parameters.additional_cgi_params,
                        noapacheupper_apphost_mode=True,
                        verify_stderr=False,  # ignore errors after FAIL task/88721509
                        current_sdk2_task=self,
                        space=80 * 1024 if subtype_test == "web_kubr_farpages" else 30 * 1024,
                    )
                else:
                    if '?' not in report_url:
                        report_url += '?'
                    reqs_task = BNRequests.create_task_BuildNoapacheupperRequests(
                        report_url,
                        users_queries.id,
                        "update testenv {}/{} resources".format(testenv_database, subtype_test),
                        rtmr_url=self.Parameters.rtmr_url,
                        additional_cgi_params=self.Parameters.additional_cgi_params,
                        requests_limit=self.Parameters.requests_limit,
                        current_sdk2_task=self,
                    )
            self.Context.build_requests_task_id = reqs_task.id
        else:
            reqs_task = sdk2.Task[self.Context.build_requests_task_id]

        break_or_finish = ctt.Status.Group.BREAK + ctt.Status.Group.FINISH

        if not (reqs_task.status in break_or_finish):
            raise sdk2.WaitTask(reqs_task, break_or_finish)
        utils.check_if_tasks_are_ok([reqs_task.id])

        # генерируем neh_cache
        if not self.Context.build_cache_task_id:
            noapache_data = self._get_resource_from_te_db(
                testenv_database,
                sdk2.Task["BUILD_NOAPACHEUPPER_DATA"],
                "out_resource_id"
            )
            noapache_exe = self._get_resource_from_te_db(
                testenv_database,
                sdk2.Task["BUILD_NOAPACHE_UPPER"],
                "noapacheupper_resource_id"
            )

            cache_task = BNNehCache.create_task_BuildNoapacheupperNehCache(
                getattr(self.Context, nsc.Params.Config.name),
                noapache_exe,
                noapache_data,
                reqs_task.Context.upper_int_requests_res_id,
                "update testenv {}/{} resources".format(testenv_database, subtype_test),
                apphost_mode=self.Parameters.noapacheupper_apphost_mode,
                current_sdk2_task=self,
            )
            self.Context.build_cache_task_id = cache_task.id
        else:
            cache_task = sdk2.Task[self.Context.build_cache_task_id]

        get_blender_cache = (subtype_test in SUPTYPES_WHICH_NEED_BLENDER)
        if get_blender_cache:
            # генерируем neh_cache
            if not self.Context.build_blender_cache_task_id:
                noapache_data = self._get_resource_from_te_db(
                    testenv_database,
                    sdk2.Task["BUILD_NOAPACHEUPPER_DATA"],
                    "out_resource_id"
                )
                noapache_exe = self._get_resource_from_te_db(
                    testenv_database,
                    sdk2.Task["BUILD_NOAPACHE_UPPER"],
                    "noapacheupper_resource_id"
                )

                blender_cache_task = BNNehCache.create_task_BuildNoapacheupperNehCache(
                    getattr(self.Context, nsc.Params.Config.name),
                    noapache_exe,
                    noapache_data,
                    reqs_task.Context.blender_requests_res_id,
                    "update testenv {}/{} resources".format(testenv_database, subtype_test),
                    apphost_mode=self.Parameters.noapacheupper_apphost_mode,
                    current_sdk2_task=self,
                    run_mode='blender',
                )
                self.Context.build_blender_cache_task_id = blender_cache_task.id
            else:
                blender_cache_task = sdk2.Task[self.Context.build_blender_cache_task_id]

            if not (blender_cache_task.status in break_or_finish):
                raise sdk2.WaitTask(blender_cache_task, break_or_finish)
            utils.check_if_tasks_are_ok([blender_cache_task.id])

        if not (cache_task.status in break_or_finish):
            raise sdk2.WaitTask(cache_task, break_or_finish)

        utils.check_if_tasks_are_ok([cache_task.id])

        # помечаем готовые ресурсы атрибутом, чтобы testenv автоматически
        # использовал их для замены текущих статических ресурсов
        attr_name = "autoupdate-resources-{}-task-id".format(testenv_database_with_subtype_test_mark).replace("-", "_")

        resources_to_autoupdate = [
            cache_task.Context.neh_cache_resource_id,
            reqs_task.Context.upper_int_requests_res_id,
            reqs_task.Context.blender_requests_res_id,
        ]
        self.set_neh_cache_attr(reqs_task.Context.upper_int_requests_res_id, cache_task.Context.neh_cache_resource_id)
        if get_blender_cache:
            self.set_neh_cache_attr(reqs_task.Context.blender_requests_res_id,
                                    blender_cache_task.Context.neh_cache_resource_id)
            resources_to_autoupdate.append(blender_cache_task.Context.neh_cache_resource_id)
        if getattr(self.Context, nsc.Params.Config.name) != self.Context.current_te_config_id:
            # update config only if its really changed
            resources_to_autoupdate.append(getattr(self.Context, nsc.Params.Config.name))

        for resource_id in resources_to_autoupdate:
            resource = sdk2.Resource.find(id=resource_id).first()
            setattr(resource, attr_name, self.id)

    @staticmethod
    def set_neh_cache_attr(requests_res_id, nec_cache_res_id):
        resource = sdk2.Resource.find(id=requests_res_id).first()
        setattr(resource, "neh_cache", nec_cache_res_id)

    @staticmethod
    def _get_resource_from_te_db(testenv_database, task_type, ctx_resource_name):
        noapache_data_tasks = sdk2.Task.find(
            task_type,
            status=ctt.Status.Group.SUCCEED,
            completed_only=True,
            tags=['TESTENV-DATABASE-{}'.format(testenv_database.upper())],
            hidden=True,
        ).limit(16)
        for t in noapache_data_tasks:
            tdb = t.Context.testenv_database
            if tdb and tdb == testenv_database:
                return getattr(t.Context, ctx_resource_name)

        eh.check_failed("There are no any task from '{}' among {}".format(
            testenv_database, [t.id for t in noapache_data_tasks]
        ))

    def import_config(self):
        if getattr(self.Context, nsc.Params.Config.name):
            return  # already imported

        # загружаем эталонный конфиг с указаннго url
        descr = "Config from {} at {} + async search + IPV6 enabled".format(
            self.Parameters.noapacheupper_config_url, str(datetime.date.today())
        )
        cfg_res = resource_types.NOAPACHEUPPER_CONFIG(
            self, descr, "noapacheupper.cfg", arch="any", ttl=90,
        )
        resource_data = sdk2.ResourceData(cfg_res)
        if self.Parameters.noapacheupper_config_url:
            orig_cfg = str(resource_data.path) + ".orig"
            load_file(self.Parameters.noapacheupper_config_url, orig_cfg)
        else:
            orig_cfg = str(sdk2.ResourceData(sdk2.Resource.find(
                id=self.Parameters.noapacheupper_orig_config_resource_id
            ).first()).path)
        cfg = sconf.NoapacheupperConfig.get_config_from_file(orig_cfg)
        cfg.enable_async_search()
        cfg.enable_ipv6()
        cfg.save_to_file(str(resource_data.path))
        resource_data.ready()
        cfg_res_id = cfg_res.id

        try:
            # get currently used resource from noapacheupper-trunk te db
            if not self.Context.current_te_config_id:
                last_te_standalone_task_id = TEClient.get_last_sandbox_task_ids(
                    'noapacheupper-trunk',
                    ['GET_RESPONSES_STANDALONE_BLENDER']
                )[0]['task_id']
                last_te_standalone_task = channel.sandbox.get_task(last_te_standalone_task_id)
                self.Context.current_te_config_id = last_te_standalone_task.ctx['noapacheupper_config_resource_id']

            # Use previous TE config if nothing changed
            te_cfg_file = str(sdk2.ResourceData(sdk2.Resource.find(
                id=self.Context.current_te_config_id
            ).first()).path)
            te_cfg = sconf.NoapacheupperConfig.get_config_from_file(te_cfg_file)
            if te_cfg == cfg:
                cfg_res_id = self.Context.current_te_config_id
        except Exception as e:
            eh.log_exception("Failed to get current_te_config_id", e)

        setattr(self.Context, nsc.Params.Config.name, cfg_res_id)


def load_file(url, fn):
    try:
        r = requests_wrapper.get_r(url)
        with open(fn, 'w') as f:
            f.write(r.text)

    except Exception:
        eh.check_failed("Can't load original config from {}:\n{}".format(url, eh.shifted_traceback()))


def url_with_cgi(url, cgi, default_scheme='http', default_path='yandsearch'):
    # TODO(mvel): wipe this trash
    parsed_url = urlparse.urlparse(url, default_scheme)
    scheme = parsed_url[0]
    netloc = parsed_url[1]
    path = parsed_url[2] if (parsed_url[2] and parsed_url[2] != '/') or not default_path else default_path
    params = ''
    query = parsed_url[4]
    if query and query[-1] != '&' and cgi and cgi[0] != '&':
        query += '&'
    if cgi:
        query += cgi
    fragment = ''
    return urlparse.urlunparse((scheme, netloc, path, params, query, fragment))
