# -*- coding: utf-8 -*-

import logging
import os
from six.moves.urllib.request import urlopen

from sandbox import sdk2
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import config_patcher_tool
from sandbox.projects.common.search.components import component
from sandbox.projects.common.search.components import DEFAULT_START_TIMEOUT, DEFAULT_SHUTDOWN_TIMEOUT


LOGGER = logging.getLogger(__name__)


class Noapacheupper(
    component.WaitPortComponentMixin,
    component.ProcessComponentMixinWithShutdownSDK2,
    component.Component,
):
    name = "upper"
    default_port = 10150
    http_collection = "yandsearch"
    _LOG_TMPL = name + "_{port}_{log_type}.log"
    _QUERY_CACHE_TMPL = name + "_{}_query_cache"
    _CONFIG_NAME_TMPL = name + "_{}_patched.cfg"

    def __init__(
        self, task, binary, config,
        rearrange_data=None,
        rearrange_dynamic_data=None,
        rearrange_data_fast=None,
        upper_data=None,
        port=default_port,
        start_timeout=DEFAULT_START_TIMEOUT,
        shutdown_timeout=DEFAULT_SHUTDOWN_TIMEOUT,
        cfg_patch_dict=None,
        disable_timeouts=True,
        patch_queues_size=True,
        query_cache_dir=None,
        eventlog_path=None,
    ):
        self.task = task
        self.binary_data = sdk2.ResourceData(binary)
        self.config_data = sdk2.ResourceData(config)
        self.upper_data_path = _path_from_resource(upper_data)
        self.rearrange_data_path = _path_from_resource(rearrange_data)
        self.rearrange_dynamic_path = _path_from_resource(rearrange_dynamic_data)
        self.rearrange_data_fast_path = _path_from_resource(rearrange_data_fast)
        self.port = port
        self.apphost_port = port + 1
        self.grpc_port = port + 2
        self.start_timeout = start_timeout
        self.cfg_patch_dict = cfg_patch_dict or {}
        self.disable_timeouts = disable_timeouts
        self.patch_queues_size = patch_queues_size
        self.eventlog_path = eventlog_path or self.path_to_log("event")
        self._init_query_cache(query_cache_dir)
        self.patch_config(task)
        component.WaitPortComponentMixin.__init__(
            self, endpoints=[("localhost", port)], wait_timeout=start_timeout
        )
        component.ProcessComponentMixinWithShutdownSDK2.__init__(
            self,
            args=[str(self.binary_data.path), "-d", "-p", str(self.port), self.cfg_dest_path],
            shutdown_url="http://localhost:{}/admin?action=shutdown&timeout={}".format(self.port, shutdown_timeout),
            log_prefix=self.name,
        )
        LOGGER.info("Component '%s' initialized successfully", self.name)

    def warmup(self):
        """
            Для верхнего метапоиска прогревочный запрос вреден (см. SEARCH-616 в Middlesearch, r1419472),
            так что вместо этого вычитываем версию (чтобы убедиться, что приложение хоть как-то заработало)
        """
        try:
            LOGGER.info("Warming up...")
            url = "http://127.0.0.1:{}/{}?info=getversion".format(self.port, self.http_collection)
            LOGGER.info("Fetch version info: %s", url)
            LOGGER.info("Use upper version: %s", urlopen(url, timeout=300).read())
        except Exception as e:
            eh.check_failed("Cannot fetch upper version. Error: {}".format(e))

    def patch_config(self, task):
        cfg_patch_path = self.prepare_cfg_patch()
        init_cfg_path = str(self.config_data.path)
        tmp_cfg_path = init_cfg_path
        patched_cfg = config_patcher_tool.patch_cfg(task, tmp_cfg_path, cfg_patch_path, self.cfg_dest_path, self.name)
        LOGGER.debug("Patched cfg:\n%s", patched_cfg)

    def _init_query_cache(self, query_cache_dir):
        if not query_cache_dir:
            query_cache_dir = self._QUERY_CACHE_TMPL.format(self.port)
        sdk2.paths.make_folder(query_cache_dir)
        self.__query_cache_dir = query_cache_dir

    def query_cache_dir(self, n):
        return os.path.abspath(os.path.join(self.__query_cache_dir, "tmpdir.{}".format(n)))

    @property
    def cfg_dest_path(self):
        return os.path.abspath(self._CONFIG_NAME_TMPL.format(self.port))

    def path_to_log(self, log_type):
        return os.path.abspath(self._LOG_TMPL.format(port=self.port, log_type=log_type))

    def prepare_cfg_patch(self):
        pure_dir = None
        if self.upper_data_path is not None:
            pure_dir = os.path.join(self.upper_data_path, "pure")
            self.rearrange_data_path = self.rearrange_data_path or os.path.join(self.upper_data_path, "rearrange")
            self.rearrange_dynamic_path = (
                self.rearrange_dynamic_path or os.path.join(self.upper_data_path, "rearrange.dynamic")
            )
            self.rearrange_data_fast_path = (
                self.rearrange_data_fast_path or os.path.join(self.upper_data_path, "rearrange.fast")
            )
        self.cfg_patch_dict.update({
            "Server.Port": str(self.port),
            "Server.AppHostOptions": "Port={}, Threads=24, GrpcPort=+2, GrpcThreads=24".format(self.apphost_port),
            "Server.LoadLog": self.path_to_log("load"),
            "Server.ServerLog": self.path_to_log("server"),
            "Server.EventLog": self.eventlog_path,
            "Server.LogExceptions": "True",
            "Server.ClientTimeout": 2000,
            "Server.ScarabReqAnsLog": "__remove__",
            "Server.SiteSearchScarabReqAnsLog": "__remove__",
            "Server.DirectScarabReqAnsLog": "__remove__",
            "Server.ReqAnsLog": "__remove__",
            "Server.ReqansUnifiedAgentLog": "__remove__",
            "Server.ReqansUnifiedAgentMiscLog": "__remove__",
            "Server.AliceReqansUnifiedAgentLog": "__remove__",
            "Server.ErrorBoosterUnifiedAgentLog": "__remove__",
            "Server.XmlReqansUnifiedAgentLog": "__remove__",
            "Server.RtReqansUnifiedAgentLog": "__remove__",
            "Server.FatReqansUnifiedAgentLog": "__remove__",
            "Server.PushAgentOptions": "__remove__",
            "Collection.UseRequiredBaseTimestamp": "False",
            "Collection.IndexDir": pure_dir,
            "Collection.RearrangeDataDir": self.rearrange_data_path,
            "Collection.RearrangeDynamicDataDir": self.rearrange_dynamic_path,
            "Collection.RearrangeDataFastDir": self.rearrange_data_fast_path,
            "Collection.MaxAllfactorsPerBasesearch": 50,
            "Collection.QueryCache[:].MemoryLimit": "__remove__",
            "Collection.QueryCache[0].Dir": self.query_cache_dir(0),
            "Collection.CacheThreadLimit": 32,  # For SEARCH-1474
            "Collection.RequestTimeoutLimit": "40s",  # SEARCH-5816
        })
        if self.disable_timeouts:
            self.cfg_patch_dict.update({
                "Collection.ScatterOptions[:].TimeoutTable": "300s",
                "Collection.ScatterOptions[:].MessengerOptions": "__remove__",
                "Collection.ScatterOptions[:].RemoteTimeoutOptions": "__remove__",
                "Collection.ConnectTimeout": "300s",
                "Collection.MessengerOptions": "__remove__",
                "Collection.UseTimeoutTable": "__remove__",
            })
        if self.patch_queues_size:
            self.cfg_patch_dict.update({
                "Server.Threads": 100,
                "Server.QueueSize": 100,
                "Collection.RequestThreads": 100,
                "Collection.RequestQueueSize": 100
            })

        patch_path = os.path.abspath("cfg_patch.json")
        fu.json_dump(patch_path, self.cfg_patch_dict, indent=1)
        return patch_path


def _path_from_resource(resource):
    return str(sdk2.ResourceData(resource).path) if resource else None
