# -*- coding: utf-8 -*-

import logging
import os
import shutil
import tempfile
from contextlib import contextmanager

from sandbox import sdk2
from sandbox.sdk2 import svn

from sandbox.common import errors
from sandbox.projects.common import constants as consts
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.noapacheupper import build as nb
from sandbox.projects.websearch.upper import resources as upper_resources
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker

TESTENV_TAG_PREFIX = "TESTENV-DATABASE"


class BuildRearrangeDataFast(nanny.ReleaseToNannyTask2, ExecutionTimeTracker):
    """
        Сборка быстрых данных для верхнего метапоиска
    """

    _MAX_REARRANGE_DATA_FAST_SIZE = 3300

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20 * 1024  # Mb
        ram = 24 * 1024
        cores = 3

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 40 * 60  # 40 min

        arcadia_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        check_size = sdk2.parameters.Bool("Check data size", default=True)

        with sdk2.parameters.RadioGroup('Vertical') as vertical:
            vertical.values['web'] = vertical.Value('WEB', default=True)
            vertical.values['video'] = vertical.Value('VIDEO')

        build_system = build_parameters.BuildSystem()
        yt_store_parameters = build_parameters.YtStoreParameters()

        arc_token_owner = sdk2.parameters.String("ARC_TOKEN owner")

        with sdk2.parameters.Output:
            data_resource = sdk2.parameters.Resource("Rearrange Data Fast", resource_type=upper_resources.RearrangeDataFast)
            bundle_resource = sdk2.parameters.Resource("Rearrange Data Fast bundle", resource_type=upper_resources.RearrangeDataFastBundle)

    class Context(ExecutionTimeTracker.Context):
        data_name = 'rearrange.fast'
        bundle_name = data_name + '.tar.gz'

    @property
    def stage_name(self):
        return 'build'

    def on_execute(self):
        with self.memoize_stage.prepare:
            self.Context.svn_info = svn.Arcadia.info(self.Parameters.arcadia_url)
            self.Context.revision = self.Context.svn_info['commit_revision']
            self.Context.svn_info['sandbox_task'] = self.id
            self.Context.svn_info['vertical'] = self.Parameters.vertical
            self.Context.version = self._get_version()
            self.Context.svn_info['fast_data_version'] = self.Context.version
            self.Parameters.description += ' v.{}'.format(self.Context.version)

            self.Parameters.data_resource = upper_resources.RearrangeDataFast(
                self, self.Parameters.description, self.Context.data_name, ttl=20,
                revision=self.Context.revision, version=self.Context.version, vertical=self.Parameters.vertical,
            )
            self.Parameters.bundle_resource = upper_resources.RearrangeDataFastBundle(
                self, self.Parameters.description, self.Context.bundle_name, ttl=20,
                revision=self.Context.revision, version=self.Context.version, vertical=self.Parameters.vertical,
            )

        data_path = str(self.Parameters.data_resource.path)
        data_name = os.path.basename(data_path)

        self.build(data_path)
        self.add_svn_info_file(data_path)
        nb.tar(self.Context.bundle_name, data_path, data_name)

        self.Context.size = nb.get_directory_size(data_path)
        self.set_info('Size: {0:.1f}Mb'.format(self.Context.size))

        if self.Parameters.check_size:
            if self.Context.size > self._MAX_REARRANGE_DATA_FAST_SIZE:
                raise errors.TaskFailure('Data size exceeds maximum allowed size {}Mb'.format(self._MAX_REARRANGE_DATA_FAST_SIZE))

    @contextmanager
    def get_arcadia(self):
        if not sdk.fuse_available():
            raise errors.TaskFailure('Fuse unavailable')
        token = sdk2.Vault.data(self.Parameters.arc_token_owner, name='ARC_TOKEN')
        with sdk.mount_arc_path(self.Parameters.arcadia_url, fallback=True, use_arc_instead_of_aapi=True, arc_oauth_token=token) as arcadia:
            yield arcadia

    def build(self, data_path):
        target = upper_resources.RearrangeDataFast.arcadia_build_path
        build_dir = tempfile.mkdtemp()

        with self.get_arcadia() as arcadia:
            logging.info('build rearrange.fast...')
            sdk.do_build(
                self.Parameters.build_system,
                source_root=arcadia,
                build_type=consts.RELEASE_BUILD_TYPE,
                targets=[target],
                results_dir=build_dir,
                clear_build=True,
                yt_store_params=sdk.YtStoreParams(
                    store=self.Parameters.ya_yt_store,
                    proxy=self.Parameters.ya_yt_proxy,
                    dir=self.Parameters.ya_yt_dir,
                    token=sdk2.Vault.data(
                        self.Parameters.ya_yt_token_vault_owner,
                        self.Parameters.ya_yt_token_vault_name or 'YT_STORE_TOKEN'
                    ) if self.Parameters.ya_yt_token_vault_owner else None,
                    put=self.Parameters.yt_store_parameters.ya_yt_put,
                    codec=self.Parameters.yt_store_parameters.ya_yt_store_codec,
                    replace_result=self.Parameters.yt_store_parameters.ya_yt_replace_result,
                    store_exclusive=self.Parameters.yt_store_parameters.ya_yt_max_cache_size,
                ),
            )
            logging.info('rearrange.fast built')

        shutil.copytree(os.path.join(build_dir, target), data_path)

    def add_svn_info_file(self, data_path):
        logging.info('write .svninfo...')
        with open(os.path.join(data_path, '.svninfo'), 'w') as svn_info_file:
            svn_info_file.write('\n'.join(
                '{}: {}'.format(k, v) for k, v in sorted(self.Context.svn_info.iteritems())
            ) + '\n')

    def _get_version(self):
        current_revision_resource = upper_resources.RearrangeDataFast.find(
            attrs={'revision': self.Context.revision, 'vertical': self.Parameters.vertical},
        ).order(-sdk2.Resource.accessed).first()
        if current_revision_resource and current_revision_resource.version:
            logging.debug('Found resource with this revision: {}'.format(current_revision_resource.id))
            return current_revision_resource.version

        resources = list(upper_resources.RearrangeDataFast.find(attrs={'vertical': self.Parameters.vertical}).order(-sdk2.Resource.accessed).limit(25))
        if resources:
            max_resource_version = max(map(lambda resource: resource.version, resources))
            return max_resource_version + 1
        return 1

    def on_release(self, additional_params):
        if any((tag.startswith(TESTENV_TAG_PREFIX) for tag in self.Parameters.tags)):
            nanny.ReleaseToNannyTask2.on_release(self, additional_params)
        sdk2.Task.on_release(self, additional_params)
