# -*- coding: utf-8 -*-

import json
import logging
import os
import sys
import time

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.resource import State
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.task import Semaphores
from sandbox.common.types.task import Status
from sandbox.common.types.notification import Transport
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import task_env
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess


if sys.version_info[0:2] >= (3, 8):  # python 3.8 or higher
    from functools import cached_property
else:
    from sandbox.projects.common.decorators import memoized_property as cached_property

from sandbox.projects.fastres.UploadFastresWizardData import FastresWizardData
from sandbox.projects.mt.spellchecker import SpellcheckerUpdate
from sandbox.projects.voicetech.resource_types import VOICETECH_TTS_RU_FASTDATA_BUNDLE
from sandbox.projects.websearch.begemot import resources as begemot_resources
from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault
from sandbox.projects.websearch.upper import resources as upper_resources
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker


NANNY_API_URL = 'http://nanny.yandex-team.ru/'


class DeployFastData(ExecutionTimeTracker):
    class Requirements(task_env.TinyRequirements):
        cores = 1
        ram = 2048
        disk_space = 10240  # 10 Gb
        environments = [
            environments.PipEnvironment('yandex-yt', version='0.10.8'),
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 1500  # 25min

        with sdk2.parameters.RadioGroup("Flow type (for CI)") as flow_type:
            flow_type.values["DEFAULT"] = flow_type.Value("Default flow", default=True)
            flow_type.values["ROLLBACK"] = flow_type.Value("Rollback flow")

        release_fast_data_on_success = sdk2.parameters.Bool("Release current REARRANGE_DATA_FAST_BUNDLE resource on success", default=False)
        use_last_released_bundle = sdk2.parameters.Bool("Use last released bundle", default=False)
        with use_last_released_bundle.value[False]:
            fast_data_bundle = sdk2.parameters.Resource(
                "Fast Data bundle to deploy",
                resource_type=[
                    upper_resources.RearrangeDataFastBundle,
                    begemot_resources.BEGEMOT_FAST_BUILD_FRESH_CONFIG,
                    begemot_resources.BEGEMOT_REALTIME_PACKAGE,
                    SpellcheckerUpdate,
                    begemot_resources.BEGEMOT_FRESH_DATA_FOR_SPELLCHECKER_PACKED,
                    FastresWizardData,
                    VOICETECH_TTS_RU_FASTDATA_BUNDLE,
                ],
                required=True,
            )

        blank_fast_data_bundle = sdk2.parameters.Resource(
            "Blank Fast Data bundle for fast activation",
            resource_type=[
                upper_resources.RearrangeDataFastBundleBlank,
            ],
            required=False,
        )

        with sdk2.parameters.RadioGroup("Deployer mode") as deployer_mode:
            deployer_mode.values['standalone'] = deployer_mode.Value('Run deployer in task', default=True)
            deployer_mode.values['nanny_service'] = deployer_mode.Value('Deploy config to nanny service with deployer')

            with deployer_mode.value['standalone']:
                use_testing_deployer = sdk2.parameters.Bool("Use testing deployer", default=False)
                with use_testing_deployer.value[False]:
                    deployer = ResourceWithLastReleasedValueByDefault(
                        "search/tools/fast_data_deployment/deployer binary",
                        resource_type=upper_resources.FastDataDeployer,
                        required=True,
                    )

            with deployer_mode.value['nanny_service']:
                nanny_service = sdk2.parameters.String('Deployer Nanny service name', required=True)
                tasks_to_wait = sdk2.parameters.List(
                    'Tasks to wait before activating snapshot with new config',
                    sdk2.parameters.Integer
                )

        only_prepare = sdk2.parameters.Bool('Just prepare data, do not activate', default=False)
        with sdk2.parameters.String('Config source') as deploy_config_source:
            deploy_config_source.values['input_json'] = deploy_config_source.Value(
                'Input JSON',
                default=True,
            )
            deploy_config_source.values['file'] = deploy_config_source.Value(
                'File',
            )

            with deploy_config_source.value.input_json:
                deploy_config = sdk2.parameters.JSON('Deploy config')

            with deploy_config_source.value.file:
                deploy_config_path = sdk2.parameters.String('Deploy config Arcadia path')

        with sdk2.parameters.Group('Vault'):
            yt_token_name = sdk2.parameters.String('YT token name', required=True)
            infra_token_owner = sdk2.parameters.String('INFRA_TOKEN owner')
            infra_token_name = sdk2.parameters.String('Infra token name', default='INFRA_TOKEN')
            solomon_token_owner = sdk2.parameters.String('SOLOMON_TOKEN owner')
            nanny_token_name = sdk2.parameters.String('Nanny token name', required=True)

        with sdk2.parameters.Output:
            deploy_config_output_resource = sdk2.parameters.Resource(
                "Deploy config",
                resource_type=upper_resources.FastDataDeployConfig,
            )

    @property
    def stage_name(self):
        return 'deploy'

    @cached_property
    def deploy_config(self):
        if self.Parameters.deploy_config_source == 'file':
            config = json.loads(sdk2.svn.Arcadia.cat('/'.join([
                sdk2.svn.Arcadia.ARCADIA_TRUNK_URL,
                self.Parameters.deploy_config_path
            ])))
        else:
            config = self.Parameters.deploy_config
        return config

    def release_fast_data(self):
        if self.Parameters.release_fast_data_on_success and self.Parameters.fast_data_bundle is not None:
            self.server.release(
                task_id=self.Parameters.fast_data_bundle.task_id,
                type=ReleaseStatus.STABLE,
                subject="Rearrange Data Fast released with DEPLOY_FAST_DATA"
            )

    def on_enqueue(self):
        super(DeployFastData, self).on_enqueue()
        if not self.Requirements.semaphores:
            self.Requirements.semaphores = Semaphores(
                acquires=[
                    Semaphores.Acquire(
                        name='lock-{}_{}-deployer'.format(service, self.Parameters.deployer_mode),
                        weight=2,
                        capacity=2
                    )
                    for service in self.deploy_config["services"].keys()
                ],
            )

    def on_execute(self):
        fast_data_bundle = self.Parameters.fast_data_bundle
        self.not_prepared_services = set()
        if self.Parameters.use_last_released_bundle:
            fast_data_bundle = sdk2.Resource[
                ResourceWithLastReleasedValueByDefault(
                    resource_type=upper_resources.RearrangeDataFastBundle
                ).default_value
            ]

        with self.memoize_stage.patch_parameters:
            self.Context.deploy_config = self.deploy_config

            if fast_data_bundle:
                self.Context.deploy_config.setdefault("resource", {}).update({
                    "url": fast_data_bundle.skynet_id,
                    "version": fast_data_bundle.version,
                    "resource_id": fast_data_bundle.id,
                })

            # infra
            self.Context.deploy_config.setdefault('deploy', {}).setdefault('infra', {})
            self.Context.deploy_config['deploy']['infra'].setdefault('title', "Release Fast Data v. {}".format(
                self.Context.deploy_config['resource']['version']
            ))
            self.Context.deploy_config['deploy']['infra'].setdefault('description', [
                "Deploy task: https://sandbox.yandex-team.ru/task/{}/view".format(str(self.id)),
                "CI flow: https://a.yandex-team.ru/projects/noapache-web/ci/releases/timeline?dir=search%2Fweb%2Frearrs_upper%2Frearrange.fast%2Fci%2FWEB&id=fast-data-release",
                "Fast-Data docs: https://docs.yandex-team.ru/noapache-fast-data/"
            ])

            if hasattr(self.Context, "__CI_CONTEXT") and 'title' in self.Context.__CI_CONTEXT and 'ci_url' in self.Context.__CI_CONTEXT:
                self.Context.deploy_config['deploy']['infra']['description'] = [
                    self.Context.__CI_CONTEXT['title'],
                    "CI launch: {}".format(self.Context.__CI_CONTEXT['ci_url'])
                ] + self.Context.deploy_config['deploy']['infra']['description']

            if self.Parameters.blank_fast_data_bundle:
                self.Context.deploy_config.setdefault("resource", {}).update({
                    "blank_url": self.Parameters.blank_fast_data_bundle.skynet_id,
                })

            self.Context.deploy_config.setdefault("deploy", {}).update({
                "only_prepare": self.Parameters.only_prepare,
            })

            # If it is CI rollback flow -> faster rollback
            if self.Parameters.flow_type == "ROLLBACK":
                for service_config in self.Context.deploy_config['services'].values():
                    service_config.setdefault('prepare', {})['degrade_level'] = 1
                    service_config.setdefault('activate', {})['degrade_level'] = 0.5

        with self.memoize_stage.save_config_resource:
            config_path = 'deploy_config.json'
            self.save_deploy_config_resource(self.Context.deploy_config, config_path)
            self.mark_released_resources('stable', ttl=90)

        if self.Parameters.deployer_mode == 'nanny_service':
            self.run_on_nanny_service(self.Context.deploy_config, self.Parameters.nanny_service)
        elif self.Parameters.deployer_mode == 'standalone':
            self.run_deployer(self.Context.deploy_config)
        else:
            raise errors.TaskFailure('Unknown deploy mode')

    def on_timeout(self, prev_status):
        if self.Context.not_prepared_services != "NotExists" and len(self.Context.not_prepared_services) > 0:
            self.set_info(''.join([
                '<p style="color:red;">Fresh is not prepared on this services:</p>',
                ''.join(['<p style="color:red;">{}</p>'.format(service_id) for service_id in self.Context.not_prepared_services])
                ]),
                do_escape=False
            )

            self.server.notification(
            body="cannot deploy on services: {}".format(' '.join([service_id for service_id in self.Context.not_prepared_services])),
                recipients=["howcanunot"],
                transport=Transport.TELEGRAM
            )

    def save_deploy_config_resource(self, deploy_config, config_path):
        self.Parameters.deploy_config_output_resource = upper_resources.FastDataDeployConfig(
            self, self.Parameters.description, config_path, ttl=90
        )
        with open(config_path, 'w') as config_fd:
            json.dump(deploy_config, config_fd, indent=2)
        sdk2.ResourceData(self.Parameters.deploy_config_output_resource).ready()
        return self.Parameters.deploy_config_output_resource

    def run_on_nanny_service(self, deploy_config, nanny_service):
        from yt import wrapper as yw

        nanny_client = NannyClient(NANNY_API_URL, sdk2.Vault.data(self.Parameters.nanny_token_name))

        with self.memoize_stage.prepare_config:
            logging.info('Update deploy config resource for {}'.format(nanny_service))
            data = nanny_client.update_service_sandbox_file(
                service_id=nanny_service,
                task_type=str(self.type),
                task_id=str(self.id),
                skip_not_existing_resources=False,
                allow_empty_changes=False,
                comment='[v.{}] Fast Data Release (task {})'.format(deploy_config['resource']['version'], str(self.id)),
            )
            logging.info('Update file response:\n{}'.format(json.dumps(data, indent=2)))

            self.Context.snapshot_id = data['runtime_attrs']['_id']
            logging.info('Set snapshot {} state to PREPARED'.format(self.Context.snapshot_id))
            nanny_client.set_snapshot_state(
                service_id=nanny_service,
                snapshot_id=self.Context.snapshot_id,
                state='PREPARED',
                comment='Prepare Fast Data release v.{} (task {})'.format(
                    deploy_config['resource']['version'], self.id
                ),
                prepare_recipe='default',
                set_as_current=True,
            )

        with self.memoize_stage.wait_tasks:
            if self.Parameters.tasks_to_wait:
                logging.info('Wait for tasks {}'.format(', '.join(map(str, self.Parameters.tasks_to_wait))))
                raise sdk2.WaitTask(self.Parameters.tasks_to_wait, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_tasks:
            any_failed = False

            for task_id in self.Parameters.tasks_to_wait:
                task = sdk2.Task[task_id]
                if task.status not in Status.Group.SUCCEED:
                    self.set_info('Task {} failed'.format(lb.task_link(task_id)))
                    any_failed = True

            if any_failed:
                raise errors.TaskFailure('Some tasks are failed')

        with self.memoize_stage.activate_config:
            yt_client = yw.YtClient(
                deploy_config['communication']['cluster'], sdk2.Vault.data(self.Parameters.yt_token_name)
            )
            skipped_services = set()

            def get_actions():
                return ['prepare'] + ['activate'] * (not deploy_config['deploy'].get('only_prepare', False))

            def list_status_paths():
                for service_id in deploy_config.get('services', {}).keys():
                    service_cypress_path = yw.ypath_join(deploy_config['communication']['cypress_dir'], service_id)
                    for geo in yt_client.list(service_cypress_path):
                        if geo == 'man':
                            skipped_services.add(service_id)
                            continue
                        feedback_path = yw.ypath_join(service_cypress_path, geo, 'feedback')
                        if yt_client.exists(feedback_path):
                            for action in get_actions():
                                yield service_id, action, yw.ypath_join(feedback_path, '@{}'.format(action))

            def split_by_status():
                failed, succeeded, inprogress = [], [], []
                for service_id, action, status_path in list_status_paths():
                    status = yt_client.get(status_path)
                    logging.info('service %s, action %s, status = %s (%s)', service_id, action, status, status_path)
                    if status.get('failed') == deploy_config['resource']['version']:
                        failed.append([service_id, action])
                    elif status.get('current') == deploy_config['resource']['version']:
                        succeeded.append([service_id, action])
                    else:
                        inprogress.append([service_id, action])
                return failed, succeeded, inprogress

            logging.info('Set snapshot {} state to ACTIVE'.format(self.Context.snapshot_id))
            nanny_client.set_snapshot_state(
                service_id=nanny_service,
                snapshot_id=self.Context.snapshot_id,
                state='ACTIVE',
                comment='Fast Data release v.{} (task {})'.format(deploy_config['resource']['version'], self.id),
                recipe='default',
                set_as_current=True,
            )

            processed = {action: set() for action in get_actions()}
            self.not_prepared_services = set(deploy_config.get('services', {}).keys())

            while True:
                failed, succeeded, inprogress = split_by_status()

                for service_id, action in succeeded:
                    if service_id not in processed[action]:
                        self.set_info('Done {} stage for {}'.format(action.upper(), service_id))
                        processed[action].add(service_id)
                        if service_id in self.not_prepared_services:
                            self.not_prepared_services.remove(service_id)

                if failed:
                    raise errors.TaskFailure('Failed to {}'.format(', '.join(map(
                        lambda x: '{} on {}'.format(x[1].upper(), x[0]),
                        failed
                    ))))

                if len(inprogress) == 0:
                    self.release_fast_data()
                    break

                self.Context.not_prepared_services = list(self.not_prepared_services)
                time.sleep(1)

            if skipped_services:
                logging.info('Skipped these services in man (see BEGEMOT-2726):\n{}'.format('\n'.join([service_id for service_id in skipped_services])))

    def run_deployer(self, deploy_config):
        if self.Parameters.use_testing_deployer:
            attrs = [
                ["released", ReleaseStatus.STABLE],
                ["released", ReleaseStatus.PRESTABLE],
                ["released", ReleaseStatus.TESTING],
            ]
            deployer_id = sdk2.Task.server.resource.read(
                type=upper_resources.FastDataDeployer,
                status=State.READY,
                attrs=json.dumps(attrs),
                any_attr=True,
                limit=1
            )["items"][0]["id"]
            deployer = sdk2.Resource[deployer_id]
        else:
            deployer = self.Parameters.deployer

        deployer_path = str(sdk2.ResourceData(deployer).path)

        config_path = 'standalone_deploy_config.json'
        with open(config_path, 'w') as config_fd:
            json.dump(deploy_config, config_fd, indent=2)

        cmd = [
            'stdbuf', '-o', 'L',
            deployer_path, '--config', config_path,
        ]

        deployer_env = os.environ.copy()
        deployer_env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_name)
        deployer_env['OAUTH_NANNY'] = sdk2.Vault.data(self.Parameters.nanny_token_name)
        if self.Parameters.infra_token_owner:
            deployer_env['INFRA_TOKEN'] = sdk2.Vault.data(self.Parameters.infra_token_owner, self.Parameters.infra_token_name)
        if self.Parameters.solomon_token_owner:
            deployer_env['SOLOMON_TOKEN'] = sdk2.Vault.data(self.Parameters.solomon_token_owner, 'SOLOMON_TOKEN')
        deployer_env['YT_READ_FROM_CACHE'] = '0'

        self.set_info('Starting deployer:\n{}'.format(' '.join(cmd)))
        with sdk2.helpers.ProcessLog(self, logger="deployer") as pl:
            process = subprocess.Popen(
                cmd,
                env=deployer_env,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
                close_fds=True,
            )
            process.stdin.close()
            try:
                while True:
                    line = process.stdout.readline()
                    if not line:
                        break
                    self.set_info(line.strip('\n'))
                result = process.wait()
                self.set_info('Deployer finished')

                if result:
                    raise errors.TaskFailure('Failed to deploy')
            finally:
                process.stdout.close()
        self.release_fast_data()
        self.set_info('Successfully deployed')
