# -*- coding: utf-8 -*-

import os

from sandbox.common import errors

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.common.types.task import Semaphores

from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault
from sandbox.projects.websearch.upper import resources as upper_resources
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker


class DeployRearrangeDataFastWithBstr(ExecutionTimeTracker):
    """
        Пушит быстрые данные для верхнего метапоиска с помощью search/tools/fast_data_deployment/deployer и tools/bstr
    """

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 0
        kill_timeout = 3600  # 1 hour

        use_last_released_bundle = sdk2.parameters.Bool("Use last released bundle", default=False)
        with use_last_released_bundle.value[False]:
            fast_data_bundle = sdk2.parameters.Resource(
                "Fast data bundle",
                resource_type=upper_resources.RearrangeDataFastBundle,
                required=True,
            )
        cypress_dir = sdk2.parameters.String("Cypress directory for communication on locke", required=True)
        rollback_mode = sdk2.parameters.Bool("Rollback to fast data released with upper binary in case of fail", default=False)
        force_mode = sdk2.parameters.Bool("Continue to deploy even if deployment fails on some instances", default=False)
        operating_degrade_level = sdk2.parameters.Float("Operating degrade level", default=None)
        nanny_services = sdk2.parameters.List("Nanny services to deploy on", required=True)

        deployer = ResourceWithLastReleasedValueByDefault(
            "search/tools/fast_data_deployment/deployer binary",
            resource_type=upper_resources.FastDataDeployer,
            required=True,
        )
        bstr = ResourceWithLastReleasedValueByDefault(
            "tools/bstr binary",
            resource_type=upper_resources.FastDataBstr,
            required=True,
        )

        with sdk2.parameters.Group('Vault'):
            yt_token_name = sdk2.parameters.String('YT token name', required=True)
            nanny_token_name = sdk2.parameters.String('Nanny token name', required=True)

    class Context(ExecutionTimeTracker.Context):
        pass

    @property
    def stage_name(self):
        return 'deploy'

    def on_enqueue(self):
        super(DeployRearrangeDataFastWithBstr, self).on_enqueue()
        if not self.Requirements.semaphores:
            self.Requirements.semaphores = Semaphores(
                acquires=[
                    Semaphores.Acquire(
                        name='lock-{}_standalone-deployer'.format(service),
                        weight=2,
                        capacity=2
                    )
                    for service in self.Parameters.nanny_services
                ],
            )

    def on_execute(self):
        fast_data_bundle = self.Parameters.fast_data_bundle
        if self.Parameters.use_last_released_bundle:
            fast_data_bundle = sdk2.Resource[
                ResourceWithLastReleasedValueByDefault(resource_type=upper_resources.RearrangeDataFastBundle).default_value
            ]

        deployer_executable = str(sdk2.ResourceData(self.Parameters.deployer).path)
        bstr_executable = str(sdk2.ResourceData(self.Parameters.bstr).path)
        progress_bar = sdk2.service_resources.TaskCustomLogs(self, 'progress.bar', str(os.path.abspath('progress.bar')))

        self.set_info('Version to deploy: {vertical}, v.{version}'.format(
            vertical=fast_data_bundle.vertical.upper(), version=fast_data_bundle.version
        ))

        cmd = [
            'stdbuf',
            '-o',
            'L',
            deployer_executable,
            '--cluster', 'locke',
            '--cypress-dir', self.Parameters.cypress_dir,
            '--file', fast_data_bundle.skynet_id,
            '--file-version', str(fast_data_bundle.version),
            '--bstr-binary', bstr_executable,
            '--progress-bar', str(progress_bar.path),
            '--verbose',
        ]
        if self.Parameters.rollback_mode:
            cmd += ['--rollback-mode']
        if self.Parameters.force_mode:
            cmd += ['--force']
        if self.Parameters.operating_degrade_level:
            cmd += ['--operating-degrade-level', str(self.Parameters.operating_degrade_level)]
        cmd += self.Parameters.nanny_services

        deployer_env = os.environ.copy()
        deployer_env['YT_TOKEN'] = sdk2.Vault.data(self.Parameters.yt_token_name)
        deployer_env['OAUTH_NANNY'] = sdk2.Vault.data(self.Parameters.nanny_token_name)

        self.set_info('Starting deployer: {}'.format(' '.join(cmd)))
        with sdk2.helpers.ProcessLog(self, logger="deployer") as pl:
            process = subprocess.Popen(
                cmd,
                env=deployer_env,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
                close_fds=True,
            )
            process.stdin.close()
            try:
                while True:
                    line = process.stdout.readline()
                    if not line:
                        break
                    self.set_info(line.strip('\n'))
                result = process.wait()
                self.set_info('Deployer finished')

                if result:
                    raise errors.TaskFailure('Failed to deploy fast data')
            finally:
                process.stdout.close()
        self.set_info('Successfully deployed')
