# -*- coding: utf-8 -*-

import datetime
import yaml

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sdk2 import svn

from sandbox.common import errors
from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.task import Status
from sandbox.projects.websearch.upper import resources as upper_resources

from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault
from sandbox.projects.websearch.upper.fast_data.BuildRearrangeDataFast import BuildRearrangeDataFast
from sandbox.projects.websearch.upper.fast_data.DeployFastData import DeployFastData
from sandbox.projects.websearch.upper.fast_data.DeployRearrangeDataFastWithBstr import DeployRearrangeDataFastWithBstr
from sandbox.projects.websearch.upper.fast_data.ExecutionTimeTracker import ExecutionTimeTracker
from sandbox.projects.websearch.upper.fast_data.TestRearrangeDataFast import TestRearrangeDataFast


VERTICAL_TO_YT_TOKEN_NAME = {
    'web': 'yt_token_for_robot-video-fastdata',
    'video': 'yt_token_for_robot-video-fastdata',
}

VERTICAL_TO_NANNY_TOKEN_NAME = {
    'web': 'nanny_token_for_robot-upper-fastdata',
    'video': 'nanny_token_for_robot-video-fastdata',
}

VERTICAL_TO_BETA_PARAMETERS = {
    'web': {
        'name': 'noapache-web-fast-data',
        'template_name': 'noapache-web-fastdata',
        'suffix': 'fast-data',
        'parent_id': 'hamster_noapache_vla_web_yp',
        'rps': 10
    },
    'video': {
        'name': 'noapache-video-fast-data',
        'template_name': 'noapache-video-fastdata',
        'suffix': 'fast-data',
        'parent_id': 'hamster_noapache_vla_video_yp',
        'rps': 6
    },
}


class ReleaseRearrangeDataFast(ExecutionTimeTracker):
    """
        Собирает, тестит, релизит, деплоит быстрые данные для верхнего метапоиска
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 50  # Mb
        environments = [
            environments.PipEnvironment('python-statface-client', version="0.154.0", custom_parameters=["requests==2.18.4"]),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass  # do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        build_system = build_parameters.BuildSystem()
        use_yt_store_for_build = sdk2.parameters.Bool("YT store for build", default=True)

        with sdk2.parameters.RadioGroup("Test mode") as test_mode:
            test_mode.values['yappy'] = test_mode.Value('Shoot noapache on Yappy beta', default=True)
            test_mode.values['local'] = test_mode.Value('Shoot local noapache')

        use_new_deployer = sdk2.parameters.Bool("Use new deployer", default=False)

        with use_new_deployer.value[True]:
            with sdk2.parameters.RadioGroup("Deployer mode") as deployer_mode:
                deployer_mode.values['standalone'] = deployer_mode.Value('Run deployer in task', default=True)
                deployer_mode.values['nanny_service'] = deployer_mode.Value('Use nanny service deployer')

                with deployer_mode.value['nanny_service']:
                    nanny_service = sdk2.parameters.String('Deployer Nanny service name', required=True)

            push_blank_resource_on_activate = sdk2.parameters.Bool("Push blank resource when activating", default=False)
            prepare_config = sdk2.parameters.JSON("Prepare config")
            deploy_config = sdk2.parameters.JSON("Deploy config", required=True)

        with use_new_deployer.value[False]:
            operating_degrade_level = sdk2.parameters.Float("Operating degrade level", default=None)
            test_services_config = sdk2.parameters.String("Required services to continue", multiline=True)
            services_config = sdk2.parameters.String("Services config", multiline=True)

        with sdk2.parameters.RadioGroup('Vertical') as vertical:
            vertical.values['web'] = vertical.Value('WEB', default=True)
            vertical.values['video'] = vertical.Value('VIDEO')

        report_service_name = sdk2.parameters.String("Name of service used to report release time (do not report if empty)")

        yt_store_token_owner = sdk2.parameters.String("YT_STORE_TOKEN owner")
        infra_token_owner = sdk2.parameters.String("INFRA_TOKEN owner")
        solomon_token_owner = sdk2.parameters.String("SOLOMON_TOKEN owner")
        statface_token_owner = sdk2.parameters.String("STATFACE_TOKEN owner")
        yappy_token_owner = sdk2.parameters.String("YAPPY_TOKEN owner")
        arc_token_owner = sdk2.parameters.String("ARC_TOKEN owner")

        with sdk2.parameters.Output:
            fast_data = sdk2.parameters.Resource(
                "New Rearrange Data Fast",
                resource_type=upper_resources.RearrangeDataFast,
                required=True,
            )
            fast_data_bundle = sdk2.parameters.Resource(
                "New Rearrange Data Fast bundle",
                resource_type=upper_resources.RearrangeDataFastBundle,
                required=True,
            )
            last_released_data = sdk2.parameters.Resource(
                "Fallback Rearrange Data Fast bundle",
                resource_type=upper_resources.RearrangeDataFastBundle,
            )
            successfully_deployed = sdk2.parameters.Bool("Successfully deployed", required=True)
            successfully_rolled_back = sdk2.parameters.Bool("Successfully rolled back")

    class Context(ExecutionTimeTracker.Context):
        service_group_name_by_task_id = dict()
        deployment_tasks = list()

    @property
    def stage_name(self):
        return 'release'

    def on_execute(self):
        with self.memoize_stage.build:
            rearrange_data_fast_url = svn.Arcadia.trunk_url(upper_resources.RearrangeDataFastBundle.arcadia_build_path)
            self.Context.revision = int(utils.svn_last_change(rearrange_data_fast_url))

            self.Parameters.last_released_data = upper_resources.RearrangeDataFastBundle.find(
                attrs={'released': ReleaseStatus.STABLE, 'vertical': self.Parameters.vertical}
            ).order(-sdk2.Resource.id).first()

            if self.Parameters.last_released_data is not None:
                self.Context.previous_data_revision = int(self.Parameters.last_released_data.revision)
                self.set_info('Last released data revision: {}'.format(self.Context.previous_data_revision))
            else:
                self.Context.previous_data_revision = 0
            self.set_info('Current revision: {}'.format(self.Context.revision))
            self.set_info('Change diff: https://a.yandex-team.ru/arc/diff/trunk/arcadia/search/web/rearrs_upper/rearrange.fast?prevRev={}&rev={}'.format(
                self.Context.previous_data_revision, self.Context.revision
            ))

            if self.Parameters.last_released_data is not None and self.Context.revision <= self.Parameters.last_released_data.revision:
                self.set_info(
                    'This or newer version of fast data has already been released: {}'.format(lb.resource_link(self.Parameters.last_released_data.id)),
                    do_escape=False
                )
                self.set_info('Build and release current version anyway until completely in production')
                self.Context.build_task_id = self.Parameters.last_released_data.task_id
                self.set_info('Build Rearrange Data Fast: {}'.format(lb.task_link(self.Context.build_task_id)), do_escape=False)
            else:
                self.Context.build_task_id = BuildRearrangeDataFast(
                    self,
                    description='Rearrange Data Fast ({vertical}) r{revision}'.format(
                        vertical=self.Parameters.vertical.upper(), revision=self.Context.revision
                    ),
                    arcadia_url=svn.Arcadia.trunk_url(revision=self.Context.revision),
                    vertical=self.Parameters.vertical,
                    ya_yt_store=self.Parameters.use_yt_store_for_build,
                    ya_yt_proxy="arnold",
                    ya_yt_dir="//home/search-runtime/fast-data/cache",
                    ya_yt_token_vault_owner=self.Parameters.yt_store_token_owner,
                    ya_yt_put=True,
                    arc_token_owner=self.Parameters.arc_token_owner,
                ).enqueue().id
                self.set_info('Build Rearrange Data Fast: {}'.format(lb.task_link(self.Context.build_task_id)), do_escape=False)

        with self.memoize_stage.arc_tests:
            self.Context.arc_tests_task_id = TestRearrangeDataFast(
                self,
                run_arc=True,
                arcadia_url=svn.Arcadia.trunk_url(revision=self.Context.revision),
                yt_store_token_owner=self.Parameters.yt_store_token_owner,
            ).enqueue().id
            self.set_info('Start Arcadia tests for new Rearrange Data Fast: {}'.format(
                lb.task_link(self.Context.arc_tests_task_id)
            ), do_escape=False)

            raise sdk2.WaitTask(self.Context.build_task_id, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_build:
            build_task = sdk2.Task[self.Context.build_task_id]
            if build_task.status not in Status.Group.SUCCEED:
                self.set_info('Failed to build Rearrange Data Fast on {} revision: {}'.format(
                    self.Context.revision, lb.task_link(self.Context.build_task_id)
                ), do_escape=False)
                raise errors.TaskFailure('Failed to build')
            self.Parameters.fast_data = build_task.Parameters.data_resource
            self.Parameters.fast_data_bundle = build_task.Parameters.bundle_resource
            self.set_info('New Rearrange Data Fast version: {}'.format(self.Parameters.fast_data.version))
            self.Parameters.description += ' ({vertical}, release v.{version})'.format(
                vertical=self.Parameters.vertical.upper(), version=self.Parameters.fast_data.version
            )

        if self.Parameters.test_mode == 'yappy':
            with self.memoize_stage.shoot_yappy_beta:
                beta_parameters = VERTICAL_TO_BETA_PARAMETERS[self.Parameters.vertical]
                self.Context.shoot_yappy_beta_task_id = TestRearrangeDataFast(
                    self,
                    run_yappy_beta=True,
                    beta_name=beta_parameters['name'],
                    template_name=beta_parameters['template_name'],
                    suffix=beta_parameters['suffix'],
                    parent_id=beta_parameters['parent_id'],
                    fast_data_bundle=self.Parameters.fast_data_bundle,
                    rps=beta_parameters['rps'],
                    use_new_deployer=True,
                    max_fail_rate=0.05,
                    yt_token_name=VERTICAL_TO_YT_TOKEN_NAME[self.Parameters.vertical],
                    nanny_token_name=VERTICAL_TO_NANNY_TOKEN_NAME[self.Parameters.vertical],
                    yappy_token_owner=self.Parameters.yappy_token_owner,
                ).enqueue().id
                self.set_info('Shoot noapache on {} with new Rearrange Data Fast: {}'.format(
                    beta_parameters['name'],
                    lb.task_link(self.Context.shoot_yappy_beta_task_id)
                ), do_escape=False)
        elif self.Parameters.test_mode == 'local':
            with self.memoize_stage.shoot_local:
                self.Context.shoot_yappy_beta_task_id = TestRearrangeDataFast(
                    self,
                    run_noapache=True,
                    fast_data=self.Parameters.fast_data,
                ).enqueue().id
                self.set_info('Shoot local noapache with new Rearrange Data Fast: {}'.format(
                    lb.task_link(self.Context.shoot_yappy_beta_task_id)
                ), do_escape=False)

        with self.memoize_stage.prepare:
            if self.Parameters.use_new_deployer and self.Parameters.prepare_config:
                self.Context.preparation_tasks = self.prepare(self.Parameters.fast_data_bundle, self.Parameters.prepare_config)
            else:
                self.Context.preparation_tasks = []

        with self.memoize_stage.save_test_tasks:
            self.Context.test_tasks = [self.Context.arc_tests_task_id, self.Context.shoot_yappy_beta_task_id]

        with self.memoize_stage.start_deploy_on_nanny_service:
            if self.Parameters.use_new_deployer and self.Parameters.deployer_mode == 'nanny_service':
                self.Context.deployment_tasks = self.deploy(
                    self.Parameters.fast_data_bundle,
                    deploy_config=self.Parameters.deploy_config,
                    use_new_deployer=True,
                    wait_tasks=self.Context.test_tasks,
                )

        with self.memoize_stage.wait_tests:
            raise sdk2.WaitTask(self.Context.test_tasks, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_tests:
            has_errors = False
            error_messages = {
                self.Context.arc_tests_task_id: 'Arcadia tests failed: {}',
                self.Context.shoot_yappy_beta_task_id: 'Failed to run noapache on with this fast data: {}',
            }
            for test_task_id, error_message in error_messages.items():
                if sdk2.Task[test_task_id].status not in Status.Group.SUCCEED:
                    self.set_info(error_message.format(
                        lb.task_link(test_task_id)
                    ), do_escape=False)
                    has_errors = True

            if has_errors:
                raise errors.TaskFailure('Tests failed')

        with self.memoize_stage.stop_prepare:
            for preparation_task in self.Context.preparation_tasks:
                try:
                    sdk2.Task[preparation_task].stop()
                except:
                    pass

        with self.memoize_stage.release:
            self.server.release(
                task_id=self.Context.build_task_id,
                type=ReleaseStatus.STABLE,
                subject="Rearrange Data Fast auto-release"
            )

        if not self.Parameters.use_new_deployer and self.Parameters.test_services_config:
            with self.memoize_stage.test_deploy:
                if self.Context.test_preparation_tasks:
                    if not all(sdk2.Task[task_id].status in Status.Group.SUCCEED for task_id in self.Context.test_preparation_tasks):
                        raise errors.TaskFailure('Failed to prepare new fast data on test services.')
                services_config = yaml.load(self.Parameters.test_services_config)
                self.set_info('Try to activate new Rearrange Data Fast on test services: {}'.format(', '.join(services_config.keys())))
                self.Context.test_activation_tasks = self.deploy(
                    self.Parameters.fast_data_bundle,
                    self.Parameters.test_services_config,
                )
                raise sdk2.WaitTask(self.Context.test_activation_tasks, Status.Group.FINISH | Status.Group.BREAK)

            with self.memoize_stage.check_test_deploy:
                self.Context.successfully_deployed_on_test_services = self.check_deploy(self.Context.test_activation_tasks)

            if not self.Context.successfully_deployed_on_test_services:
                self.Parameters.successfully_deployed = False

                with self.memoize_stage.rollback_test_deployment:
                    self.Context.rollback_tasks = self.rollback(self.Parameters.test_services_config)
                    raise sdk2.WaitTask(self.Context.rollback_tasks, Status.Group.FINISH | Status.Group.BREAK)

                with self.memoize_stage.check_rollback_test_deployment:
                    self.Context.successfully_rolled_back_on_test_service = self.check_rollback(self.Context.rollback_tasks)

                raise errors.TaskFailure('Failed to deploy new fast data on test services. Rollback to previous release {}'.format(
                    'succeeded' if self.Context.successfully_rolled_back_on_test_service else 'failed'
                ))

        with self.memoize_stage.deploy:
            if not self.Context.deployment_tasks:
                if self.Parameters.use_new_deployer:
                    self.Context.deployment_tasks = self.deploy(self.Parameters.fast_data_bundle, deploy_config=self.Parameters.deploy_config, use_new_deployer=True)
                else:
                    self.Context.deployment_tasks = self.deploy(self.Parameters.fast_data_bundle, services_yaml_config=self.Parameters.services_config)
            raise sdk2.WaitTask(self.Context.deployment_tasks, Status.Group.FINISH | Status.Group.BREAK)

        with self.memoize_stage.check_deploy:
            self.Parameters.successfully_deployed = self.check_deploy(self.Context.deployment_tasks)

        if not self.Parameters.successfully_deployed:
            """
            with self.memoize_stage.rollback:
                if self.Parameters.use_new_deployer:
                    self.Context.rollback_tasks = self.rollback(deploy_config=self.Parameters.deploy_config, use_new_deployer=True)
                else:
                    self.Context.rollback_tasks = self.rollback(services_yaml_config='\n'.join([
                        self.Parameters.services_config, self.Parameters.test_services_config
                    ]))
                raise sdk2.WaitTask(self.Context.rollback_tasks, Status.Group.FINISH | Status.Group.BREAK)

            with self.memoize_stage.check_rollback:
                self.Parameters.successfully_rolled_back = self.check_rollback(self.Context.rollback_tasks)
            """

            raise errors.TaskFailure('Failed to deploy new fast data.')

        if self.Parameters.last_released_data is not None and self.Parameters.last_released_data.task_id != self.id:
            sdk2.Task[self.Parameters.last_released_data.task_id].mark_released_resources(ReleaseStatus.STABLE, ttl=90)

    def deploy_new(self, fast_data_bundle, deploy_config, action='activate', deployer_mode=None, wait_tasks=[]):
        self.set_info('{} {} (version {})'.format(
            action.capitalize(),
            lb.resource_link(fast_data_bundle.id, 'Rearrange Data Fast'),
            fast_data_bundle.version
        ), do_escape=False)

        services = ', '.join(deploy_config.get('services', {}).keys())
        if not services:
            self.set_info('No services to {} on'.format(action))
            return []

        deploy_config.setdefault('deploy', {}).setdefault('infra', {})
        deploy_config['deploy']['infra'].setdefault('title', "Release Fast Data v. {}".format(fast_data_bundle.version))
        deploy_config['deploy']['infra'].setdefault('description', [
            "Sandbox release task: https://sandbox.yandex-team.ru/task/{}".format(self.id)
        ] + (["Sandbox releases scheduler: https://sandbox.yandex-team.ru/scheduler/{}".format(self.scheduler)] if self.scheduler else []) + [
            "Change diff: https://a.yandex-team.ru/arc/diff/trunk/arcadia/search/web/rearrs_upper/rearrange.fast?prevRev={}&rev={}".format(
                self.Context.previous_data_revision, fast_data_bundle.revision
            )
        ])

        blank_fast_data_bundle = None
        if self.Parameters.push_blank_resource_on_activate:
            blank_fast_data_bundle = ResourceWithLastReleasedValueByDefault(resource_type=upper_resources.RearrangeDataFastBundleBlank).default_value

        deployment_task = DeployFastData(
            self,
            {
                "cores": 1 if deployer_mode == 'nanny_service' else 2,
            },
            description='Rearrange Data Fast ({vertical}) auto-release for {services}'.format(
                vertical=self.Parameters.vertical.upper(), services=services,
            ),
            kill_timeout=60 * 15,  # 15 minutes
            fast_data_bundle=fast_data_bundle,
            blank_fast_data_bundle=blank_fast_data_bundle,
            deployer_mode=deployer_mode,
            use_testing_deployer=False,
            nanny_service=self.Parameters.nanny_service,
            tasks_to_wait=wait_tasks,
            only_prepare=action == 'prepare',
            deploy_config=deploy_config,
            yt_token_name=VERTICAL_TO_YT_TOKEN_NAME[self.Parameters.vertical],
            infra_token_owner=self.Parameters.infra_token_owner,
            solomon_token_owner=self.Parameters.solomon_token_owner,
            nanny_token_name=VERTICAL_TO_NANNY_TOKEN_NAME[self.Parameters.vertical],
        ).enqueue()
        self.Context.service_group_name_by_task_id[str(deployment_task.id)] = services

        return [deployment_task.id]

    def prepare(self, fast_data_bundle, prepare_config):
        return self.deploy_new(fast_data_bundle, prepare_config, action='prepare', deployer_mode='standalone')

    def deploy(self, fast_data_bundle, services_yaml_config='', deploy_config=None, rollback=False, use_new_deployer=False, wait_tasks=[]):
        if rollback:
            self.set_info('Rollback to {}'.format(
                lb.resource_link(fast_data_bundle.id, 'v.{}:r{}'.format(fast_data_bundle.version, fast_data_bundle.revision))
            ), do_escape=False)
        self.set_info('Deploy {} (version {})'.format(
            lb.resource_link(fast_data_bundle.id, 'Rearrange Data Fast'),
            fast_data_bundle.version
        ), do_escape=False)

        deployment_tasks = []
        if use_new_deployer:
            deployment_tasks.extend(self.deploy_new(fast_data_bundle, deploy_config, action='activate', deployer_mode=self.Parameters.deployer_mode, wait_tasks=wait_tasks))
        else:
            services_config = yaml.load(services_yaml_config)
            if not services_config:
                self.set_info('No services to deploy on')
                return deployment_tasks
            for services_group_name, services in services_config.items():
                deployment_task = DeployRearrangeDataFastWithBstr(
                    self,
                    description='Rearrange Data Fast ({vertical}) auto-release for {services}'.format(
                        vertical=self.Parameters.vertical.upper(), services=services_group_name,
                    ),
                    fast_data_bundle=fast_data_bundle,
                    cypress_dir='//home/search-runtime/fast-data',
                    rollback_mode=rollback,
                    force_mode=rollback,
                    operating_degrade_level=self.Parameters.operating_degrade_level,
                    nanny_services=services,
                    yt_token_name=VERTICAL_TO_YT_TOKEN_NAME[self.Parameters.vertical],
                    nanny_token_name=VERTICAL_TO_NANNY_TOKEN_NAME[self.Parameters.vertical],
                ).enqueue()
                self.Context.service_group_name_by_task_id[str(deployment_task.id)] = services_group_name
                deployment_tasks.append(deployment_task.id)

        self.set_info('Deployment started on {}'.format(
            ', '.join(self._get_task_links(deployment_tasks))
        ), do_escape=False)

        return deployment_tasks

    def check_deploy(self, deployment_tasks):
        if not deployment_tasks:
            return True
        succeeded_deployments = set(filter(
            lambda task_id: sdk2.Task[task_id].status in Status.Group.SUCCEED, deployment_tasks
        ))
        if succeeded_deployments:
            self.set_info('Deployment successfully finished on {}'.format(
                ', '.join(self._get_task_links(succeeded_deployments))
            ), do_escape=False)

        failed_deployments = set(deployment_tasks) - succeeded_deployments
        if failed_deployments:
            self.set_info('Deployment failed on {}'.format(
                ', '.join(self._get_task_links(failed_deployments))
            ), do_escape=False)
        return len(failed_deployments) == 0

    def rollback(self, services_yaml_config='', deploy_config=None, use_new_deployer=False):
        self.server.release(
            task_id=self.Context.build_task_id,
            type=ReleaseStatus.CANCELLED,
            subject="Rearrange Data Fast auto-release"
        )
        if self.Parameters.last_released_data is None:
            self.Parameters.last_released_data = self.Parameters.fast_data_bundle
        return self.deploy(self.Parameters.last_released_data, services_yaml_config=services_yaml_config, deploy_config=deploy_config, rollback=True, use_new_deployer=use_new_deployer)

    def check_rollback(self, rollback_tasks):
        successfully_rolled_back = self.check_deploy(rollback_tasks)
        if not successfully_rolled_back:
            self.set_info('Failed to rollback')
        return successfully_rolled_back

    def _get_task_links(self, task_ids):
        return map(lambda task_id: lb.task_link(task_id, self.Context.service_group_name_by_task_id[str(task_id)]), task_ids)

    def _get_statface_report(self, report_path):
        import statface_client
        statface_token = sdk2.Vault.data(self.Parameters.statface_token_owner, name='STATFACE_TOKEN')
        sf_client = statface_client.StatfaceClient(host=statface_client.STATFACE_PRODUCTION, oauth_token=statface_token)
        return sf_client.get_report(report_path)

    def on_success(self, prev_status):
        super(ReleaseRearrangeDataFast, self).on_success(prev_status)

        if self.Parameters.report_service_name:
            self.report_stats()

    def report_stats(self):
        import statface_client

        report = {
            'fielddate': datetime.datetime.now().strftime('%Y-%m-%d %H:%M:00'),
            'service': self.Parameters.report_service_name,
            'vertical': self.Parameters.vertical,
        }
        for stage in self.stages():
            report[stage] = self.stage_time(stage) / 60.0

        sf_report = self._get_statface_report('Yandex/Fast Data/Release Stages Time')
        sf_report.upload_data(scale=statface_client.constants.MINUTELY_SCALE, data=report)
