import re
import requests
import logging

logger = logging.getLogger(__name__)


class JenkinsApiError(RuntimeError):
    def __init__(self, message):
        super(RuntimeError, self).__init__(message)


def _call_and_log(method, url, *args, **kwargs):
    logging.debug("Going to call {url}".format(url=url))
    response = method(url=url, *args, **kwargs)
    logging.debug("Got response from {url} with code {status_code}".format(url=url, status_code=response.status_code))
    return response


def get_build_job_url(host, job_name):
    return "{host}/job/{job_name}/build".format(host=host, job_name=job_name)


def get_build_with_parameters_job_url(host, job_name):
    return "{host}/job/{job_name}/buildWithParameters".format(host=host, job_name=job_name)


def get_queue_item_url(host, queue_item_id):
    return "{host}/queue/item/{queue_item_id}/api/json".format(host=host, queue_item_id=queue_item_id)


def get_job_info_url(host, job_name, job_id):
    return "{host}/job/{job_name}/{job_id}/api/json".format(host=host, job_name=job_name, job_id=job_id)


class JOB_TAB:
    STATUS = ""
    STDOUT = "console"
    STDOUT_PLAIN = "consoleText"


def get_job_url(host, job_name, job_id, tab_name=""):
    return "{host}/job/{job_name}/{job_id}/{tab_name}".format(
        host=host,
        job_name=job_name,
        job_id=job_id,
        tab_name=(tab_name or "")
    )


def get_job_stdout_url(host, job_name, job_id, plain_text=True):
    return get_job_url(
        host=host,
        job_name=job_name,
        job_id=job_id,
        tab_name=JOB_TAB.STDOUT_PLAIN if plain_text else JOB_TAB.STDOUT
    )


def get_job_blueocean_url(host, job_name, job_id, tab_name=""):
    return "{host}/blue/organizations/jenkins/{job_name}/detail/{job_name}/{job_id}/{tab_name}".format(
        host=host,
        job_name=job_name,
        job_id=job_id,
        tab_name=tab_name
    )


def build_job(host, auth, job_name, parameters=None):
    """
    :returns str created queue_item_id
    """

    if parameters is None:
        response = _call_and_log(
            method=requests.post,
            url=get_build_job_url(host, job_name),
            auth=auth
        )
    else:
        response = _call_and_log(
            method=requests.post,
            url=get_build_with_parameters_job_url(host, job_name),
            auth=auth,
            data=parameters
        )

    if response.status_code != 201:
        raise JenkinsApiError(
            "Jenkins build returned non-201 response: {status_code} (body: '{body}')"
                .format(status_code=response.status_code, body=response.text)
        )

    response_location = response.headers["Location"]
    if not response_location:
        raise JenkinsApiError(
            "Jenkins build returned response without Location header (body: '{body}')"
                .format(body=response.text)
        )

    return _extract_queue_item_id_from_location_url(location=response_location, host=host)


def _extract_queue_item_id_from_location_url(location, host):
    if not location.startswith(host):
        raise JenkinsApiError(
            "Queue location url doesn't start with host: '{location}' and '{host}'"
                .format(location=location, host=host)
        )

    location_path = location[len(host) :]
    location_re = re.compile(r"^/queue/item/(?P<queue_item_id>\d+)/?$")
    location_match = location_re.match(location_path)
    if location_match is None:
        raise JenkinsApiError(
            "Queue location url doesn't match to re: {location_path}".format(location_path=location_path)
        )

    return location_match.group("queue_item_id")


def get_queue_item(host, auth, queue_item_id):
    response = _call_and_log(
        method=requests.post,
        url=get_queue_item_url(host=host, queue_item_id=queue_item_id),
        auth=auth
    )
    response.raise_for_status()
    return response.json()


def get_job_info(host, auth, job_name, job_id):
    response = _call_and_log(
        method=requests.post,
        url=get_job_info_url(host=host, job_name=job_name, job_id=job_id),
        auth=auth
    )
    response.raise_for_status()
    return response.json()


def download_job_stdout(host, auth, job_name, job_id):
    response = _call_and_log(
        method=requests.get,
        url=get_job_stdout_url(host=host, job_name=job_name, job_id=job_id, plain_text=True),
        auth=auth
    )
    response.raise_for_status()
    return response.text()
