# -*- coding: utf-8 -*-
import logging
import re

import datetime
import requests
from requests import RequestException

from sandbox import sdk2
from sandbox.common.errors import TemporaryError

from sandbox.sandboxsdk.environments import PipEnvironment

from urllib import quote


def escape(s):
    """Escape a URL including any /."""
    return quote(s.encode('utf-8'), safe='~')


TANKER_URL_TEMPLATE = 'https://tanker-api.yandex-team.ru/projects/export/tjson/?project-id={}&branch-id={}'
INTICKET_URL_TEMPLATE_KEY = 'https://tanker.yandex-team.ru?project={}&branch={}&keyset={}&key={}'
INTICKET_URL_TEMPLATE_KEY_LANG = 'https://tanker.yandex-team.ru?project={}&branch={}&keyset={}&key={}&focus={}'


class TankerKeysDownloadError(TemporaryError):
    pass


class CheckWebmasterTankerUrls(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment('startrek_client')]

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 30
        kill_timeout = 3600
        project = sdk2.parameters.String('Project in tanker', required=True)
        branches = sdk2.parameters.List('Source tanker branches', required=True)
        st_queue = sdk2.parameters.String('Очередь в стартреке', required=True, default='WMC')
        vault_item_name = sdk2.parameters.String('OAuth token (st+tanker) vault name',
                                                 required=True,
                                                 default='tanker-st-token')
        vault_item_owner = sdk2.parameters.String('OAuth token (st+tanker) vault owner', required=True, default='WEBMASTER')
        url_reg_exp = sdk2.parameters.String('Url RegExp', required=True, default='https?://[a-z0-9./_#-]*')
        ignore_reg_exp = sdk2.parameters.List('Ignore RegExp List')
        context_reg_exp_list = sdk2.parameters.List('Context 200 RegExp List',
                                                    description="Регулярки по которым парсятся ссылки из контекста",
                                                    default='https?://[a-z0-9./_#-]*')
        st_followers = sdk2.parameters.List('Followers')
        st_tags = sdk2.parameters.List('Tags')
        is_debug = sdk2.parameters.Bool('Is Debug')

    def get_keysets(self, branch):
        try:
            resp = requests.get(TANKER_URL_TEMPLATE.format(self.Parameters.project, branch), timeout=300)
        except RequestException as e:
            raise TankerKeysDownloadError(e)
        if resp.status_code >= 500:
            raise TankerKeysDownloadError(resp.status_code)
        elif resp.status_code != 200:
            resp.raise_for_status()

        return resp.json()['keysets']

    def fill_result(self, result, path_list, url, status_code, req):
        if status_code == 200:
            if len(req.history) > 0:
                self.fill_result(result, path_list, url, req.history[0].status_code, req)
            return

        if status_code == -1:
            result_arr = result['ignore']
        elif status_code == 0:
            result_arr = result['error']
        elif status_code >= 500:
            result_arr = result['5xx']
        elif status_code >= 400:
            result_arr = result['4xx']
        elif status_code >= 300:
            result_arr = result['3xx']
        else:
            result_arr = result['xxx']

        result_arr.append([path_list, status_code, url])

    def render_result(self, result_arr):
        content = ''
        for line in result_arr:
            path_list = line[0]
            status_code = str(line[1])
            url = line[2]
            content += '**status_code**: %%{}%%\n**url**: %%{}%%\n'.format(status_code, url)
            for path in path_list:
                branch = path[0]
                ks = path[1]
                key_encoded = path[2].encode('UTF-8')
                lang = path[3]
                content += '{}\n'.format(INTICKET_URL_TEMPLATE_KEY_LANG.format(self.Parameters.project, branch, ks, escape(key_encoded), lang))

        return content + '\n'

    def render_context_result(self, result_arr):
        content = ''
        for line in result_arr:
            path_list = line[0]
            url = line[1]
            content += '**url**: %%{}%%\n'.format(url)
            for path in path_list:
                branch = path[0]
                ks = path[1]
                key_encoded = path[2].encode('UTF-8')
                content += '{}\n'.format(INTICKET_URL_TEMPLATE_KEY.format(self.Parameters.project, branch, ks, escape(key_encoded)))

        return content + '\n'

    def on_execute(self):
        from startrek_client import Startrek

        branch_data = {}
        for branch in iter(self.Parameters.branches):
            branch_data[branch] = self.get_keysets(branch)

        result = {
            'ignore': [],
            'error': [],
            '3xx': [],
            '4xx': [],
            '5xx': [],
            'xxx': []
        }

        already = {}

        tanker_content_urls_list = []

        for branch in iter(branch_data):
            keysets = branch_data[branch]
            for ks in iter(keysets):
                keys = keysets[ks]['keys']
                for key in iter(keys):
                    langs = keys[key]['translations']
                    for l in langs:
                        path = [branch, ks, key, l]

                        if 'form' in langs[l]:
                            text = langs[l]['form']
                        else:
                            text = langs[l]['form1']

                        url_list = re.findall(self.Parameters.url_reg_exp, text)

                        for url in url_list:
                            # region check url
                            if url in already:
                                already[url]['path_list'].append(path)
                                continue

                            already[url] = {
                                'path_list': [path]
                            }

                            # region ignore
                            is_ignore = False
                            for re_str in self.Parameters.ignore_reg_exp:
                                if re.search(re_str, url):
                                    self.fill_result(result, already[url]['path_list'], url, -1, None)
                                    is_ignore = True
                                    break
                            if is_ignore:
                                continue
                            # endregion ignore

                            try:
                                req = requests.get(url)
                                self.fill_result(result, already[url]['path_list'], url, req.status_code, req)
                            except BaseException:
                                self.fill_result(result, already[url]['path_list'], url, 0, req)
                            # endregion check url

                    # region comment urls
                    for context_reg_exp in self.Parameters.context_reg_exp_list:
                        url_list = re.findall(context_reg_exp, keys[key]['info']['context'])
                        if len(url_list) > 0:
                            tanker_content_urls_list.append([[branch, ks, key], url_list])
                            continue
                    # endregion comment urls

        # region check comment urls

        tanker_content_200_urls_list = []
        already = {}

        for urls_line in tanker_content_urls_list:
            path = urls_line[0]
            urls = urls_line[1]

            for url in urls:
                if url in already:
                    already[url]['path_list'].append(path)
                    continue

                already[url] = {
                    'path_list': [path]
                }
                try:
                    req = requests.get(url)
                    if req.status_code == 200:
                        tanker_content_200_urls_list.append([already[url]['path_list'], url])
                except BaseException:
                    None
        # endregion check comment urls

        today = datetime.date.today()
        date_in_title = today.strftime('%Y-%m-%d')
        deadline = (today + datetime.timedelta(days=4)).strftime('%Y-%m-%d')

        content = '**__Замените ссылки на актуальные:__**\n\n'

        if len(result['error']) > 0:
            render_str = '**error**\n' + self.render_result(result['error'])
            content += render_str
            if self.Parameters.is_debug:
                logging.info('---- error ----')
                logging.info(render_str)

        if len(result['5xx']) > 0:
            render_str = '**5xx**\n' + self.render_result(result['5xx'])
            content += render_str
            if self.Parameters.is_debug:
                logging.info('---- 5xx ----')
                logging.info(render_str)

        if len(result['4xx']) > 0:
            render_str = '**4xx**\n' + self.render_result(result['4xx'])
            content += render_str
            if self.Parameters.is_debug:
                logging.info('---- 4xx ----')
                logging.info(render_str)

        if len(result['xxx']) > 0:
            render_str = '**xxx**\n' + self.render_result(result['xxx'])
            content += render_str
            if self.Parameters.is_debug:
                logging.info('---- xxx ----')
                logging.info(render_str)

        if len(result['3xx']) > 0:
            render_str = '<{**3xx**\n' + self.render_result(result['3xx']) + '}>'
            content += render_str
            if self.Parameters.is_debug:
                logging.info('---- 3xx ----')
                logging.info(render_str)

        if len(result['ignore']) > 0:
            render_str = '<{**ignore**\n'
            for line in result['ignore']:
                url = line[2]
                render_str += '**url**: %%{}%%\n'.format(url)
            render_str += '}>\n\n'

            content += render_str
            if self.Parameters.is_debug:
                logging.info('---- ignore ----')
                logging.info(render_str)

        if len(tanker_content_200_urls_list) > 0:
            render_str = '<{**Ссылки из контекста со статусом 200**\n' + self.render_context_result(tanker_content_200_urls_list) + '}>'
            content += render_str
            if self.Parameters.is_debug:
                logging.info('---- tanker_content_200_urls ----')
                logging.info(render_str)

        st_client = Startrek(token=sdk2.Vault.data(self.Parameters.vault_item_owner, self.Parameters.vault_item_name),
                                useragent=self.Parameters.vault_item_owner)

        related = st_client.issues.find('(queue: WMC components: "Релиз версии"  Status: Открыт)')
        links = []
        for rel_issue in related:
            links.append({'issue': rel_issue.key, 'relationship': 'relates'})

        if self.Parameters.is_debug:
            logging.info('---- followers ----')
            logging.info(self.Parameters.st_followers)
            logging.info('---- tags ----')
            logging.info(self.Parameters.st_tags)
            logging.info('---- links ----')
            logging.info(links)
        else:
            content += '\n\n\n**The ticket was created automatically by means of https://sandbox.yandex-team.ru/task/{}/view**'.format(self.id)
            ticket = st_client.issues.create(
                queue=self.Parameters.st_queue,
                summary='[TANKER URL CHECKER] — Поправить ссылки в танкере ({})'.format(date_in_title),
                type='task',
                description=content,
                tags=self.Parameters.st_tags,
                deadline=deadline,
                followers=self.Parameters.st_followers,
                links=links,
            )
            self.Context.ticket_id = ticket.key

            logging.info('Тикет {} создан(https://st.yandex-team.ru/{})'.format(ticket.key, ticket.key))
