# -*- coding: utf-8 -*-
import logging

import datetime
import requests
from requests import RequestException

from sandbox import sdk2
from sandbox.common.errors import TemporaryError

from sandbox.sandboxsdk.environments import PipEnvironment

from urllib import quote


def escape(s):
    """Escape a URL including any /."""
    return quote(s.encode('utf-8'), safe='~')


TANKER_URL_TEMPLATE = 'https://tanker-api.yandex-team.ru/projects/export/tjson/?project-id={}&branch-id={}'
STARTREK_URL = 'https://st-api.yandex-team.ru/v2/issues'
INTICKET_URL_TEMPLATE_BRANCH = 'https://tanker.yandex-team.ru?project={}&branch={}'
INTICKET_URL_TEMPLATE_KEYSET = 'https://tanker.yandex-team.ru?project={}&branch={}&keyset={}'
INTICKET_URL_TEMPLATE_KEY = 'https://tanker.yandex-team.ru?project={}&branch={}&keyset={}&key={}'
SERVICE_GROUP = "Webmaster"


class Language:
    def __init__(self, name, desc):
        self.name = name
        self.desc = desc

    def __repr__(self):
        return self.desc + ' (' + self.name + ')'


class TranslateDirection:
    def __init__(self, source_language, target_languages):
        self.source_language = source_language
        self.target_languages = target_languages


TRANSLATE_DIRECTIONS = {
    'en': TranslateDirection(
        Language('ru-RU', 'Русский/Russian'),
        [Language('en-US', 'Английский/English US')]
    ),
}


class StartrackTicketCreationError(TemporaryError):
    pass


class TankerKeysDownloadError(TemporaryError):
    pass


class CreateWebmasterTranslationTicket(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = [PipEnvironment('startrek_client')]

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 30
        kill_timeout = 3600
        project = sdk2.parameters.String('Project in tanker', required=True)
        branches = sdk2.parameters.List('Source tanker branches', required=True)
        st_queue = sdk2.parameters.String('Startrek queue to create a ticket', required=True, default='TRANSLATE')
        vault_item_name = sdk2.parameters.String('OAuth token (st+tanker) vault name',
                                                 required=True,
                                                 default='tanker-st-token')
        vault_item_owner = sdk2.parameters.String('OAuth token (st+tanker) vault owner', required=True, default='WEBMASTER')

    def get_keysets(self, branch):
        try:
            resp = requests.get(TANKER_URL_TEMPLATE.format(self.Parameters.project, branch),
                                timeout=300)
        except RequestException as e:
            raise TankerKeysDownloadError(e)
        if resp.status_code >= 500:
            raise TankerKeysDownloadError(resp.status_code)
        elif resp.status_code != 200:
            resp.raise_for_status()

        return resp.json()['keysets']

    def on_execute(self):
        from startrek_client import Startrek

        branch_data = dict()
        for branch in iter(self.Parameters.branches):
            branch_data[branch] = self.get_keysets(branch)
        result = dict()

        for branch in iter(branch_data):
            keysets = branch_data[branch]
            for ks in iter(keysets):
                keys = keysets[ks]['keys']
                for key in iter(keys):
                    langs = keys[key]['translations']
                    for l in langs:
                        if (l in TRANSLATE_DIRECTIONS) and (langs[l]['status'] in ('expired', 'requires_translation')):
                            if l not in result:
                                result[l] = {}
                            if branch not in result[l]:
                                result[l][branch] = {}
                            if ks not in result[l][branch]:
                                result[l][branch][ks] = []
                            result[l][branch][ks].append(key)

        if not result:
            logging.info('Nothing to translate')
            return

        today = datetime.date.today()
        date_in_title = today.strftime('%Y-%m-%d')
        deadline = (today + datetime.timedelta(days=4)).strftime('%Y-%m-%d')

        for l in result:
            content = (
                'Сервис (группа сервисов):\n'
                '{}\n\n'
                'Тип задачи:\n'
                'Перевод\n\n'
                'Вид текста:\n'
                'Интерфейс и формы\n\n'
                'Суть задачи:\n'
                'Перевод интерфейса {}\n\n'
                'Ссылки на ключи в Танкере:\n'
            ).format(
                SERVICE_GROUP,
                date_in_title
            )

            # index = 0
            branches = result[l]
            for branch in branches:
                # content += '\n* **Branch (({} {}))**\n'.format(INTICKET_URL_TEMPLATE_BRANCH.format(self.Parameters.project, branch), branch)
                for i, ks in enumerate(branches[branch], start=1):
                    if ks.startswith('support:desktop'):
                        continue
                    # index += 1
                    # content += '  {} (({} {}:{}))\n'.format(index, INTICKET_URL_TEMPLATE_KEYSET.format(self.Parameters.project, branch, ks), branch, ks)
                    for k in branches[branch][ks]:
                        kEncoded = k.encode('UTF-8')
                        # content += '    - (({} {}))\n'.format(INTICKET_URL_TEMPLATE_KEY.format(self.Parameters.project, branch, ks, escape(kEncoded)), kEncoded)
                        content += '{}\n'.format(INTICKET_URL_TEMPLATE_KEY.format(self.Parameters.project, branch, ks, escape(kEncoded)))

            content += (
                '\n'
                'Язык исходного текста:\n'
                '{}\n\n'
                'Язык перевода:\n'
                '{}\n\n'
                'Дедлайн:\n'
                '{}\n\n'
                'Обоснование дедлайна:\n'
                'Плановое обновление\n\n'
            ).format(
                TRANSLATE_DIRECTIONS[l].source_language,
                ','.join(map(str, TRANSLATE_DIRECTIONS[l].target_languages)),
                deadline
            )

            content += '**The ticket was created automatically by means of https://sandbox.yandex-team.ru/task/{}/view**'.format(self.id)

            st_client = Startrek(token=sdk2.Vault.data(self.Parameters.vault_item_owner, self.Parameters.vault_item_name),
                                 useragent=self.Parameters.vault_item_owner)

            related = st_client.issues.find('(queue: WMC components: "Релиз версии"  Status: Открыт)')
            links = []
            for rel_issue in related:
                links.append({'issue': rel_issue.key, 'relationship': 'relates'})

            ticket = st_client.issues.create(
                queue=self.Parameters.st_queue,
                summary=(
                    '[' + SERVICE_GROUP + '] — Перевод — ' + TRANSLATE_DIRECTIONS[l].source_language.name +
                    ' > ' + ','.join(map(lambda x: x.name, TRANSLATE_DIRECTIONS[l].target_languages)) + ' — Перевод интерфейса ' + date_in_title
                ),
                type='translation',
                description=content,
                assignee='robot-baron-cohen',
                tags=['upgrade'],
                sourceLanguage=TRANSLATE_DIRECTIONS[l].source_language.name,
                targetLanguage=list(map(lambda x: x.name, TRANSLATE_DIRECTIONS[l].target_languages)),
                serviceServiceGroup=SERVICE_GROUP,
                components=[50937],
                deadline=deadline,
                followers=['trifonova-j'],
                links=links,
                abcService=[{"id": 83}],
            )

            logging.info('Startrek response code: {}'.format(ticket.key))

            self.Context.ticket_id = ticket.key
