# coding: utf-8

import os
import logging

import datetime
import dateutil
import dateutil.parser
import sqlite3
import traceback

from sandbox.projects.common.nanny import nanny
from sandbox import sdk2
from sandbox.sandboxsdk import environments


class CacheInfo(object):
    def __init__(self, runtime, server, table_path, table_name, table_time):
        self.runtime = runtime
        self.server = server
        self.table_path = table_path
        self.table_name = table_name.replace(".", "_").replace("-", "_")
        self.table_time = table_time
        self.file_name = "cache_" + self.table_name

    def __str__(self):
        return str((
            self.runtime,
            self.server,
            self.table_path,
            self.table_name,
            self.table_time,
            self.file_name,
        ))


def convert_yt_stream(stream):
    for record in stream:
        part, part_size = record["subkey"].split("/")
        yield (record["key"], part, part_size, record["value"])


def checkout_yt_source(yt_client, resource_date, server_name, prefix, cache_sources):
    logging.info("checkout_yt_source: %s, %s, %s", resource_date, server_name, prefix)

    need_update = False
    for table in yt_client.search(prefix, node_type="table", attributes=["modification_time"], depth_bound=1):
        logging.info("table %s", table)
        table_time = dateutil.parser.parse(table.attributes["modification_time"]).replace(tzinfo=None)
        table_path = str(table)
        table_name = table_path.split("/")[-1]
        if table_time > resource_date:
            logging.info("checkout_yt_source - need update: %s", table)
            need_update = True
        cache_sources.append(CacheInfo("YT", server_name, table_path, table_name, table_time))

    logging.info("checkout_yt_source - done: %s", cache_sources)

    return need_update


def build_cache(yt_client, root, cache):
    import yt
    root.mkdir(exist_ok=True)

    path = os.path.join(str(root), cache.file_name)
    db = sqlite3.connect(path)
    cursor = db.cursor()

    cursor.execute("PRAGMA synchronous = 0")
    cursor.execute("CREATE TABLE IF NOT EXISTS cache (host TEXT NOT NULL, part TEXT NOT NULL, part_size INTEGER NOT NULL, mds_key TEXT NOT NULL)")
    cursor.execute("CREATE UNIQUE INDEX IF NOT EXISTS idx_cache ON cache(host, part)")

    stream = None
    if cache.runtime == "YT":
        stream = convert_yt_stream(yt_client.read_table(cache.table_path, raw=False, format=yt.wrapper.DsvFormat()))
    else:
        raise Exception("Unknown runtime type %s") % cache.runtime

    cursor.executemany("INSERT OR IGNORE INTO cache(host, part, part_size, mds_key) VALUES(?, ?, ?, ?)", stream)
    db.commit()
    return cache.file_name


def build_and_release_cache(self, yt_client, resource_type, released_resource_date, server_name, prefix, env_type):
    logging.info("build_and_release_cache")

    cache_sources = []
    try:
        if checkout_yt_source(yt_client, released_resource_date, server_name, prefix, cache_sources):
            files = []
            cache_date = released_resource_date

            resource = sdk2.Resource[resource_type](self, "MDS archive keys cache", "RESOURCE", ttl=90)
            resource_data = sdk2.ResourceData(resource)
            logging.info("creating resource %s", resource_data.path)

            for source in cache_sources:
                logging.info("cache_source: %s", source)
                cache_date = max(cache_date, source.table_time)
                files.append(build_cache(yt_client, resource_data.path, source))

            logging.info("prepared files: %s", files)
            resource_data.ready()
            self.Context.envs_to_release.append(env_type)

    except Exception:
        logging.error(traceback.format_exc())

    for source in cache_sources:
        if os.path.isfile(source.file_name):
            os.remove(source.file_name)

    logging.info("build_and_release_cache - done")
    return len(self.Context.envs_to_release) > 0


class UpdateWebmasterArchiveKeysCache(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Update sqlite3 MDS keys database
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 20480

        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

    class Context(sdk2.Task.Context):
        envs_to_release = []

    class Parameters(sdk2.Task.Parameters):
        mr_server = sdk2.parameters.String("YT proxy", default="arnold.yt.yandex.net")
        yt_token_name = sdk2.parameters.String("Name of the sandbox vault record storing yt token", default="robot-webmaster-yt-token")
        yt_token_owner = sdk2.parameters.String("Owner of the sandbox vault record storing yt token", default="WEBMASTER")
        auto_release = sdk2.parameters.Bool("Auto release resource", default=False)
        ttl = sdk2.parameters.Integer("TTL for released resource (days)", default=90, required=True)

    def on_execute(self):
        from yt.wrapper import YtClient

        yt_token = sdk2.Vault.data(self.Parameters.yt_token_owner, self.Parameters.yt_token_name)
        yt_client = YtClient(proxy=self.Parameters.mr_server, token=yt_token)

        targets = (
            ("WMCONSOLE_URLS_CACHE_TEST_LINKS", "arnold.yt.yandex.net", "//home/webmaster/test/links/mds/logs", "testing"),
            ("WMCONSOLE_URLS_CACHE_PROD_LINKS", "arnold.yt.yandex.net", "//home/webmaster/prod/links/mds/logs", "stable"),
        )

        for resource_type, server_name, prefix, env_type in targets:
            latest_resource = None
            logging.info("fetching %s", resource_type)

            resources = sdk2.Resource[resource_type].find(
                attrs=dict()
            ).limit(100)

            for resource in resources:
                logging.info("resource %s, %s, %s, %s", resource_type, resource.created, resource.state, resource.released)
                if resource.released == env_type:
                    if latest_resource is None and resource.state == u"READY":
                        logging.info("set latest resource to %s, %s, %s, %s, %s", resource_type, resource.created, resource.state, resource.released, type(resource.released))
                        latest_resource = resource
                        continue

            released_resource_date = datetime.datetime(1970, 1, 1)
            if latest_resource:
                released_resource_date = latest_resource.created.replace(tzinfo=None)

            if build_and_release_cache(self, yt_client, resource_type, released_resource_date, server_name, prefix, env_type):
                break

    def on_success(self, prev_status):
        if not self.Parameters.auto_release:
            return

        for env_type in set(self.Context.envs_to_release):
            additional_parameters = dict(
                releaser=self.author,
                release_status=env_type,
                release_subject="Automatic release of " + self.Parameters.description,
                email_notifications=dict(to=[], cc=[]),
            )
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
            self.mark_released_resources(
                env_type,
                ttl=self.Parameters.ttl
            )
