import json
import logging
import os
import datetime

from sandbox import sdk2
from sandbox.projects.wmc.ydbHelper import YdbHelper
from sandbox.sandboxsdk import environments
import sandbox.common.errors as ce
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog


class WmcYdbSchemeBackupData(sdk2.Resource):
    """ A directory with scheme """
    ttl = 21


class WmcYdbToYtBackup(sdk2.Task):
    """Export YDB database to YT"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 512

        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

    class Parameters(sdk2.Parameters):

        proxy = sdk2.parameters.String(
            "YT proxy (cluster)",
            default="hahn",
            required=True,
        )

        destination_path = sdk2.parameters.String(
            "Path to the dir where backup dir will be saved",
            description="There must not be a slash in the end of destination!"
                        " In mentioned directory the new directory with name"
                        " represented by timestamp as name will be created",
            required=True,
        )

        ydb_endpoint = sdk2.parameters.String(
            "YDB endpoint",
            description="host:port",
            default_value="ydb-ru-prestable.yandex.net:2135",
            required=True,
        )

        ydb_database = sdk2.parameters.String(
            "YDB database name",
            default_value="/ru-prestable/wmcon/test/webmaster",
            required=True,
        )

        check_interval_time = sdk2.parameters.Integer(
            "Check progress status interval time (sec.)",
            default=300,
            required=False,
        )

        min_backup_count = sdk2.parameters.Integer(
            "Minimum guaranteed count of backup dirs in YT destination path",
            default=7,
            required=True,
        )

        yt_yav_secret_key = sdk2.parameters.String(
            "yt token name",
            default_value='yt.arnold.token',
            description="Token name to extract from YAV, "
                        "necessary to access tables on yt directly"
        )

        ydb_yav_secret_key = sdk2.parameters.String(
            "ydb token name",
            default_value='ydb.auth.key',
            description="Token name to extract from YAV, "
                        "necessary if you want to export an environment different from PROD"
        )

        ydb_yav_secret = sdk2.parameters.YavSecret("YDB YAV secret", required=True)
        yt_yav_secret = sdk2.parameters.YavSecret("YT YAV secret", required=True)
        path_to_backup = sdk2.parameters.String(
            'Path to backup',
            default_value='webmaster3',
            required=True,
            description="Path to backup"
        )

    def prepare_yt_target_directory(self, yt_token):
        logging.info("YT target directory preparing started.")

        from yt import wrapper as yt
        yt.config.set_proxy(self.Parameters.proxy)
        yt.config["token"] = yt_token

        destination = self.Parameters.destination_path

        today = str(datetime.datetime.now().strftime("%Y-%m-%d"))
        self.Context.yt_backup_path = os.path.join(destination, today)

        logging.info("Check data for deletion in YT directory {}".format(destination))

        all_files_list = yt.list(destination)
        logging.info("List of all files in YT directory {}: {}".format(destination, all_files_list))

        if today in all_files_list:
            raise ce.TaskError("File with name {} already exists in {} YT directory.".format(today, destination))

        backup_list = []
        for node in all_files_list:
            node_path = os.path.join(destination, node)
            if yt.get_type(node_path) == "map_node":
                backup_list.append(node_path)
        backup_list.sort()
        logging.info("List of backup dirs in YT directory {}: {}".format(destination, backup_list))

        number_of_backups = self.Parameters.min_backup_count
        if len(backup_list) > number_of_backups:
            for backup in backup_list[:len(backup_list) - number_of_backups]:
                logging.info("Backup dir {} is old and will be deleted".format(backup))
                yt.remove(backup, recursive=True)

    @staticmethod
    def __extract_export_yt_operation_id(data):
        parsed_output = json.loads(data)
        status = parsed_output["status"]
        if status != "SUCCESS":
            raise ce.TaskError(
                "Output status on backup starting isn't SUCCESS."
                "Current status: {}. Output: {}".format(status, parsed_output)
            )
        return parsed_output['id']

    @staticmethod
    def __extract_export_yt_tables(data):
        parsed_output = json.loads(data)
        return parsed_output["metadata"]["settings"]["items"]

    @staticmethod
    def __check_progress_status(data):
        parsed_output = json.loads(data)
        progress_status = parsed_output["metadata"]["progress"]
        if progress_status == 'PROGRESS_DONE':
            logging.info("Progress status: {}. Now the copy of the data is in YT.".format(progress_status))
            return True
        else:
            logging.info("Progress status: {}".format(progress_status))
            return False

    def start_export_operation(self, yh, database, tables_to_export):
        operation_id = WmcYdbToYtBackup.__extract_export_yt_operation_id(
            yh.start_export(
                database,
                tables_to_export,
                self.Context.yt_backup_path
            )
        )
        return operation_id

    def export_scheme(self, yh, yt_backup_path):
        from yt import wrapper as yt

        yt.config.set_proxy(self.Parameters.proxy)
        yt.config["token"] = self.Parameters.yt_yav_secret.data()[self.Parameters.yt_yav_secret_key]

        with ProcessLog(self, logger="make_directory") as pl:
            work_dir = os.getcwd()
            sp.check_call(["mkdir", "-p", self.Parameters.path_to_backup], shell=False, stdout=pl.stdout, stderr=pl.stderr, cwd=work_dir)
            pl.logger.info("Data after extraction : {}".format(os.listdir(work_dir)))

        yh.operation_dump_sheme(self.Parameters.path_to_backup)
        resource = WmcYdbSchemeBackupData(self, "Directory with buckap scheme", self.Parameters.path_to_backup)
        data = sdk2.ResourceData(resource)
        data.path.mkdir(0o755, parents=True, exist_ok=True)
        data.ready()

        name_scheme_arc = "scheme.tar.gz"
        with ProcessLog(self, logger="ydb_cli_unpack") as pl:
            work_dir = os.getcwd()
            sp.check_call(["tar", "-czf", name_scheme_arc, self.Parameters.path_to_backup], shell=False, stdout=pl.stdout, stderr=pl.stderr, cwd=work_dir)
            pl.logger.info("Data after extraction : {}".format(os.listdir(work_dir)))

        with open(name_scheme_arc, "rb") as f:
            yt.write_file(yt.ypath_join(yt_backup_path, "/scheme.tar.gz"), f)

    def on_execute(self):
        from yt import wrapper as yt

        ydb_token = self.Parameters.ydb_yav_secret.data()[self.Parameters.ydb_yav_secret_key]
        yt_token = self.Parameters.yt_yav_secret.data()[self.Parameters.yt_yav_secret_key]

        database = self.Parameters.ydb_database

        yt.config.set_proxy(self.Parameters.proxy)
        yt.config["token"] = yt_token
        yh = YdbHelper(
            ydb_token,
            self.Parameters.ydb_endpoint,
            database,
            yt_token,
            self.Parameters.proxy,
            self
        )
        with self.memoize_stage.start_export:
            self.prepare_yt_target_directory(yt_token)
            response = yh.start_export(
                database,
                self.Parameters.path_to_backup,
                self.Context.yt_backup_path
            )
            self.Context.operation_id = WmcYdbToYtBackup.__extract_export_yt_operation_id(
                response
            )
            self.Context.tables_to_export = WmcYdbToYtBackup.__extract_export_yt_tables(response)
            # no reason to ask yet, there's no way export is going to finish that fast
            raise sdk2.WaitTime(self.Parameters.check_interval_time)

        if not self.Context.operation_done:
            self.Context.operation_done = WmcYdbToYtBackup.__check_progress_status(
                yh.operation_get(self.Context.operation_id))
            if not self.Context.operation_done:
                raise sdk2.WaitTime(self.Parameters.check_interval_time)

            yh.operation_forget(self.Context.operation_id)
            logging.info("Backup is finished. Will now sort tables in place")

        self.export_scheme(yh, self.Context.yt_backup_path)

        items = self.Context.tables_to_export
        for item in items:
            destination = item["destinationPath"].replace("<append=true>", "", 1)
            primary_keys = list(json.loads(yh.describe(item["sourcePath"]))['primary_key'])
            yt.run_sort(destination, sort_by=primary_keys)
        logging.info("Backup is done, sort operations finished.")
