import os

import sandbox.common.errors as ce
from sandbox import sdk2
from sandbox.projects.kikimr.resources import YdbCliBinary
from sandbox.sdk2.helpers import subprocess as sp, ProcessLog

import logging


class YdbHelper(object):
    def __init__(self,
                 ydb_token,  # type: str
                 endpoint,  # type: str
                 database,  # type: str
                 # necessary if you intend to use yt operations (e.g. yt export)
                 yt_token,  # type: str
                 yt_proxy,  # type: str
                 # task to inject loggers to
                 task,      # type: sdk2.Task
                 ):
        self.ydb_token = ydb_token
        self.endpoint = endpoint
        self.database = database

        self.yt_token = yt_token
        self.yt_proxy = yt_proxy
        self.task = task

        ydb_cli_resource = YdbCliBinary.find(
            attrs=dict(released="stable", platform="linux")
        ).first()

        if ydb_cli_resource is None:
            raise ce.TaskError("Cannot find {} resource".format(YdbCliBinary.name))
        self.ydb_cli = str(sdk2.ResourceData(ydb_cli_resource).path)

        postfix = ".tgz"
        if self.ydb_cli.endswith(postfix):
            with ProcessLog(self.task, logger="ydb_cli_unpack") as pl:
                work_dir = os.getcwd()
                sp.check_call(["tar", "-zxf", self.ydb_cli], shell=False, stdout=pl.stdout, stderr=pl.stderr, cwd=work_dir)
                pl.logger.info("Data after extraction : {}".format(os.listdir(work_dir)))
                self.ydb_cli = os.path.join(work_dir, os.path.basename(self.ydb_cli)[:-len(postfix)])
        else:
            raise ce.TaskError("Ydb cli has extension other than .tgz")

    def __common_params(self):
        return [self.ydb_cli, "--endpoint={}".format(self.endpoint), "--database={}".format(self.database)]

    def __get_output(self, log_name, args):
        with ProcessLog(self.task, logger=log_name) as pl:
            public_args = filter(lambda x: not x.startswith('--token'), args[1:])
            logging.info("Executing: [ydb " + ' '.join(public_args) + ']')
            process = sp.Popen(args, env={"YDB_TOKEN": self.ydb_token}, stdout=sp.PIPE, stderr=pl.stderr)
            outs = process.communicate()
            if process.returncode != 0:
                raise ce.TaskError('Ydb call failed.\nStderr: {output}'.format(output=outs[1]))
            logging.info('Output from YDB cli: {output}'.format(output=outs[0]))
            return outs[0]

    def describe(self,
                 path,  # type: str
                 ):
        args = self.__common_params()
        args += ["scheme", "describe", path]
        args += ["--format=proto-json-base64"]
        return self.__get_output('describe_tables', args)

    def start_export(self,
                     database,  # type: str
                     path,  # type: str
                     output_folder
                     ):
        args = self.__common_params()
        args += ["export", "yt"]
        args += ["--proxy={}".format(self.yt_proxy)]
        args += ["--token={}".format(self.yt_token)]
        destination = os.path.join(output_folder, path)
        args += ["--item=source={},destination={}".format(database + '/' + path, destination)]
        args += ["--use-type-v3"]
        args += ["--format=proto-json-base64"]
        return self.__get_output('export_yt', args)

    def operation_get(self, operation_id):
        args = self.__common_params()
        args += ["operation", "get", str(operation_id)]
        args += ["--format=proto-json-base64"]
        return self.__get_output('operation_get', args)

    def operation_forget(self, operation_id):
        args = self.__common_params()
        args += ["operation", "forget", str(operation_id)]
        return self.__get_output('operation_forget', args)

    def operation_dump_sheme(self, path):
        args = self.__common_params()
        args += ["tools", "dump"]
        args += ["--output={}".format(path)]
        args += ["--path={}".format(path)]
        args += ["--scheme-only"]
        return self.__get_output('operation_dump_sheme', args)
