import json
import os
import logging
import argparse

from sandbox.projects.woland.common import WolandPanel

APPHOST_GRAPHS_DIR = 'apphost/conf/verticals'
GENERATED_PANELS_PATH = 'apphost/verticals'
ALLOWED_CTYPES = ('prod', 'prestable')


def get_graph_source_signal_perc(graph_name, source_name):
    return """
        na(
            perc(
                or(unistat-SOURCES-{graph_name}-{source_name}-Failures_dmmm, const(0)),
                or(unistat-SOURCES-{graph_name}-{source_name}-Requests_dmmm, const(1))
            )
        )
        """.format(
        source_name=source_name,
        graph_name=graph_name,
        ).replace('\n', '').replace(' ', '')


def get_graph_response_perc(graph_name, suffix):
    return """
        na(
            perc(
                or(unistat-SOURCES-http_adapter_{graph_name}-SELF-ResponseCode-{suffix}_dmmm, const(0)),
                or(unistat-SOURCES-{graph_name}-SELF-Successes_dmmm, const(1))
            )
        )
        """.format(
        graph_name=graph_name,
        suffix=suffix,
        ).replace('\n', '').replace(' ', '')


def get_vertical_signals(verticals_path, vertical_name):
    vertical_path = os.path.join(verticals_path, vertical_name)
    signals = []
    if not os.path.exists(vertical_path):
        logging.error('Path %s not found!' % vertical_path)
        return signals

    for root, _, files in os.walk(vertical_path):
        for filename in files:
            if filename.endswith('.json') and not filename.startswith('_'):
                graph_name = filename[:-5]
                # Add 5xx and 4xx signals for each graph in vertical
                for suffix in ('4xx', '5xx'):
                    signals.append(
                        {
                            'signal': get_graph_response_perc(graph_name, suffix),
                            'caption': '{} {} requests'.format(graph_name, suffix),
                            'warning': [5, 10],
                            'critical': [10, None],
                        }
                    )
                logging.debug('Proceeding %s file' % filename)
                with open(os.path.join(root, filename), 'r') as f:
                    parsed = json.load(f)

                sources = parsed.get('settings', {}).get('nodes', {}) if isinstance(parsed, dict) else {}
                logging.debug('%s sources found' % len(sources.keys()))

                for source, desc in sources.items():
                    # Skip not importans sources, 'cause who cares about something that is not important?!
                    if desc.get('exclude_from_common_stats', False):
                        logging.debug('Source %s skipped due it is excluded from common stats' % source)
                        continue

                    logging.debug('Creating signal for graph %s source %s' % (graph_name, source))
                    signals.append(
                        {
                            'signal': get_graph_source_signal_perc(graph_name, source),
                            'caption': '{}/{} fail requests'.format(graph_name, source),
                            'warning': [5, 10],
                            'critical': [10, None],
                        }
                    )

    logging.debug('Totally collect %s signals for vertical %s' % (len(signals), vertical_name))
    return signals


def get_vertical_settings(arcadia_path, vertical_name):
    vertical_settings_path = os.path.join(arcadia_path, APPHOST_GRAPHS_DIR, vertical_name, "_vertical_settings.json")
    with open(vertical_settings_path, "r") as settings_file:
        return json.load(settings_file)


def select_production_geos_and_ctypes(vertical_settings):
    used_geos = set()
    used_ctypes = set()
    for configuration in vertical_settings["requiredLocations"]:
        ctype, geo = configuration.split(';')
        ctype = ctype[len('ctype='):]
        geo = geo[len('geo='):]

        if ctype in ALLOWED_CTYPES:
            used_geos.add(geo)
            used_ctypes.add(ctype)
    return list(used_geos), list(used_ctypes)


def get_rendered_panels_files(arcadia_path):
    abs_graphs_path = os.path.join(arcadia_path, APPHOST_GRAPHS_DIR)
    for vertical_name in os.listdir(abs_graphs_path):
        logging.debug('Going to proceed %s vertical' % vertical_name)
        filename = os.path.join(GENERATED_PANELS_PATH, vertical_name + '-sources-fails.yaml')
        vertical_settings = get_vertical_settings(arcadia_path, vertical_name)

        geos, ctypes = select_production_geos_and_ctypes(vertical_settings)
        prj = vertical_settings["mainPrj"] if "mainPrj" in vertical_settings else vertical_name.lower()
        yield (filename, WolandPanel(
            hosts='ASEARCH',
            itype='apphost',
            ctype=ctypes,
            prj=prj,
            geo=geos,
            signals=get_vertical_signals(abs_graphs_path, vertical_name)
        ).get_rendered_yaml())


def main():
    parser = argparse.ArgumentParser(description='Woland panels generator for apphost graphs')
    parser.add_argument('-d', '--debug', action='store_true', help='If set, debug messages will be shown.')
    parser.add_argument('-p', '--arcadia_path', default='.', help='Path to local Arcadia checkout.')

    args = parser.parse_args()

    log_level = logging.INFO

    if args.debug:
        log_level = logging.DEBUG

    formatter_str = "%(asctime)-15s %(levelname)-8s %(module)s>%(funcName)s: %(message)s"
    logging.basicConfig(level=log_level, format=formatter_str)
    for filename, file_content in get_rendered_panels_files(args.arcadia_path):
        print(filename)
        print(file_content)


if __name__ == "__main__":
    main()
