from __future__ import print_function
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.woland import resources as wr
import json
import logging
import requests
import six
from requests.packages.urllib3.util.retry import Retry

from . import post_alerts


class JugglerException(TaskFailure):
    pass


class WolandAlerting(sdk2.Task):
    names = {}

    class Parameters(sdk2.Parameters):
        dry_run = sdk2.parameters.Bool('Dry run: print the list of alerts to be generated, do not make any calls to juggler')
        woland_binary_id = sdk2.parameters.Resource('Woland binary', resource_type=wr.WOLAND_EXECUTABLE, required=True)
        woland_panels_id = sdk2.parameters.Resource('Woland panels SB resource id', resource_type=wr.WOLAND_PANELS, required=True)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024  # 1 Gb

    def on_execute(self):
        session = requests.Session()
        session.mount('https://', requests.adapters.HTTPAdapter(max_retries=Retry(total=10, backoff_factor=0.1, status_forcelist=[202])))
        woland_binary_path = str(sdk2.ResourceData(self.Parameters.woland_binary_id).path)
        woland_panels_path = str(sdk2.ResourceData(self.Parameters.woland_panels_id).path)

        with self.memoize_stage.start:
            logs = []
            to_add = post_alerts.prepare_alerts(woland_binary_path, woland_panels_path, logs)
            if logs:
                self.set_info('\n'.join(logs))
                raise TaskFailure("Alerts generation failed, see above")

            if self.Parameters.dry_run:
                logging.info("Stats:\n%s" % '\n'.join('%s %s signals' % (prefix, len(values)) for prefix, values in six.iteritems(to_add)))
                logging.info(json.dumps(to_add, indent=2))
                self.set_info('Dry run, see common.log with the list of signals. I will not upload anything.')
                return
            if not to_add:
                return

            logging.debug('\n=== Uploading alerts ===')
            self.Context.operations = []
            for prefix, alerts in six.iteritems(to_add):
                operation = session.post(
                    "https://yasm.yandex-team.ru/srvambry/alerts/replace/background",
                    json={
                        'prefix': prefix,
                        'alerts': alerts,
                    },
                    verify=False
                )
                logging.debug(operation.text)
                operation = operation.json()
                if 'error' in operation:
                    raise JugglerException(operation['error'])
                op_id = operation['response']['result']['operation_id']
                self.set_info(
                    '== {prefix} ({count} signals) ==\n'
                    'Started upload operation #{operation}\n'
                    'To get progress: curl https://yasm.yandex-team.ru'
                    '/srvambry/alerts/replace/status --data \'{{"operation_id":"{operation}"}}\' | jq .'.format(operation=op_id, prefix=prefix, count=len(alerts)))
                self.Context.operations.append(op_id)
            self.Context.AttemptsLeft = 256

        if not self.Context.AttemptsLeft:
            raise TaskFailure("Failed to verify process completion (no attempts left)")

        self.Context.AttemptsLeft = self.Context.AttemptsLeft - 1
        towait = []
        for op in self.Context.operations:
            result = session.post(
                'https://yasm.yandex-team.ru/srvambry/alerts/replace/status',
                json={"operation_id": op},
                verify=False
            ).json()
            found = result.get('error_code') != 'not_found'
            result = result.get('response', {}).get('result', {})
            if found and not result.get('failed') and result.get('status') != 'finished':
                towait.append(op)
                continue

            self.set_info(json.dumps(result, indent=2))
            if result.get('failed'):
                raise JugglerException(result['message'])

        self.Context.operations = towait
        if towait:
            raise sdk2.WaitTime(300)
