#!/usr/bin/env python3

import json
import logging
import re
import subprocess
import traceback
import six

JugglerRe = re.compile(r'[^A-Za-z0-9\-+=./]')


def _cleanup_caption(s):
    return JugglerRe.sub('', s.replace(' ', '-'))


def build_alerts(used_names, disp_name, chart, urls_in_description=False):
    '''
    :param used_names: dict: string -> int, count of used unique signal names, should be preserved between function calls
    :param chart: dict, output of `woland -d ...`
    :param urls_in_description: whether print signal value (False) or signal url (True) in notifications. Both are not supported by juggler.
    '''
    alert_data = {}
    alert_data["signal"] = chart["name"]
    alert_data["warn"] = chart["warning"]
    alert_data["crit"] = chart["critical"]
    alert_data["mgroups"] = ["ASEARCH"]
    if "caption" in chart:
        alert_data["description"] = chart["caption"]

    check = {}
    check['namespace'] = chart['namespace']
    if 'flaps' in chart:
        check['flaps'] = chart['flaps']
    if "alerts" in chart and chart['alerts']:
        check["aggregator"] = "logic_or"
        check["aggregator_kwargs"] = {
            "unreach_service": [{
            "check": "yasm_alert:virtual-meta"}],
            "unreach_mode": "force_ok",
            "nodata_mode": "force_ok",
            "ok_desc": "OK!",
        }
        check["notifications"] = []

        for k, v in six.iteritems(chart["alerts"]):
            for st, logins in six.iteritems(v):
                if logins:
                    notif_params = {
                        "template_name": "on_status_change",
                        "template_kwargs": {
                            "status": [st],
                            "method": [k],
                            "login": logins
                        },
                        "description": chart["caption"]
                    }
                    check['notifications'].append(notif_params)

    if 'juggler_check_tags' in chart:
        check['tags'] = []
        for tag in chart['juggler_check_tags']:
            check['tags'].append(tag)

    itypes = chart["itype"]
    if not isinstance(itypes, list):
        itypes = [itypes]
    alerts = []
    disp_name = _cleanup_caption(disp_name)
    if len(chart["ctype"]):
        for prj in chart.get("prj", [None]) or [None]:
            for geo in chart["geo"]:
                for itype in itypes:

                    if not isinstance(chart["ctype"], list):
                        ctype_list = [chart["ctype"]]
                    else:
                        ctype_list = chart["ctype"][:]

                    # By some reason, prestable could only in SAS. If not, this assumption should be re-designed
                    if geo != 'sas' and 'prestable' in ctype_list:
                        ctype_list.pop(ctype_list.index('prestable'))

                    for ctype in ctype_list:
                        alert_data['tags'] = {
                            'itype': itype  # list here is not supported
                        }
                        for k in ['ctype', 'prj', 'geo']:
                            v = locals().get(k)
                            if v:
                                alert_data['tags'][k] = [v]

                        try:
                            displayed_prj = chart.get('displayed_prj') or prj
                            name = "woland_%s." % disp_name + '_'.join([itype, ctype or '', displayed_prj or '', geo or ''])
                        except Exception:
                            logging.debug("Error generating signal with itype {itype}, ctype {ctype}, prj {prj}, geo {geo})".format(**locals()))
                            raise

                        if name in used_names:
                            used_names[name] += 1
                            name += '_' + str(used_names[name])
                        else:
                            used_names[name] = 0
                        alert_data["name"] = name
                        caption = _cleanup_caption(chart["caption"])
                        if check:
                            check["host"] = name
                            check["service"] = caption
                            if urls_in_description:
                                url = "https://yasm.yandex-team.ru/chart/hosts=ASEARCH;itype=" + itype + ";ctype=" + ctype + ";geo=" + geo + ";signals=" + alert_data["signal"]
                                if prj:
                                    url += ';prj=' + prj
                                check["aggregator_kwargs"]["warn_desc"] = "WARNING on [Chart](" + url + ")"
                                check["aggregator_kwargs"]["crit_desc"] = "CRITICAL on [Chart](" + url + ")"
                                check["aggregator_kwargs"]["nodata_desc"] = "NO DATA on [Chart](" + url + ")"
                            alert_data["juggler_check"] = check.copy()
                        interval = chart.get('interval', {})
                        if interval:
                            try:
                                alert_data['value_modify'] = {
                                    'type': interval['aggregation'],
                                    'window': int(interval['window']),
                                }
                            except Exception as x:
                                logging.info('[%s/%s] Failed adding interval alert for %s: %s' % (disp_name, caption, alert_data['signal'], x))
                        alerts.append(alert_data.copy())

    return alerts


def prepare_alerts(woland_binary_path, woland_panels_path, logs):
    woland_cmd_line = [
            woland_binary_path,
            '-d', '_snippets/_alert', # panel name is 2 element counting from 0.
            '--panels-dir', woland_panels_path
            ]
    subprocess.check_output(woland_cmd_line)
    with open('output.json') as a:
        alerts_cfg = json.load(a)
    to_add = {}
    for disp_name, panel in six.iteritems(alerts_cfg['panels']):
        logging.debug("\n=== Building alerts for %s ===" % panel)
        used_names = {}
        to_add['woland_' + disp_name] = []
        try:
            # set panel name in params list
            woland_cmd_line[2] = panel
            logging.debug(subprocess.check_output(woland_cmd_line))
            with open('output.json') as f:
                data = json.load(f)
            for chart in data["sig_spec"]:
                to_add['woland_' + disp_name] += build_alerts(used_names, disp_name, chart)
        except Exception:
            logs.append('== Failed generating alerts for %s ==' % panel)
            logs.append(traceback.format_exc())
    return to_add


if __name__ == '__main__':
    import argparse
    import os

    class HelpFormatter (argparse.ArgumentDefaultsHelpFormatter, argparse.RawDescriptionHelpFormatter):
        pass

    try:
        os.environ['COLUMNS'] = subprocess.check_output(['stty', 'size']).split()[1]
    except Exception:
        pass
    args = argparse.ArgumentParser(description='Woland to juggler alerts converter', formatter_class=HelpFormatter)
    args.add_argument('woland', help='Path to the binary')
    args.add_argument('--panels-dir', help='Path to panels directory')
    args = args.parse_args()

    logs = []
    result = prepare_alerts(args.woland, args.panels_dir, logs)
    print('\n'.join(logs))
    print(json.dumps(result, indent=2))
