import os
import logging
import tarfile
import tempfile
import shutil

from sandbox.projects.woland.common import WolandPanel

from sandbox import sdk2

from contextlib import contextmanager

BALANCER_CONFIGS_SANDBOX_TYPE = 'BALANCER_GENCFG_CONFIGS_L7_TGZ'
CONFIG_FILES_PATH_IN_TAR = 'generated/l7-balancer'
PROJECT_BY_CONFIG_NAME = {
    'l7heavy_service_search_man.cfg': 'l7-balancer-knoss-search',
    'l7heavy_service_search_sas.cfg': 'l7-balancer-knoss-search',
    'l7heavy_service_search_vla.cfg': 'l7-balancer-knoss-search',

    'l7heavy_service_morda_man.cfg': 'l7-balancer-knoss-morda',
    'l7heavy_service_morda_sas.cfg': 'l7-balancer-knoss-morda',
    'l7heavy_service_morda_vla.cfg': 'l7-balancer-knoss-morda',

    'l7heavy_production_tun_man_only.cfg': 'l7-geo-only',
    'l7heavy_production_tun_sas_only.cfg': 'l7-geo-only',
    'l7heavy_production_tun_vla_only.cfg': 'l7-geo-only',

    'l7heavy_production_tun_man.cfg': 'l7-balancer',
    'l7heavy_production_tun_sas.cfg': 'l7-balancer',
    'l7heavy_production_tun_vla.cfg': 'l7-balancer',

    'l7heavy_production_fukraine_man.cfg': 'l7-proxy',
    'l7heavy_production_fukraine_sas.cfg': 'l7-proxy',
    'l7heavy_production_fukraine_vla.cfg': 'l7-proxy',

    'l7heavy_production_yaru_man.cfg': 'l7-balancer-yaru',
    'l7heavy_production_yaru_sas.cfg': 'l7-balancer-yaru',
    'l7heavy_production_yaru_vla.cfg': 'l7-balancer-yaru',

    'l7heavy_experiments_vla.cfg': 'l7-balancer-exp-term',

    'any_sas.cfg': 'any',
}


@contextmanager
def create_context_temporary_directory():
    try:
        temp_dir = tempfile.mkdtemp()
        yield temp_dir
    finally:
        shutil.rmtree(temp_dir)


def load_json_configs(configs_path):
    logging.info('configs_path {}'.format(configs_path))

    config_jsons = []
    for config_name in os.listdir(configs_path):
        if config_name not in PROJECT_BY_CONFIG_NAME:
            logging.info('skip config %s' % config_name)
            continue
        with open(os.path.join(configs_path, config_name), 'r') as config:
            project = PROJECT_BY_CONFIG_NAME[config_name]

            import lua_to_json

            config_json = lua_to_json.loads(config.read().encode('utf-8'))
            config_jsons.append((
                config_json,
                project,
            ))

    return config_jsons


def get_balancer_json_configs():
    resource = sdk2.Resource[BALANCER_CONFIGS_SANDBOX_TYPE].find(state='READY').first()
    configs_path = str(sdk2.ResourceData(resource).path)

    with create_context_temporary_directory() as temp_dir:
        logging.info('configs path: {}'.format(configs_path))

        with tarfile.open(configs_path) as tar:
            tar.extractall(path=temp_dir)

        return load_json_configs(os.path.join(temp_dir, CONFIG_FILES_PATH_IN_TAR))

    raise Exception('failed to get balancer configs')


def iterate_over_report_sections(config_json):
    for key, value in config_json.items():
        if key == 'report':
            if 'uuid' in value:
                yield value['uuid']
        if isinstance(value, dict):
            for inner_value in iterate_over_report_sections(value):
                yield inner_value


def get_reports_from_configs(config_jsons):
    report_uuids_by_project = {}
    for config_json, project in config_jsons:
        for report_uuid in iterate_over_report_sections(config_json):
            if project not in report_uuids_by_project:
                report_uuids_by_project[project] = set()
            report_uuids_by_project[project].add(report_uuid)
    return report_uuids_by_project


def get_signal_perc(report_uuid, signal_name):
    return """perc(
        or(balancer_report-report-{report_uuid}-{signal_name}_summ, const(0)),
        or(balancer_report-report-{report_uuid}-requests_summ, const(1))
    )""".format(
        report_uuid=report_uuid,
        signal_name=signal_name,
    ).replace('\n', '').replace(' ', '')


def get_client_fail_signal_perc(report_uuid):
    return get_signal_perc(report_uuid, 'fail')


def get_backend_fail_signal_perc(report_uuid):
    return get_signal_perc(report_uuid, 'backend_fail')


def get_outgoing_404_signal_perc(report_uuid):
    return get_signal_perc(report_uuid, 'outgoing_404')


def is_outgoing_404_needed_for_project(project):
    return project not in ['any']


def generate_signals(project, report_uuids):
    signals = []
    for report_uuid in report_uuids:
        for single_report_uuid in report_uuid.split(','):
            signals.append({
                'signal': get_client_fail_signal_perc(single_report_uuid),
                'caption': '{} fail requests'.format(single_report_uuid),
                'warning': [10, 30],
                'critical': [30, None],
            })
            signals.append({
                'signal': get_backend_fail_signal_perc(single_report_uuid),
                'caption': '{} backend fail requests'.format(single_report_uuid),
                'warning': [2, 5],
                'critical': [5, None],
            })
            if is_outgoing_404_needed_for_project(project):
                signals.append({
                    'signal': get_outgoing_404_signal_perc(single_report_uuid),
                    'caption': '{} outgoing_404 requests'.format(single_report_uuid),
                    'warning': [30, 50],
                    'critical': [50, None],
                })
    return signals


def get_rendered_panels_files():
    logging.info('get rendered panel files for balancer')
    report_uuids_by_project = get_reports_from_configs(get_balancer_json_configs())
    for project in report_uuids_by_project.keys():
        filename = '{project}-sources-fails.yaml'.format(project=project)

        yield (filename, WolandPanel(
            hosts='ASEARCH',
            itype='balancer',
            ctype=['prod', 'stable', 'prestable', 'experiment'],
            prj=project,
            geo=['man', 'sas', 'vla'],
            signals=generate_signals(project, report_uuids_by_project[project])
        ).get_rendered_yaml())
