# -*- coding: utf-8 -*-

import logging
import re
import requests
import uuid

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.sdk2 as sdk2

from sandbox.sandboxsdk.environments import PipEnvironment


def make_request(token, url, body=None, request_type="post"):
    headers = {
        "Authorization": "OAuth %s" % token,
        "Accept": "application/json",
        "Content-Type": "application/json",
    }
    assert request_type in {"get", "post"}
    logging.debug(
        "Making %s request to '%s' with data: %s",
        request_type, url, body,
    )
    if request_type == "post":
        response = requests.post(url, headers=headers, json=body)
    elif request_type == "get":
        response = requests.get(url, headers=headers)
    logging.debug("Got response %d: %s", response.status_code, response.json())
    response.raise_for_status()
    return response


class OkClient:
    def __init__(self, ok_token, url="https://ok.yandex-team.ru/api"):
        self.ok_token = ok_token
        self.url = url

    def add_approvement(self, issue_key, author, text, all_approvers):
        make_request(token=self.ok_token, url="{}/approvements/".format(self.url), body={
            "object_id": issue_key,
            "uid": uuid.uuid4().hex,
            "text": text,
            "stages": [
                {
                    "need_all": False,
                    "stages": [{"approver": approver} for approver in approvers.split(",")],
                } for approvers in all_approvers
            ],
            "author": author,
            "is_parallel": True,
        })

    def get_approvement_status(self, approvement_uuid):
        response = make_request(token=self.ok_token, url="{}/approvements/{}".format(self.url, approvement_uuid), request_type="get")
        return response.json()["status"]

    def suspend_approvement(self, approvement_uuid):
        make_request(token=self.ok_token, url="{}/approvements/{}/suspend/".format(self.url, approvement_uuid))


class XurmaAddAgreement(sdk2.Task):
    """
    Adds agreement to a release ticket
    Used by the release machine
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = task_env.TaskTags.all_rm
        disk_space = 2 * 1024  # 2 Gb
        cores = 1
        environments = [
            PipEnvironment("startrek_client", custom_parameters=["requests==2.18.4"]),
        ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # 10 min
        default_wait_time = 4 * 60  # 4 hours

        component_name = sdk2.parameters.String("Component name", required=True)
        release_number = sdk2.parameters.Integer("Release number", required=True)
        text = sdk2.parameters.String("Text of the agreement", required=True)
        approvers = sdk2.parameters.List("List of the approvers", value_type=sdk2.parameters.String, required=True)
        wait_time = sdk2.parameters.Integer("Wait time (minutes)",  default_value=default_wait_time)

    def on_execute(self):
        with self.memoize_stage.wait_launch:
            raise sdk2.WaitTime(self.Parameters.wait_time * 60)

        secret = sdk2.yav.Secret("sec-01dcrpzkqqjwvxqfk1rdc61s0d")
        ok_token = secret.data()["ok"]
        startrek_token = secret.data()["startrek"]

        from startrek_client import Startrek
        startrek_client = Startrek(useragent="python", token=startrek_token)
        ok_client = OkClient(ok_token)
        issues = startrek_client.issues.find(query="AFRELEASE", filter={"tags": [self.Parameters.component_name]})

        for issue in issues:
            logging.info("Working with %s", issue.key)
            issue_status = issue.status.key
            approvement_status = issue.approvementStatus
            for tag in issue.tags:
                match = re.search(r"release_number_(\d+)", tag)
                if match is None:
                    continue
                release_number = int(match.group(1))
            if release_number > self.Parameters.release_number:
                continue
            is_active_issue = (release_number == self.Parameters.release_number)
            if is_active_issue:
                if approvement_status != "Запущено":
                    logging.info("Adding a new approvement to %s", issue.key)
                    ok_client.add_approvement(issue.key, self.author, self.Parameters.text, self.Parameters.approvers)
            else:
                if issue_status != "closed":
                    logging.info("Closing %s", issue.key)
                    for t in issue.transitions.get_all():
                        if t.id.startswith("close"):
                            t.execute()
                if approvement_status == "Запущено":
                    for c in issue.comments.get_all():
                        match = re.search(r"https://ok.yandex-team.ru/approvements/(.+)\?_embedded=1", c.text)
                        if match is None:
                            continue
                        approvement_uuid = match.group(1)
                        current_status = ok_client.get_approvement_status(approvement_uuid)
                        if current_status == "in_progress":
                            logging.info("Suspending %s in %s", approvement_uuid, issue.key)
                            ok_client.suspend_approvement(approvement_uuid)
