# -*- coding: utf-8 -*-

import json
import logging
import re

import sandbox.projects.common.file_utils as fu
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.resources as rm_res
import sandbox.sdk2 as sdk2


class XurmaPrepareChangelog(sdk2.Task):
    """
    Task that prepares a changelog for the release machine
    """

    GRAPH_NAMES = (
        # Xurma
        "admet",
        "admod",
        "afraudita",
        "bs",
        "btr",
        "dsp",
        "edadeal",
        "prebilling",
        "taxi",
        "zen",
        # Fury
        "adfox",
        "captcha",
        "market",
        "megaera",
        "online",
        "rtbs",
        "rtprebilling",
        "rtxaron",
        "supermod",
        # Other
        "bigmod",
        "viewer",
    )
    ALL_GRAPH_NAMES = r"\[({})\]".format("|".join(GRAPH_NAMES).upper())

    SIGNIFICANT_PATHS = (
        "/attic",
        "/antifraud/libs",
        "/antifraud/py_libs",
        "logfeller_synchronizer",
        "ytpump",
        "xurma",
    )
    ALL_SIGNIFICANT_PATHS = "|".join(SIGNIFICANT_PATHS)

    class Requirements(sdk2.Task.Requirements):
        client_tags = task_env.TaskTags.all_rm
        disk_space = 2 * 1024  # 2 Gb
        cores = 1

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # 10 min
        original_changelog = sdk2.parameters.Resource("Original changelog", resource_type=rm_res.RELEASE_MACHINE_CHANGELOG)
        graph_name = sdk2.parameters.String("The name of the graph", required=True)
        with sdk2.parameters.Output():
            final_changelog = sdk2.parameters.Resource("Final changelog", resource_type=rm_res.RELEASE_MACHINE_CHANGELOG)

    def on_enqueue(self):
        super(XurmaPrepareChangelog, self).on_enqueue()
        with self.memoize_stage.create_changelog_on_enqueue:
            self.Parameters.final_changelog = rm_res.RELEASE_MACHINE_CHANGELOG(
                task=self,
                description=self.Parameters.original_changelog.description,
                path="{}_final_changelog.json".format(self.Parameters.original_changelog.component),
            )

    def on_execute(self):
        logging.info("Reading the original changelog")
        original_changelog = json.loads(sdk2.ResourceData(self.Parameters.original_changelog).path.read_text())
        logging.info("Preparing the final changelog")
        graph_name = self.Parameters.graph_name
        final_changelog = self.prepare_changelog(original_changelog, graph_name)
        logging.info("Saving the final changelog")
        self.save_final_changelog(final_changelog)

    @staticmethod
    def prepare_changelog(changelog, graph_name):
        all_changes = changelog["all_changes"]
        for change_item in all_changes:
            change_item["changes"] = XurmaPrepareChangelog.prepare_changes(change_item["changes"], graph_name)
        return changelog

    @staticmethod
    def prepare_changes(changes, graph_name):
        prepared_changes = []
        for change in changes:
            XurmaPrepareChangelog.process_change(change, graph_name)
            if change["commit_importance"] > 0:
                prepared_changes.append(change)
        return sorted(prepared_changes, key=lambda c: c["commit_importance"], reverse=True)

    @staticmethod
    def process_change(change, graph_name):
        revision_paths = change["revision_paths"]
        good_paths = []
        for _, path in revision_paths:
            if re.search(XurmaPrepareChangelog.ALL_SIGNIFICANT_PATHS, path) and not re.search("test", path):
                good_paths.append(path)
        summary = change["summary"]
        final_summary = summary.split("\n")[0]
        if final_summary.startswith("Update resource by sandbox task"):
            final_summary = "Auto-update files:\n" + "\n".join(good_paths)
        final_importance = 3
        if not good_paths:
            final_importance = 0
        elif re.search(r"\[{}]".format(graph_name), final_summary, re.IGNORECASE):
            final_importance = 5
        elif re.search(XurmaPrepareChangelog.ALL_GRAPH_NAMES, final_summary, re.IGNORECASE):
            final_importance = 0
        elif final_summary.startswith("Auto-update files"):
            final_importance = 1
        change["summary"] = final_summary
        change["commit_importance"] = final_importance

    def save_final_changelog(self, final_changelog):
        fu.write_file(self.Parameters.final_changelog.path, json.dumps(final_changelog, indent=2))
        self.Parameters.final_changelog.component = self.Parameters.original_changelog.component
        self.Parameters.final_changelog.major_release_num = self.Parameters.original_changelog.major_release_num
        self.Parameters.final_changelog.minor_release_num = self.Parameters.original_changelog.minor_release_num
