import logging

from sandbox.projects.yabs.AnomalyAuditReport.lib.config_reader import AnomalyAuditConfigReader
from sandbox.projects.yabs.AnomalyAuditReport.lib.query_builder import QueryBuilderFactory, ColumnType
from sandbox.projects.yabs.AnomalyAuditReport.lib.query_executor import QueryExecutorFactory
from sandbox.projects.yabs.AnomalyAuditReport.lib.report_container import AnomalyAuditReportFactory
from sandbox.projects.yabs.AnomalyAuditReport.lib.report_container import ReportResultContainerBase
from sandbox.projects.yabs.AnomalyAuditReport.lib.config_reader import Database, Totals, SelectLevel

from yabs.server.cs.pylibs.banner_flags import get_banner_flags_list


class AnomalyAudit:

    @staticmethod
    def create_anomaly_audit_report(current_date, previous_date,
                                    run_configs: list[dict],
                                    yql_token, yql_temp_folder='//tmp/yql', load_column_dictionaries: bool = False
                                    ) -> tuple[list[ReportResultContainerBase], list[Exception]]:
        """Main Function to generate Anomaly Report.

        Args:
            current_date (str): Current Date (YYYY-MM-DDTHH:00:00).
            previous_date (str): Date to Compare With (YYYY-MM-DDTHH:00:00)'
            run_configs (list[dict]): Run Configuration JSON
            yql_token (str): YQL Token
            yql_temp_folder (str): TMP Folder for YQL request to YT
            load_column_dictionaries (bool): Load Dictionaries for Columns Values


        Returns:
            List[ReportResultContainerBase]: List of Results
            List[Exception]: Error List

        """
        if not yql_token:
            raise ValueError('YQL Token is empty')

        result = []
        errors = []
        values_description = dict()

        config = AnomalyAuditConfigReader.get_config(run_configs)

        saved_current_date = current_date
        saved_previous_date = previous_date

        for report_config in config:
            try:
                logging.info(f"Working on {report_config['reportName']}")

                current_date = saved_current_date
                previous_date = saved_previous_date
                if report_config['selectLevel'] == SelectLevel.Date:
                    current_date = saved_current_date[0:10]
                    previous_date = saved_previous_date[0:10]
                else:
                    current_date = saved_current_date
                    previous_date = saved_previous_date

                query_builder = QueryBuilderFactory.get_query_builder(report_config, current_date, previous_date)
                query_executor = QueryExecutorFactory.get_query_executor_by_config(report_config, yql_token)

                table_columns = AnomalyAudit._get_table_columns(report_config, query_builder, query_executor)
                columns = AnomalyAudit._get_report_columns(report_config, table_columns)
                columns_count = len(columns) - 1

                totals = AnomalyAudit._get_total_count(query_builder, query_executor)

                result_container = AnomalyAuditReportFactory.get_report_result_container(report_config,
                                                                                         current_date, previous_date,
                                                                                         totals)

                for i, column in enumerate(columns):
                    column_name, column_type, db_column_name = column
                    AnomalyAudit._execute_query(query_builder, query_executor,
                                                result_container,
                                                column_name, column_type, totals, db_column_name,
                                                errors)

                    logging.info(f'{i}/{columns_count}. Column {column_name} is processed')

                logging.info(f"Writing in {report_config['outputFormat']}")

                if result_container.has_result():
                    if load_column_dictionaries and not values_description:
                        values_description = AnomalyAudit._get_values_description(yql_token, yql_temp_folder)

                    result_container.add_value_descriptions(values_description)
                    result.append(result_container)

                logging.info(f"Working on {report_config['reportName']} is finished")

            except Exception as error:
                errors.append(error)
                logging.error(f"Error '{error}' occurs while processing {report_config['reportId']}")

        return result, errors

    @staticmethod
    def _get_total_count(query_builder, query_executor) -> Totals:
        try:
            logging.info('Loading Total Count')

            query = query_builder.get_total_query()
            query_result = query_executor.execute_query(query)

            result = query_result.get_next_row()
            logging.info(f"Total Counts {result}")

            totals = Totals()
            totals.current = int(result[0])
            totals.previous = int(result[1])
            totals.current_good = int(result[2])
            totals.previous_good = int(result[3])

            return totals

        except Exception as error:
            logging.error(f"Error '{error}' occurs while getting total count")

    @staticmethod
    def _execute_query(query_builder, query_executor, result_container,
                       column_name, column_type, totals, db_column_name,
                       errors: list[Exception]):
        try:
            query = query_builder.get_query(column_name, column_type, totals, db_column_name)
            query_result = query_executor.execute_query(query)
            result_container.add_result(query_result)

        except Exception as error:
            logging.error(f"Error '{error}' occurs while processing column {column_name}")
            errors.append(error)

    @staticmethod
    def _get_report_columns(report_config: dict, table_columns: list) -> list[tuple]:

        result = []
        grouped_columns = dict()
        for column_name, column_type in table_columns:
            tmp_column_name = column_name.strip().lower()

            # Check that group column is among table columns
            if tmp_column_name in report_config['groupedColumnsSet']:
                grouped_columns[tmp_column_name] = column_name
                if tmp_column_name not in report_config['manualColumns']:
                    continue

            if tmp_column_name in report_config['excludedColumns'] and tmp_column_name not in report_config['manualColumns']:
                continue

            if report_config['manualColumns'] and tmp_column_name not in report_config['manualColumns']:
                continue

            if tmp_column_name in report_config['bitsColumns']:
                result.append((column_name, ColumnType.Bits, tmp_column_name))
            elif tmp_column_name in report_config['wideBitsColumns']:
                result.append((column_name, ColumnType.WideBits, tmp_column_name))
            elif tmp_column_name in report_config['arrayColumns']:
                result.append((column_name, ColumnType.Array, tmp_column_name))
            elif tmp_column_name in report_config['dictionaryColumns']:
                result.append((report_config['dictionaryColumns'][tmp_column_name], ColumnType.Dictionary, tmp_column_name))
            else:
                result.append((column_name, ColumnType.Other, tmp_column_name))

        for group in report_config['groupedColumns']:
            group_is_correct = True
            current_group = list()
            for item in group:
                if item not in grouped_columns:
                    group_is_correct = False
                    break
                else:
                    current_group.append(grouped_columns[item])

            logging.info(group)
            if group_is_correct:
                result.append((', '.join(current_group), ColumnType.Group))

        return result

    @staticmethod
    def _get_table_columns(report_config: dict, query_builder, query_executor) -> list:
        logging.info(f"Loading table {report_config['tableName']} structure")

        query = query_builder.get_table_columns_query()
        table_columns = query_executor.execute_schema_query(query)

        logging.info(f"Table {report_config['tableName']} structure is loaded")

        return table_columns

    @staticmethod
    def _get_values_description(yql_token: str, yql_temp_folder: str) -> dict:
        description = dict()

        query_executor = QueryExecutorFactory.get_query_executor(Database.YT, yql_temp_folder, yql_token)

        query = "select GoodEvent, Description from `home/yabs/dict/GoodEvent`;"
        description['fraudbits'] = AnomalyAudit._get_column_value_description(query_executor, query, 'FraudBits')

        query = "select GoodEvent, Description from `home/yabs/dict/WideGoodEvent`;"
        description['widefraudbits'] = AnomalyAudit._get_column_value_description(
            query_executor, query, 'WideFraudBits')

        query = "select Math::Round(Math::Log2(DropCostBit)), Description from `home/yabs/dict/GoodEvent` where DropCostBit != 0;"
        description['costbits'] = AnomalyAudit._get_column_value_description(query_executor, query, 'CostBits')

        query = "select RegionID, Name from `home/yabs/dict/RegionName` where Lang == 'ru';"
        region_description = AnomalyAudit._get_column_value_description(query_executor, query, 'RegionID')
        description['regionid'] = region_description
        description['origregionid'] = region_description
        description['statisticregionid'] = region_description
        description['geocoefregionid'] = region_description

        query = "select BMCategoryID, Description from `home/yabs-cs/export/BMCategory`;"
        description['bmcategoryid'] = AnomalyAudit._get_column_value_description(query_executor, query, 'BMCategoryID')

        query = "select PageID, Description from `home/yabs/dict/Page`;"
        description['pageid'] = AnomalyAudit._get_column_value_description(query_executor, query, 'PageID')

        query = "select SelectType, Description from `home/yabs/dict/replica/SelectType`;"
        description['selecttype'] = AnomalyAudit._get_column_value_description(query_executor, query, 'SelectType')

        query = "select id, name from `home/balance/prod/dict/v_product`;"
        description['productid'] = AnomalyAudit._get_column_value_description(query_executor, query, 'v_product id')

        query = "select SSPID, Description from `home/yabs/dict/SSPInfo`;"
        description['sspid'] = AnomalyAudit._get_column_value_description(query_executor, query, 'SSPID')

        banner_flags = dict()
        for item in get_banner_flags_list():
            banner_flags[item['Flag']] = item['Description']

        description['bannerflags'] = banner_flags

        return description

    @staticmethod
    def _get_column_value_description(query_executor, query: str, column_name: str) -> dict:
        logging.info(f'Loading {column_name} description: {query}')
        result_description = {}

        try:
            result = query_executor.execute_query(query)

            while result.is_next_row():
                result_row = result.get_next_row()
                result_description[abs(result_row[0])] = result_row[1]  # ID column in Bit dictionary has sign
        except Exception as error:
            logging.error(f'Error {error} occurred while loading description for {column_name}')

        return result_description
