from enum import Enum
from dataclasses import dataclass
import logging
from library.python import resource
import json
from jsonschema import validate


class Cluster(Enum):
    Hahn = 'hahn'
    HahnCHYT = 'chyt.hahn'
    ClickHouse = 'yabsclickhouse'


class Database(Enum):
    YT = 'YT'
    ClickHouse = 'CH'
    ClickHouseOverYT = 'CHYT'

    @classmethod
    def has_value(cls, value):
        return value in cls._value2member_map_


class SelectLevel(Enum):
    Date = 'Date'
    DateTIme = 'DateTime'


class OutputFormat(Enum):
    HTML = 'html'
    TXT = 'txt'
    CSV = 'csv'


@dataclass
class Threshold:
    EXPAND_VALUE: int = 0
    SMALL_EXPAND_VALUE: int = 0
    LOWER_SIGNIFICANT_BOUND: float = 0


@dataclass
class Totals:
    current: int = 0
    previous: int = 0

    current_good: int = 0
    previous_good: int = 0


# https://a.yandex-team.ru/arc_vcs/sandbox/projects/Afisha/infra/AfishaStPinger/ya.make?rev=r8965736
default_config_resource_path = "sandbox/projects/yabs/AnomalyAuditReport/lib/default_config.yaml"


mutable_properties = {'thresholds', 'outputFormat', 'saveTo', 'columnsManualSettings'}


class AnomalyAuditConfigReader:
    @staticmethod
    def get_config(report_configs: list[dict]) -> list:

        if not report_configs:
            raise ValueError('Report Configurations are not supplied')

        resource_content = resource.find('sandbox/projects/yabs/AnomalyAuditReport/lib/config_schema.json')
        if resource_content is None:
            raise ValueError('Resource config_schema.json is not found')

        report_config_schema = json.loads(resource_content)

        validate(report_configs, report_config_schema)

        for report_config in report_configs:

            if report_config['selectGoodEvents'] and not report_config['whereClauseGoodEvents']:
                raise ValueError('Where Clause to Select Good events is empty')

            report_config['cluster'] = Cluster(report_config['cluster'].lower())
            report_config['database'] = Database(report_config['database'])

            if report_config['cluster'] == Cluster.HahnCHYT and report_config['database'] != Database.ClickHouseOverYT:
                raise ValueError('Database and Cluster are not matched')

            if report_config['cluster'] == Cluster.ClickHouse and report_config['database'] != Database.ClickHouse:
                raise ValueError('Database and Cluster are not matched')

            report_config['selectLevel'] = SelectLevel(report_config['selectLevel'])
            if report_config['selectLevel'] == SelectLevel.DateTIme and report_config['database'] == Database.ClickHouse:
                raise ValueError('ClickHouse does not support DateTime selection')

            report_config['outputFormat'] = OutputFormat(report_config['outputFormat'].lower())

            thresholds = dict()
            for threshold in report_config['thresholds']:
                threshold_name = 'default'
                if 'column_name' in threshold:
                    threshold_name = threshold['column_name'].strip().lower()

                thresholds[threshold_name] = Threshold(threshold['expand_value'],
                                                       threshold['small_expand_value'],
                                                       threshold['lower_significant_bound']
                                                       )
            report_config['thresholds'] = thresholds

            manual_columns = set()
            for item in report_config['manualColumns']:
                manual_columns.add(item.strip().lower())
            report_config['manualColumns'] = manual_columns

            bits_columns = set()
            for item in report_config['bitsColumns']:
                bits_columns.add(item.strip().lower())
            report_config['bitsColumns'] = bits_columns

            wide_bits_columns = set()
            for item in report_config['wideBitsColumns']:
                wide_bits_columns.add(item.strip().lower())
            report_config['wideBitsColumns'] = wide_bits_columns

            array_columns = set()
            for item in report_config['arrayColumns']:
                array_columns.add(item.strip().lower())
            report_config['arrayColumns'] = array_columns

            grouped_columns = set()
            for group in report_config['groupedColumns']:
                for item in group:
                    item = item.strip().lower()
                    grouped_columns.add(item)
            report_config['groupedColumnsSet'] = grouped_columns

            excluded_columns = set()
            for item in report_config['excludedColumns']:
                excluded_columns.add(item.strip().lower())
            report_config['excludedColumns'] = excluded_columns

            dictionary_columns = dict()
            for item in report_config['dictionaryColumns']:
                dictionary_columns[item['columnName'].strip().lower()] = item['selectValue']
            report_config['dictionaryColumns'] = dictionary_columns

        logging.debug(report_config['thresholds'])

        return report_configs
