from abc import ABC, abstractmethod
import logging
from retrying import retry

from yql.api.v1.client import YqlClient
from yql.client import operation
import yt.wrapper as yt

from sandbox.projects.yabs.AnomalyAuditReport.lib.query_result import YQLQueryResult
from sandbox.projects.yabs.AnomalyAuditReport.lib.query_result import QueryResultBase
from sandbox.projects.yabs.AnomalyAuditReport.lib.config_reader import Cluster, Database


class QueryExecutorBase(ABC):
    def __init__(self, yql_token: str, cluster: Cluster, yql_tmp_folder: str = ''):
        self._yql_token = yql_token
        self._cluster = cluster
        self._yql_tmp_folder = yql_tmp_folder
        self._yql_client = YqlClient(db=cluster.value, token=yql_token)

    @retry(stop_max_attempt_number=3, wait_fixed=2000)
    def execute_query(self, query: str) -> QueryResultBase:
        try:
            request = self._get_request(query)
            request.run()

            results = request.get_results()
            if not results.is_success:
                msg = '\n'.join([str(err) for err in results.errors])
                logging.error('Error when executing query %s: %s' % (query, msg))

            return YQLQueryResult(results, request.share_url)
        except:
            logging.warning(f'Error while executing query {query}', exc_info=True)
            raise

    @abstractmethod
    def execute_schema_query(self, query: str) -> list:
        ...


class YQLQueryExecutor(QueryExecutorBase):

    def execute_schema_query(self, query: str) -> list[tuple]:
        result = self.execute_query(query)
        columns = []
        while result.is_next_row():
            current_row = result.get_next_row()
            columns.append((current_row[0], current_row[1]))  # First element is column name and the second is type

        return columns

    @abstractmethod
    def _get_request(self, query: str) -> operation.YqlSqlOperationRequest:
        ...


class ClickHouseQueryExecutor(YQLQueryExecutor):
    def _get_request(self, query: str) -> operation.YqlSqlOperationRequest:
        request = self._yql_client.query(f'use {self._cluster.value}; {query}', sql=False, clickhouse_syntax=True)
        request.type = operation.YqlOperationType.CLICKHOUSE

        return request


class YTQueryExecutor(YQLQueryExecutor):
    def __init__(self, yql_token: str, cluster: Cluster, yql_tmp_folder: str = ''):
        super().__init__(yql_token, cluster, yql_tmp_folder)

        yt.config["proxy"]["url"] = cluster.value
        yt.config["token"] = yql_token

    def execute_schema_query(self, query: str) -> list[tuple]:
        schema = yt.get(query)
        columns = []
        for column in schema:
            columns.append((column['name'], column['type']))

        return columns

    def _get_request(self, query: str) -> operation.YqlSqlOperationRequest:
        # Доступ к временной папке YT при работе из YQL https://yql.yandex-team.ru/docs/yt/quickstart#outstaff-yt-tmp
        request = self._yql_client.query(f"PRAGMA yt.TmpFolder='{self._yql_tmp_folder}'; {query}", sql=True)
        request.type = operation.YqlOperationType.SQLv1

        return request


class QueryExecutorFactory:

    @staticmethod
    def get_query_executor_by_config(report_config: dict, yql_token: str) -> QueryExecutorBase:
        database = report_config['database']
        cluster = report_config['cluster']

        if database == Database.ClickHouse:
            return ClickHouseQueryExecutor(yql_token, cluster)
        elif database == Database.YT:
            return YTQueryExecutor(yql_token, cluster)
        elif database == Database.ClickHouseOverYT:
            return ClickHouseQueryExecutor(yql_token, cluster)
        else:
            raise ValueError(f'Database {database} is not supported')

    @staticmethod
    def get_query_executor(database: Database, yql_tmp_folder: str, yql_token: str) -> QueryExecutorBase:
        if database == Database.ClickHouse:
            return ClickHouseQueryExecutor(yql_token, Cluster.ClickHouse, yql_tmp_folder)
        elif database == Database.YT:
            return YTQueryExecutor(yql_token, Cluster.Hahn, yql_tmp_folder)
        elif database == Database.ClickHouseOverYT:
            return ClickHouseQueryExecutor(yql_token, Cluster.HahnCHYT, yql_tmp_folder)
        else:
            raise ValueError(f'Database {database} is not supported')
