from abc import ABC, abstractmethod
from collections import defaultdict
from dataclasses import dataclass
import prettytable
from sandbox.projects.yabs.AnomalyAuditReport.lib.query_result import QueryResultBase
from sandbox.projects.yabs.AnomalyAuditReport.lib.config_reader import OutputFormat, Totals


@dataclass
class RowResult:
    column_value: any
    current_result: int = 0
    previous_result: int = 0
    current_good_result: int = 0
    previous_good_result: int = 0


class ReportResultContainerBase(ABC):
    def __init__(self, report_config, current_date, previous_date, totals: Totals):
        super().__init__()

        self.report_id = report_config['reportId']
        self.report_name = report_config['reportName']

        self._thresholds = report_config['thresholds']

        self._select_good_events = report_config['selectGoodEvents']

        self.current_date = current_date
        self.previous_date = previous_date

        self.totals = totals

        self.value_descriptions = dict()
        self._result_rows = defaultdict(list[RowResult])
        self._column_query_url = dict()

    def add_value_descriptions(self, value_descriptions: dict):
        self.value_descriptions = value_descriptions

    def get_container_attributes(self) -> dict:
        attributes = {'report_id': self.report_id,
                      'report_name': self.report_name,
                      'output_format': self.output_format.value,
                      'current_date': self.current_date,
                      'previous_date': self.previous_date,
                      'current': self.totals.current,
                      'current_good': self.totals.current_good,
                      'previous': self.totals.previous,
                      'previous_good': self.totals.previous_good
                      }

        return attributes

    def add_result(self, query_result: QueryResultBase):
        columns = query_result.get_result_columns()
        if not columns:
            return

        # Two last columns are current_result and previous_result. Others are Group By columns
        main_column = "/".join(columns[0:-4])
        self._column_query_url[main_column] = query_result.query_url

        while query_result.is_next_row():
            row = query_result.get_next_row()
            self._result_rows[main_column].append(
                RowResult("/".join(map(str, row[0:-4])), row[-4], row[-3], row[-2], row[-1]))

    def has_result(self):
        if self._result_rows:
            return True
        else:
            return False

    def get_result(self) -> str:
        return self._get_table_header() + self._get_table_content()

    def _get_table_header(self) -> str:
        header = "Report Name: {}\nCurrent Date: {}\nPrevious Date: {}\nCurrent Total: {}\nPrevious Total: {}\n".format(self.report_name,
                                                                                                                        self.current_date,
                                                                                                                        self.previous_date,
                                                                                                                        self.totals.current,
                                                                                                                        self.totals.previous
                                                                                                                        )

        if self._select_good_events:
            header += "Current Good Total: {}\nPrevious Good Total: {}\n".format(
                self.totals.current_good, self.totals.previous_good)

        return header

    @abstractmethod
    def _get_table_content(self) -> str:
        ...

    def _get_result_table(self) -> prettytable.PrettyTable:
        result_table = prettytable.PrettyTable()

        if self._select_good_events:
            result_table.field_names = ["Номер", "Аномалия",
                                        "Имя Колонки", "Значение Колонки", "Описание значения",
                                        "Значение До", "Значение После",
                                        "Процент До", "Процент После",
                                        "Изменение (%)",
                                        "Значение хороших До", "Значение хороших После",
                                        "Доля засчитанных до(%)", "Доля засчитанных после(%)",
                                        "Изменение хороших (%)", "Query URL"
                                        ]
        else:
            result_table.field_names = ["Номер", "Аномалия",
                                        "Имя Колонки", "Значение Колонки", "Описание значения",
                                        "Значение До", "Значение После",
                                        "Процент До", "Процент После",
                                        "Изменение (%)", "Query URL"
                                        ]

        counter = 1

        for column_name in self._result_rows:

            threshold = self._thresholds.get(column_name.strip().lower())
            if not threshold:
                threshold = self._thresholds.get('default')

            value_description = self.value_descriptions.get(column_name.strip().lower(), {})

            for row in self._result_rows[column_name]:
                current_row = [
                    counter,
                    self._get_anomaly_name(row.current_result, row.previous_result, threshold),
                    column_name,
                    row.column_value,
                    value_description.get(row.column_value if not row.column_value.isnumeric()
                                          else int(row.column_value), ''),

                    # Значение До После
                    row.previous_result,
                    row.current_result,

                    # Процент До После
                    "{:.3f}".format(
                        row.previous_result / self.totals.previous * 100),
                    "{:.3f}".format(
                        row.current_result / self.totals.current * 100),

                    "{:.3f}".format(
                        abs(row.current_result - row.previous_result) / row.previous_result * 100) if row.previous_result > 0 else 0
                ]

                if self._select_good_events:
                    current_row += [
                        # Значение До После
                        row.previous_good_result,
                        row.current_good_result,

                        # Доля засчитанных До После
                        "{:.3f}".format(
                            row.previous_good_result / row.previous_result * 100) if row.previous_result > 0 else 0,
                        "{:.3f}".format(
                            row.current_good_result / row.current_result * 100) if row.current_result > 0 else 0,

                        "{:.3f}".format(
                            abs(row.current_good_result - row.previous_good_result) / row.previous_good_result * 100) if row.previous_good_result > 0 else 0
                    ]

                current_row.append(self._column_query_url.get(column_name, ''))
                result_table.add_row(current_row)
                counter += 1

        return result_table

    def _get_anomaly_name(self, current_value, previous_value, threshold) -> str:

        if previous_value == 0 and current_value > 0 and current_value / self.totals.current > threshold.LOWER_SIGNIFICANT_BOUND:
            return 'Появилось значение'

        if previous_value > 0 and current_value == 0 and previous_value / self.totals.previous > threshold.LOWER_SIGNIFICANT_BOUND:
            return 'Пропало значение'

        if current_value > self.totals.current * threshold.LOWER_SIGNIFICANT_BOUND and previous_value > 0:
            if current_value / self.totals.current > previous_value / self.totals.previous * threshold.SMALL_EXPAND_VALUE:
                return 'Возросло'
            elif current_value / self.totals.current * threshold.SMALL_EXPAND_VALUE < previous_value / self.totals.previous:
                return 'Упало'

        if previous_value > self.totals.previous * threshold.LOWER_SIGNIFICANT_BOUND and current_value > 0:
            if current_value / self.totals.current > previous_value / self.totals.previous * threshold.SMALL_EXPAND_VALUE:
                return 'Возросло'
            elif current_value / self.totals.current * threshold.SMALL_EXPAND_VALUE < previous_value / self.totals.previous:
                return 'Упало'

        if current_value < self.totals.current * threshold.LOWER_SIGNIFICANT_BOUND and current_value > 0 and previous_value > 0:
            if current_value / self.totals.current > previous_value / self.totals.previous * threshold.EXPAND_VALUE:
                return 'Возросло сильно'
            elif current_value / self.totals.current * threshold.EXPAND_VALUE < previous_value / self.totals.previous:
                return 'Упало сильно'

        if previous_value < self.totals.previous * threshold.LOWER_SIGNIFICANT_BOUND and current_value > 0 and previous_value > 0:
            if current_value / self.totals.current > previous_value / self.totals.previous * threshold.EXPAND_VALUE:
                return 'Возросло сильно'
            elif current_value / self.totals.current * threshold.EXPAND_VALUE < previous_value / self.totals.previous:
                return 'Упало сильно'

        return 'Not Defined'


class TxtAnomalyAuditReportContainer(ReportResultContainerBase):

    def __init__(self, report_config, current_date, previous_date, totals: Totals):
        super().__init__(report_config, current_date, previous_date, totals)

        self.output_format = OutputFormat.TXT

    def _get_table_content(self) -> str:
        result_table = self._get_result_table()
        result_table.border = True
        result_table.header = True

        return result_table.get_string()


class HTMLAnomalyAuditReportContainer(ReportResultContainerBase):

    def __init__(self, report_config, current_date, previous_date, totals: Totals):
        super().__init__(report_config, current_date, previous_date, totals)

        self.output_format = OutputFormat.HTML

    def _get_table_content(self) -> str:
        result_table = self._get_result_table()
        result_table.format = True
        result_table.border = True
        result_table.hrules = prettytable.ALL
        result_table.vrules = prettytable.ALL

        return result_table.get_html_string(attributes={"id": {self.report_id}, "class": "red_table"})


class CSVAnomalyAuditReportContainer(ReportResultContainerBase):

    def __init__(self, report_config, current_date, previous_date, totals: Totals):
        super().__init__(report_config, current_date, previous_date, totals)

        self.output_format = OutputFormat.CSV

    def _get_table_header(self) -> str:
        return ''

    def _get_table_content(self) -> str:
        result_table = self._get_result_table()

        return result_table.get_csv_string()


class AnomalyAuditReportFactory:

    @ staticmethod
    def get_report_result_container(report_config, current_date, previous_date, totals: Totals) -> ReportResultContainerBase:

        if report_config['outputFormat'] == OutputFormat.HTML:
            return HTMLAnomalyAuditReportContainer(report_config, current_date, previous_date, totals)
        elif report_config['outputFormat'] == OutputFormat.TXT:
            return TxtAnomalyAuditReportContainer(report_config, current_date, previous_date, totals)
        elif report_config['outputFormat'] == OutputFormat.CSV:
            return CSVAnomalyAuditReportContainer(report_config, current_date, previous_date, totals)
        else:
            raise ValueError(f"Output Format {report_config['outputFormat']} is not supported")
