from datetime import datetime, timedelta
from sandbox import sdk2
from sandbox import common

from sandbox.projects.yabs.AnomalyAuditReport.sandbox_task.task_resources import AnomalyAuditReportResource
from sandbox.projects.yabs.AnomalyAuditReport.lib.anomaly_audit import AnomalyAudit
from sandbox.projects.yabs.AnomalyAuditReport.lib.report_container import ReportResultContainerBase


class AnomalyAuditReportTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        description = 'Anomaly Audit Report'

        push_tasks_resource = True

        with sdk2.parameters.Group('Dates') as dates_block:
            current_date = sdk2.parameters.String(
                'Current Date (YYYY-MM-DDTHH:00:00)',
                default=(datetime.now() + timedelta(days=-1)).strftime("%Y-%m-%dT%H:00:00"),
                required=True
            )

            previous_date = sdk2.parameters.String(
                'Date to Compare With (YYYY-MM-DDTHH:00:00)',
                default=(datetime.now() + timedelta(days=-8)).strftime("%Y-%m-%dT%H:00:00"),
                required=True
            )

        with sdk2.parameters.Group('YQL Settings') as yav_block:
            tokens = sdk2.parameters.YavSecret(
                'YAV secret with YQL Token',
                required=True,
                default='sec-01fsa4f05jf6qx82paddraw9fk'
            )

            yql_temp_folder = sdk2.parameters.String(
                'Temporary Folder in YT for YQL requests',
                required=True,
                default='//tmp/yql'
            )

        with sdk2.parameters.Group('Report To Generate') as reports_block:
            report_resource_ttl = sdk2.parameters.Integer('TTL for Report Resource FIle', default_value=14)

            load_column_dictionaries = sdk2.parameters.Bool('Load Column Dictionaries', default_value=False)

            report_configs = sdk2.parameters.JSON('Configuration of Reports to generate', required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        self.set_info('Anomaly Audit Report. Start working')

        result, errors = AnomalyAudit.create_anomaly_audit_report(self.Parameters.current_date,
                                                                  self.Parameters.previous_date,
                                                                  self.Parameters.report_configs,
                                                                  AnomalyAuditReportTask.get_token(
                                                                      self.Parameters.tokens, 'yql_token'),
                                                                  self.Parameters.yql_temp_folder,
                                                                  self.Parameters.load_column_dictionaries
                                                                  )
        self.set_info('Anomaly Audit Report. Finished')
        self.set_info('Process Result')

        has_result = False
        for report_container in result:
            if not report_container.has_result():
                continue

            has_result = True

            resource_data = sdk2.ResourceData(self._get_resource(report_container))
            with resource_data.path.open('w') as f:
                f.write(report_container.get_result())

            resource_data.ready()

        if errors:
            for error in errors:
                self.set_info(error)
            raise common.errors.TaskFailure('Errors Occurred! Please check the Log!')

        if has_result:
            self.set_info("Anomaly Audit Report. Completed with differences. Please Check!")
        else:
            self.set_info("Anomaly Audit Report. Completed without differences")

    def _get_resource(self, report_container: ReportResultContainerBase) -> sdk2.Resource:

        attributes = report_container.get_container_attributes()
        attributes['ttl'] = self.Parameters.report_resource_ttl

        return AnomalyAuditReportResource(self, attributes['report_name'], f"{attributes['report_name']}.{attributes['output_format']}", **attributes)

    @ staticmethod
    def get_token(yav_token_param: sdk2.parameters.YavSecret, default_key_name: str):
        tokens = yav_token_param.data()
        token_key = yav_token_param.default_key if yav_token_param.default_key else default_key_name
        token = tokens[token_key]
        if token:
            return token
        else:
            raise ValueError(f'{token_key} is not found in YavSecret')
