from datetime import datetime, timedelta
from sandbox import sdk2
import sandbox.common.types.notification as Notification
import sandbox.common.types.task as TaskCommonType
import sandbox.common.types.resource as ResourceCommonType
from startrek_client import Startrek
import requests
from collections import defaultdict
from library.python import resource as project_resource
import json
from prettytable import PrettyTable
import csv

from sandbox.projects.yabs.qa.utils.general import get_task_html_hyperlink, get_resource_html_hyperlink, html_hyperlink
from sandbox.projects.yabs.AnomalyAuditReport.sandbox_task import AnomalyAuditReportTask

from sandbox.common.types.task import ReleaseStatus
from sandbox.common.types.resource import State


ST_API_URL = 'https://st-api.yandex-team.ru'
ST_URL = 'https://st.yandex-team.ru/'


ST_API_TEST_URL = 'https://st-api.test.yandex-team.ru'
ST_TEST_URL = 'https://st.test.yandex-team.ru/'


DUTY_PAGE_API = 'https://wiki-api.yandex-team.ru/_api/frontend/BannernajaKrutilka/duty/DutyPlan/.grid?fromat=json'
SANDBOX_URL = 'https://sandbox.yandex-team.ru'


class AnomalyAuditReportWrapperTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        description = 'Anomaly Audit Report'
        push_tasks_resource = True

        current_date = sdk2.parameters.String(
            'Current Date (YYYY-MM-DD)',
            default=(datetime.now() + timedelta(days=-1)).strftime("%Y-%m-%dT%H:00:00"),
            required=False
        )

        compare_with = sdk2.parameters.List(
            'Compare with (offset from Current Date)',
            default=[-1, -7],
            required=True
        )

        st_token = sdk2.parameters.YavSecret(
            'YAV secret with Startrek Token',
            required=True,
            default='sec-01dc7ehd7m4m0f3ghxjtfbxqks'
        )

        st_queue = sdk2.parameters.String(
            'Startrek Queue',
            required=True,
            default='BSAUDIT'
        )

        st_type = sdk2.parameters.String(
            'Startrek Issue Type',
            required=True,
            default='Task'
        )

        st_ticket_summary = sdk2.parameters.String(
            'Startrek Ticket Summary',
            required=False,
            default='Anomaly Audit Report'
        )

        st_use_test_serv = sdk2.parameters.Bool('Use Startrek test server', default_value=True)

        with sdk2.parameters.Group('YQL Settings') as yav_block:
            yql_token = sdk2.parameters.YavSecret(
                'YAV secret with YQL Token (override in Sub Task)',
                required=False
            )

        notifications = [
            sdk2.Notification(
                [TaskCommonType.Status.FAILURE, TaskCommonType.Status.EXCEPTION, TaskCommonType.Status.Group.BREAK],
                ['levshukov-v', 'kochurovvs'],
                Notification.Transport.EMAIL
            )
        ]

        ticket_followers = sdk2.parameters.List(
            'Additional Ticket Followers',
            default=['kochurovvs'],
            required=False
        )

        with sdk2.parameters.Group('Report To Generate') as reports_block:
            report_resource_ttl = sdk2.parameters.Integer('TTL for Report Resource FIle', default_value=14)

            load_column_dictionaries = sdk2.parameters.Bool('Load Column Dictionaries', default_value=False)

            def get_config():
                resource_content = project_resource.find('sandbox/projects/yabs/AnomalyAuditReportSandboxWrapper/sandbox_task/default_config.json')
                if resource_content is not None:
                    try:
                        return json.loads(resource_content)
                    except Exception:
                        return ''
                else:
                    return ''

            report_configs = sdk2.parameters.JSON(
                'Configuration of Reports to generate',
                default_value=get_config(),
                required=True
            )

        search_released_resource = sdk2.parameters.Bool(
            "Search for Released Resource",
            default=False
        )

        release_version = sdk2.parameters.String(
            "Release version",
            choices=[(status, status) for status in ReleaseStatus],
            default=ReleaseStatus.TESTING
        )

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        sub_tasks_ids = list()
        current_date = ''

    def on_create(self):
        if not self.Parameters.search_released_resource:
            return

        tasks_resource = None
        if self.Parameters.release_version == ReleaseStatus.TESTING:
            tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={'name': 'AnomalyAuditReportResource', 'released': 'testing'},
                state=[State.READY],
            ).first()

        if not tasks_resource:
            tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={'name': 'AnomalyAuditReportResource', 'released': 'stable'},
                state=[State.READY],
            ).first()

        self.Requirements.tasks_resource = tasks_resource

    def on_execute(self):
        if self.Context.sub_tasks_ids:
            self.set_info('Anomaly Audit Wrapper Report. Sub Tasks are completed')
            return

        self.set_info('Anomaly Audit Wrapper Report. In Process')

        parameters = dict()

        parameters['owner'] = self.owner

        if self.Parameters.yql_token:
            parameters['tokens'] = self.Parameters.yql_token

        parameters['report_resource_ttl'] = self.Parameters.report_resource_ttl
        parameters['load_column_dictionaries'] = self.Parameters.load_column_dictionaries

        if not self.Parameters.current_date:
            parameters['current_date'] = (datetime.now() + timedelta(days=-1)).strftime("%Y-%m-%dT%H:00:00")
            self.Context.current_date = parameters['current_date']
            current_date = (datetime.now() + timedelta(days=-1))
        else:
            parameters['current_date'] = self.Parameters.current_date
            self.Context.current_date = parameters['current_date']
            current_date = datetime.fromisoformat(self.Parameters.current_date)

        for item in self.Parameters.compare_with:
            parameters['previous_date'] = (datetime.date(current_date + timedelta(days=int(item)))).strftime("%Y-%m-%dT%H:00:00")

            self.set_info(f'Anomaly Audit Wrapper Report. Create Sub Tasks for {item} day(s)')
            self._run_sub_tasks(parameters)

        self.set_info(f'Anomaly Audit Wrapper Report. Waiting for {len(self.Context.sub_tasks_ids)} Sub Tasks')

        # После того как событие произойдёт, задача будет переключена в статус ENQUEUED и начнёт исполнение заново.
        raise sdk2.WaitTask(self.Context.sub_tasks_ids, TaskCommonType.Status.Group.FINISH, wait_all=True)

    def _run_sub_tasks(self, parameters: dict):
        for report_config in self.Parameters.report_configs:
            parameters['report_configs'] = [report_config]
            self._run_sub_task(report_config['reportId'], parameters)

    def _run_sub_task(self, report_id: str, parameters: dict):
        parameters['description'] = f"{report_id} {parameters['current_date']}-{parameters['previous_date']}"
        sub_task = AnomalyAuditReportTask(self, **parameters)
        sub_task.enqueue()

        self.Context.sub_tasks_ids.append(sub_task.id)
        self.set_info('Sub Task {} is created'.format(get_task_html_hyperlink(sub_task.id)), do_escape=False)

    def on_finish(self, prev_status, status):
        self.set_info('Anomaly Audit Wrapper Report. Process Sub Tasks Results')

        on_duty_colleagues = self._get_bsaudit_on_duty()
        self.set_info(f'BSAUDIT On Duty: {on_duty_colleagues}')

        if not on_duty_colleagues:
            self.set_info('BSAUDIT On Duty is empty')
            on_duty_colleagues = ['levshukov-v']

        on_duty_colleagues += self.Parameters.ticket_followers

        issue = self._create_ticket_in_startrek(f'{self.Parameters.st_ticket_summary} {self.Context.current_date}',
                                                f'Sandbox Task  (({SANDBOX_URL}/task/{self.id} {SANDBOX_URL}/task/{self.id}))',
                                                on_duty_colleagues)

        has_anomaly_reports = False
        for sub_task_id in self.Context.sub_tasks_ids:
            sub_task = sdk2.Task[sub_task_id]

            if sub_task.status != TaskCommonType.Status.SUCCESS:
                self.set_info('Sub Task {} has an error'.format(get_task_html_hyperlink(sub_task.id)), do_escape=False)
                issue.comments.create(text=f'Sandbox Task has an issue! Please check and re-run (({SANDBOX_URL}/task/{sub_task.id} {SANDBOX_URL}/task/{sub_task.id}))',
                                      summonees=on_duty_colleagues)

            sub_task_resources = sdk2.Resource.find(task_id=sub_task.id).limit(10)
            for resource in sub_task_resources:
                if 'ANOMALY_AUDIT_REPORT' in str(resource.type):
                    has_anomaly_reports = True
                    if resource.state == ResourceCommonType.State.READY:
                        self._create_comment_with_result(issue, resource, on_duty_colleagues)
                    else:
                        self.set_info('Resource {} is broken'.format(get_resource_html_hyperlink(resource.id)), do_escape=False)
                        issue.comments.create(text=f'Sandbox Task has an issue! Resource is broken: (({SANDBOX_URL}/task/{sub_task.id} {SANDBOX_URL}/task/{sub_task.id}))',
                                              summonees=on_duty_colleagues)

        if not has_anomaly_reports:
            issue.comments.create(text='Anomaly Audit Report completed without differences. Please Close this Task',
                                  summonees=on_duty_colleagues)

    def _create_comment_with_result(self, issue, resource: sdk2.Resource, on_duty_colleagues: list):
        self.set_info('Create Comment by Resource {}'.format(get_resource_html_hyperlink(resource.id)), do_escape=False)

        comment = f'Anomaly Audit Report {resource.report_name} {resource.current_date} / {resource.previous_date}: (({resource.http_proxy} {resource.http_proxy}))\n'
        comment += f'Current/Previous Totals: {resource.current} {resource.previous}\n'
        comment += f'Current/Previous Good Totals: {resource.current_good} {resource.previous_good}\n'

        resource_data = sdk2.ResourceData(resource)
        with resource_data.path.open('r') as f:
            rd = csv.reader(f, delimiter=',')
            pt = PrettyTable(next(rd))
            for row in rd:
                pt.add_row(row)
                self._add_item_to_check_list(issue, {'text': f'{resource.report_name} {" ".join(row[1:4])}', 'checked': False})

            comment += pt.get_html_string(attributes={"id": {resource.report_id}, "class": "red_table"})

        issue.comments.create(text=comment, summonees=on_duty_colleagues)

    def _create_ticket_in_startrek(self, summary: str, description: str, on_duty_colleagues: list):

        client = Startrek(useragent='Anomaly Audit Report',
                          base_url=ST_API_TEST_URL if self.Parameters.st_use_test_serv else ST_API_URL,
                          token=AnomalyAuditReportTask.get_token(self.Parameters.st_token, 'startrek_token'))

        try:
            issue = client.issues.create(
                queue=self.Parameters.st_queue,
                type={'name': self.Parameters.st_type},
                summary=summary,
                description=description,
                assignee=on_duty_colleagues[0],
                followers=on_duty_colleagues,
                tags=['tracker_queue', 'daily_eventlog_analysis'],
            )

            issue_url = f'{ST_TEST_URL if self.Parameters.st_use_test_serv else ST_URL}{issue.key}'
            self.set_info('Ticket {} is created'.format(html_hyperlink(issue_url, issue.key)), do_escape=False)

            return issue

        except Exception as error:
            self.set_info(f'Error while creating Startrek issue: {error}')

    def _add_item_to_check_list(self, issue, check_list_item: dict):
        requests.post('{}/v2/issues/{}/checklistItems'.format(ST_API_TEST_URL if self.Parameters.st_use_test_serv else ST_API_URL, issue.key),
                      data=json.dumps([check_list_item]),
                      headers={'Authorization': 'OAuth {}'.format(AnomalyAuditReportTask.get_token(self.Parameters.st_token, 'startrek_token')),
                               'Content-type': 'application/json'},
                      verify=False
                      )

    def _get_bsaudit_on_duty(self):
        dt = datetime.now()
        first_day_of_week = str(datetime.date(dt - timedelta(days=dt.weekday())))
        self.set_info(f'First Day of the Week: {first_day_of_week}')

        on_duty_colleagues = self.get_current_week_on_duty(first_day_of_week, sdk2.Vault.data("YABS-YTSTAT", "yabs_duty_token"))
        return on_duty_colleagues.get('BSAUDIT', list())

    def _get_column_name(self, column):
        if column[-1].isdigit():
            column = column[:-1]
        if column[0] == '*':
            column = column[1:]
        return column

    def get_current_week_on_duty(self, first_day_of_week, wiki_token):
        headers = {'Authorization': 'OAuth ' + wiki_token}
        request = requests.get(DUTY_PAGE_API, headers=headers, verify=False)

        rows = request.json()['data']['rows']
        cols = [col['title'] for col in request.json()['data']['structure']['fields']]

        current_week_duties = []
        for row in reversed(rows):
            for i in range(len(row)):
                if row[i]['raw'] == first_day_of_week:
                    current_week_duties = row
                    break
            if current_week_duties:
                break

        result = defaultdict(list)
        for num, cell in enumerate(current_week_duties):
            column_name = self._get_column_name(cols[num])
            if isinstance(cell['raw'], list) and cell['raw']:
                result[column_name] += ([cell["raw"][i] for i in range(len(cell["raw"]))])

        return result
