from sandbox import sdk2
from sandbox.sandboxsdk import svn

import base64
import numbers
import os
import sys
import time

from constant import ID_SECRET, KEY_KEY_BASE64, KEY_TIME_TO, KEY_TIME_FROM, SIZE_KEY_MAX


class GenerateKey(sdk2.Task):
    """Generate new secret key"""

    name = "BANNER_USER_CHOICE_GENERATE_KEY"

    class Requirements(sdk2.Task.Requirements):
        disk_space = 50

    class Parameters(sdk2.Task.Parameters):
        description = "Generate new secret key"
        secret_id = sdk2.parameters.String("Use secret with id:", default=ID_SECRET, required=True)
        key_lifetime = sdk2.parameters.Integer("Key lifetime, seconds:", default=1 * 24 * 60 * 60, required=True)
        signature_lifetime = sdk2.parameters.Integer("Signature lifetime, seconds:", default=10 * 60, required=True)
        min_sum_key_lifetime = sdk2.parameters.Integer("Minimal total remaining key lifetime, seconds", default=7 * 24 * 60 * 60, required=True)

    def get_key_lifetime_verified(self):
        if not self.key_lifetime_verified:
            self.key_lifetime_verified = self.Parameters.key_lifetime
            assert self.key_lifetime_verified > 0
        return self.key_lifetime_verified

    def get_signature_lifetime_verified(self):
        if not self.signature_lifetime_verified:
            self.signature_lifetime_verified = self.Parameters.signature_lifetime
            assert self.signature_lifetime_verified > 0
        return self.signature_lifetime_verified

    def get_min_sum_key_lifetime_verified(self):
        if not self.min_sum_key_lifetime_verified:
            self.min_sum_key_lifetime_verified = self.Parameters.min_sum_key_lifetime
            assert self.min_sum_key_lifetime_verified > 0
        return self.min_sum_key_lifetime_verified

    def on_execute(self):
        self.key_lifetime_verified = None
        self.signature_lifetime_verified = None
        self.min_sum_key_lifetime_verified = None

        vault_oauth_token = sdk2.Vault.data('robot-yabs-cs', 'yabscs_vault_token')

        sys.path.append(svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/library/python/vault_client/"))
        sys.path.append(svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/yabs/server/cs/pylibs/banner_user_choice"))
        sys.path.append(svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/yabs/server/cs/pylibs/vault_content_manager/"))
        import vault_client
        import vault

        try:
            key_validities = vault.get_key_validities_json(module=vault_client, vault_oauth_token=vault_oauth_token, secret_id=self.Parameters.secret_id)
        except vault.ClientError as client_error:
            if client_error.code == vault.ClientError.ERROR_SECRET_ACCESS:
                raise client_error
            key_validities = []
        except vault.ContentError:
            key_validities = []

        time_now = int(round(time.time()))

        if isinstance(key_validities, list):
            if key_validities:
                key_validity_valid_count = 0
                for key_validity in key_validities:
                    try:
                        key_base64 = key_validity[KEY_KEY_BASE64]
                        time_from = key_validity[KEY_TIME_FROM]
                        time_to = key_validity[KEY_TIME_TO]
                    except KeyError:
                        break

                    try:
                        key = base64.b64decode(key_base64)
                    except TypeError:
                        break

                    if len(key) != SIZE_KEY_MAX:
                        break

                    if not isinstance(time_from, numbers.Integral) or not isinstance(time_to, numbers.Integral):
                        break

                    if time_from > time_to:
                        break

                    if time_now >= time_to + self.get_signature_lifetime_verified():
                        break

                    key_validity_valid_count = key_validity_valid_count + 1

                del key_validities[key_validity_valid_count:]
        else:
            key_validities = []

        if not key_validities:
            time_from = time_now
        else:
            time_from = key_validities[0][KEY_TIME_TO]
            if time_from < time_now:
                time_from = time_now
        time_to = time_from

        time_to_min = time_now + self.get_min_sum_key_lifetime_verified()

        while time_to < time_to_min:
            time_to += self.get_key_lifetime_verified()
            key_validities.insert(0, {
                KEY_KEY_BASE64: base64.b64encode(os.urandom(SIZE_KEY_MAX)),
                KEY_TIME_FROM: time_from,
                KEY_TIME_TO: time_to,
            })
            time_from = time_to

        vault.set_key_validities_json(key_validities, module=vault_client, vault_oauth_token=vault_oauth_token, secret_id=self.Parameters.secret_id)
