import logging
import time
import datetime
import re
from collections import defaultdict

from sandbox import sdk2

from sandbox.projects.yabs.CommitHeaven.lib.email_helper import EmailHelper
from sandbox.projects.yabs.CommitHeaven.lib.yt_helper import YTHelper
from sandbox.projects.yabs.CommitHeaven.lib.staff_helper import StaffHelper


logger = logging.getLogger(__name__)
SECONDS_IN_A_DAY = 24 * 60 * 60


class CommitHeaven(sdk2.Task):
    class Parameters(sdk2.Parameters):
        tokens = sdk2.parameters.YavSecret(
            "YAV secret identifier",
            default="sec-01fa0g0tcbqfky7nt5kjgwy898"
        )
        department_list = sdk2.parameters.List("List of departments subscribed to CommitHeaven results", default=["yandex_monetize_banner"], required=True)
        time_intervals_list_str = sdk2.parameters.List("List of time intervals for inspection", default=[30, 60, 90], required=True)
        stats_depth = sdk2.parameters.Integer("Number of chiefs to get commit info from subordinate", default=10, required=True)
        sort_by_interval = sdk2.parameters.Integer("Interval, by which sort stats", default=60, required=True)
        with sdk2.parameters.RadioGroup("Sorting by which parameter") as sort_by_parameter:
            sort_by_parameter.values["commits"] = sort_by_parameter.Value("Commits", default=True)
            sort_by_parameter.values["linesAdded"] = sort_by_parameter.Value("Lines added")
            sort_by_parameter.values["linesDeleted"] = sort_by_parameter.Value("Lines deleted")
        newbie = sdk2.parameters.Integer("How many days one counts as a newbie", default=90, required=True)
        dry_run = sdk2.parameters.Bool("Dry run, don't send any emails", default=False, required=True)
        send_info_per_chief = sdk2.parameters.Bool("Send info to all chiefs", default=True, required=True)
        with send_info_per_chief.value[False]:
            whom_to_send_emails = sdk2.parameters.List("Who will get emails with all stats (logins)", default=[], required=True)

    def min_timestamp(self, days):
        return int(time.time() - days * SECONDS_IN_A_DAY)

    def get_commits_from_time_interval(self, commits, days):
        start_time = self.min_timestamp(days)
        return [commit for commit in commits if commit['timestamp'] >= start_time]

    def filter_commits_to_junk(self, commits):
        return [commit for commit in commits if not re.search('(^|;)junk', commit['paths'])]

    def create_zero_stats(self, persons, intervals):
        return {
            person: {
                interval: {
                    'linesAdded': 0,
                    'linesDeleted': 0,
                    'commits': 0
                }
                for interval in intervals
            }
            for person in persons
        }

    def timestamp_from_date(self, date_str):
        return time.mktime(datetime.datetime.strptime(date_str, "%Y-%m-%d").timetuple())

    seen_people = set()

    def get_dfs_order(self, person, group_names, subordinates, dfs_order):
        logger.info(f"Person: {person}, subordinates: {subordinates[person]}")
        if person not in subordinates or person in self.seen_people:
            return
        self.seen_people.add(person)
        group = group_names[person]
        if group not in dfs_order:
            dfs_order.append(group)
        for subordinate in subordinates[person]:
            self.get_dfs_order(subordinate, group_names, subordinates, dfs_order)

    def on_execute(self):
        time_intervals_list = [int(days_str) for days_str in self.Parameters.time_intervals_list_str]

        self.set_info("Getting staff token")
        staff_token = self.Parameters.tokens.data()['staff_token']

        self.set_info("Getting staff helper")
        staff_helper = StaffHelper(staff_token)

        self.set_info("Getting all persons from all departments")
        self.set_info(self.Parameters.department_list)

        persons = set()
        for department in self.Parameters.department_list:
            persons |= set(staff_helper.get_all_persons_from_department(department))

        self.set_info("Getting person info from staff")
        chiefs = {}
        group_names = {}
        person_names = {}
        is_chief = {}
        is_newbie = {}
        for person in persons:
            chiefs[person] = staff_helper.get_chief_by_login(person)
            group_names[person] = staff_helper.get_group_name_by_login(person)
            person_names[person] = staff_helper.get_name_by_login(person)
            is_chief[person] = staff_helper.is_chief(person)
            joined_at = staff_helper.get_joined_at(person)
            is_newbie[person] = self.timestamp_from_date(joined_at) >= time.time() - self.Parameters.newbie * SECONDS_IN_A_DAY
        logger.info(f"Chief by login: {chiefs}")
        logger.info(f"Group name by login: {group_names}")
        logger.info(f"Name by login: {person_names}")
        logger.info(f"Is person a chief of his own group: {is_chief}")
        logger.info(f"Is person a newbie: {is_newbie}")

        self.set_info("Getting depth-first order of groups")

        subordinates = defaultdict(lambda: [])
        for person in persons:
            if chiefs[person] in persons:
                subordinates[chiefs[person]].append(person)
        dfs_order = []
        for person in persons:
            start = person
            while chiefs[start] in persons:
                start = chiefs[start]
            self.get_dfs_order(start, group_names, subordinates, dfs_order)

        logger.info(f"DFS order of groups: {dfs_order}")

        self.set_info("Getting YT token")
        yt_token = self.Parameters.tokens.data()['yt_token']

        self.set_info("Getting YT client")
        self.yt_helper = YTHelper(yt_token)

        self.set_info("Getting all commits")
        commits = self.yt_helper.get_all_commits_from_yt(persons, self.min_timestamp(max(time_intervals_list)))
        logger.info(f"All commits:\n{commits}")

        self.set_info("Getting all stats")
        self.set_info(f"Total commits fetched {len(commits)}")
        stats = self.create_zero_stats(persons, time_intervals_list)
        commits = self.filter_commits_to_junk(commits)
        for days in time_intervals_list:
            self.set_info(f"Getting stats for the past {days} days")
            for commit in self.get_commits_from_time_interval(commits, days):
                login = commit["login"]
                stats[login][days]["linesAdded"] += commit["linesAdded"]
                stats[login][days]["linesDeleted"] += commit["linesDeleted"]
                stats[login][days]["commits"] += 1

        logger.info(f"All stats: {stats}")

        self.set_info("Getting stats per chief")
        stats_per_chief = defaultdict(lambda: {})
        for login, stats_by_interval in stats.items():
            chief = login
            for level in range(self.Parameters.stats_depth + 1):
                if chief not in persons:
                    break
                stats_per_chief[chief][login] = stats_by_interval
                chief = chiefs[chief]

        dry_run = self.Parameters.dry_run

        if self.scheduler and self.scheduler > 0:
            if datetime.datetime.now().day != 20:
                self.set_info("Today's date is not 20, switching to dry-run")
                dry_run = True

        self.set_info("Getting email helper")
        email_helper = EmailHelper(dry_run, self.id)

        self.set_info("Sending emails in {}".format("dry-run mode" if dry_run else "normal mode"))
        if self.Parameters.send_info_per_chief:
            for chief, stats in stats_per_chief.items():
                if len(stats) > 1:
                    email_helper.sendresults(
                        to=chief,
                        unsorted_stats=stats,
                        intervals=time_intervals_list,
                        group_names=group_names,
                        person_names=person_names,
                        is_chief=is_chief,
                        is_newbie=is_newbie,
                        sort_by_interval=self.Parameters.sort_by_interval,
                        sort_by_parameter=self.Parameters.sort_by_parameter,
                        group_order=dfs_order,
                        task=self
                    )
        else:
            for to in self.Parameters.whom_to_send_emails:
                email_helper.sendresults(
                    to=to,
                    unsorted_stats=stats,
                    intervals=time_intervals_list,
                    group_names=group_names,
                    person_names=person_names,
                    is_chief=is_chief,
                    is_newbie=is_newbie,
                    sort_by_interval=self.Parameters.sort_by_interval,
                    sort_by_parameter=self.Parameters.sort_by_parameter,
                    group_order=dfs_order,
                    task=self
                )
