import logging
from collections import namedtuple, defaultdict

from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from smtplib import SMTP
import getpass


logger = logging.getLogger(__name__)


class EmailHelper():
    def __init__(self, dry_run, task_id):
        self.dry_run = dry_run
        self.task_id = task_id

    def sendmail(self, to, subject, text):
        msg = MIMEMultipart()
        msg["Subject"] = subject
        user = ""
        try:
            user = getpass.getuser()
        except BaseException as e:
            logger.info("getpass.getuser() error: {}".format(repr(e)))
        if not user:
            user = "yt_compare"
        froma = user + "@yandex-team.ru"
        msg["From"] = froma
        msg["To"] = to
        msg.attach(MIMEText(text, "html"))
        try:
            s = SMTP("outbound-relay.yandex.net")
            res = s.sendmail(froma, to.split(','), msg.as_string())
            if res:
                logger.info(f"Rejected sendmail from {froma} to:", res)
            s.quit()
        except BaseException as e:
            logger.info("sendmail error: {}".format(repr(e)))

    def sendresults(
        self,
        to,
        unsorted_stats,
        intervals,
        group_names,
        person_names,
        is_chief,
        is_newbie,
        sort_by_interval,
        sort_by_parameter,
        group_order,
        task
    ):
        logger.info(f"Length of unsorted_stats: {len(unsorted_stats)}")
        PersonStats = namedtuple("PersonStats", ["login", "stats"])
        sorted_stats = sorted([PersonStats(login, stats) for login, stats in unsorted_stats.items()],
                                key=lambda x: x.stats[sort_by_interval][sort_by_parameter])
        stats_by_group = defaultdict(lambda: [])
        for person_stats in sorted_stats:
            stats_by_group[group_names[person_stats.login]].append(person_stats)

        def make_cell(text, cell_type="td", colspan=None, rowspan=None, color=None, background_color=None):
            colspan = f"colspan=\"{colspan}\"" if colspan else ""
            rowspan = f"rowspan=\"{rowspan}\"" if rowspan else ""
            if color:
                text = f"<font color=\"{color}\">{text}</font>"
            background_color = f"background-color: {background_color};" if background_color else ""
            return f"<{cell_type} {colspan} {rowspan} style=\"text-align: center; {background_color}\">{text}</{cell_type}>\n"

        def join_cells(cells):
            return "".join(cell for cell in cells)

        def make_cells(texts):
            return join_cells(make_cell(text) for text in texts)

        def make_cells_from_info(info):
            return join_cells([make_cell(info["commits"]),
                                make_cell(f"+{info['linesAdded']}", color="green"),
                                make_cell(f"-{info['linesDeleted']}", color="red")])

        def make_row(cells):
            return f"<tr>{cells}</tr>\n"

        def make_link(link, text):
            return f"<a href=\"{link}\">{text}</a>"

        logger.info("Stats by group: ", stats_by_group)

        def make_table(group, group_stats):
            text = "<table style=\"width:100%\" border=\"1\">\n"
            text += make_row(make_cell(group, cell_type="th", colspan=3 * len(intervals) + 1))
            text += make_row(join_cells([make_cell("name/login", cell_type="th", rowspan=2)] +
                                        [make_cell(f"{days} days", cell_type="th", colspan=3) for days in intervals]))

            text += make_row(join_cells([make_cell("Commits"), make_cell("Added"), make_cell("Deleted")]) * len(intervals))

            for person_stats in group_stats:
                login = person_stats.login
                arcanum_link = f"https://a.yandex-team.ru/users/{login}"
                staff_link = f"https://staff.yandex-team.ru/{login}"

                name_text = ""
                if is_chief[login]:
                    name_text += "👑 "
                name_text += make_link(staff_link, person_names[login])
                name_text += "<br>"
                name_text += make_link(arcanum_link, login)

                name_cell = make_cell(name_text, background_color="#edffd3" if is_newbie[login] else None)
                stats_cells = join_cells([make_cells_from_info(person_stats.stats[days]) for days in intervals])
                text += make_row(name_cell + stats_cells)

            text += "</table>\n"

            return text

        text = ""

        text += f"https://sandbox.yandex-team.ru/task/{self.task_id}/view"

        logger.info(f"Group stats: {stats_by_group}")

        for group in group_order:
            if group in stats_by_group:
                logger.info(f"Group in order: {group}")
                text += make_table(group, stats_by_group[group])

        for group, group_stats in stats_by_group.items():
            if group not in group_order:
                logger.info(f"Group not in order: {group}")
                text += make_table(group, group_stats)

        logger.info(f"Sending html:\n{text}")

        email = to + "@yandex-team.ru"

        logger.info(f"Trying to send stats for user {to}")

        if self.dry_run:
            logger.info(f"Trying to send a message to {email}, with text \"{text}\"")
            task.set_info(f"Commit stats for the past {max(intervals)} days for user {to}\n{text}", do_escape=False)
        else:
            logger.info(f"Sending a message to {email}, with text \"{text}\"")
            self.sendmail(email, f"Commit stats for the past {max(intervals)} days for user {to}", text)
            self.sendmail("okwedook@yandex-team.ru", f"Commit stats for the past {max(intervals)} days for user {to}", text)
