import aiohttp
import asyncio
import requests
import logging

from sandbox.common.errors import SandboxException

STAFF_GROUP_API = 'https://staff-api.yandex-team.ru/v3/groupmembership/'
STAFF_PERSON_API = 'https://staff-api.yandex-team.ru/v3/persons'

logger = logging.getLogger(__name__)


class StaffHelper:
    def __init__(self, token):
        self._token = token

    async def get_all_persons_from_department_async(self, department_url):
        def make_params(min_id, department_url):
            return {
                "_nopage": 1,
                "_sort": "id",
                "_query": f'id > {min_id} and (group.url == "{department_url}" or group.ancestors.url == "{department_url}")',
                "_fields": "id,person.login",
                "group.type": "department",
                "person.official.is_dismissed": "false",
            }
        url = STAFF_GROUP_API
        token = self._token
        headers = {"Authorization": f"OAuth {token}"}
        persons = []
        async with aiohttp.ClientSession(headers=headers) as session:
            last_id = -1
            while True:
                params = make_params(last_id, department_url)
                async with session.get(url=url, params=params) as response:
                    response_data = await response.json()
                    if not response_data["result"]:
                        break
                    last_id = response_data["result"][-1]["id"]
                    persons += response_data["result"]
        return persons

    def get_all_persons_from_department(self, department_url):
        info = asyncio.run(self.get_all_persons_from_department_async(department_url))
        return [x["person"]["login"] for x in info]

    cache = {}

    def make_cache_person_request(self, login):
        if login in self.cache:
            return
        headers = {"Authorization": f"OAuth {self._token}"}
        fields = [
            "chief.login",
            "department_group.department",
            "name",
            "official.join_at"
        ]
        params = {"_fields": ",".join(fields), "login": f"{login}"}
        resp = requests.get(STAFF_PERSON_API, headers=headers, params=params)
        if resp.status_code != 200:
            raise SandboxException("Something wrong with communication with /staff: "
                                   "\n\tstatus_code = {}, reason = {}, request = {}"
                                   .format(resp.status_code, resp.reason, resp.request.url))
        resp = resp.json()
        assert resp["total"] == 1, "One and only one person should be returned by login, but returned {}:\n{}" \
                    .format(resp["total"], resp["result"])
        self.cache[login] = resp["result"][0]

    def make_person_request(self, login):
        self.make_cache_person_request(login)
        return self.cache[login]

    def get_chief_by_login(self, login):
        return self.make_person_request(login)["chief"]["login"]

    def get_group_name_by_login(self, login):
        return self.make_person_request(login)["department_group"]["department"]["name"]["full"]["ru"]

    def get_name_by_login(self, login):
        name = self.make_person_request(login)["name"]
        return name["first"]["ru"] + " " + name["last"]["ru"]

    def is_chief(self, login):
        dep = self.make_person_request(login)
        return login in [chief["person"]["login"] for chief in dep["department_group"]["department"]["heads"]]

    def get_joined_at(self, login):
        join_at = self.make_person_request(login)
        return join_at["official"]["join_at"]
